package org.etsi.osl.metrico.prometheus;


import org.etsi.osl.metrico.model.Job;
import org.etsi.osl.metrico.services.JobService;
import org.etsi.osl.tmf.pm628.model.ExecutionStateType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import java.time.Duration;
import java.time.OffsetDateTime;
import java.util.concurrent.TimeUnit;

/**
 * This class contains the Prometheus Queries for the metrics.
 */
@Service
public class PrometheusQueries {

    private static final Logger logger = LoggerFactory.getLogger(PrometheusQueries.class);
    private final JobService jobService;

    public PrometheusQueries(JobService jobService) {
        this.jobService = jobService;
    }

    public static String sendQueryToPrometheus(String prometheusUrl, String query){
        RestTemplate restTemplate = new RestTemplate();

        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(prometheusUrl)
                .path("/api/v1/query")
                .queryParam("query", query);
        logger.atInfo().log("Prometheus URL: " + prometheusUrl + " Query: " + query);

        ResponseEntity<String> response = restTemplate.getForEntity(builder.toUriString(), String.class);

        return response.getBody();
    }

    public Job startPeriodicQuery(String prometheusUrl, String query, OffsetDateTime startDateTime, OffsetDateTime endDateTime, Integer executionInterval) {

        final Runnable queryHandler = new Runnable() {
            public void run() {
                new Thread(() -> {
                    sendQueryToPrometheus(prometheusUrl, query);
                }).start();
            }
        };
        Job job = jobService.startJob(queryHandler, startDateTime, endDateTime, executionInterval);
        if(job.getState() == ExecutionStateType.FAILED){
            return job;
        }
        // Schedule a task to stop the job after the specified delay
        if (endDateTime != null) {
            long stopAfterSeconds = Duration.between(OffsetDateTime.now(), endDateTime).getSeconds();
            JobService.getScheduler().schedule(() -> {
                JobService.stopJob(job.getUuid());
                job.setState(ExecutionStateType.COMPLETED);
            }, stopAfterSeconds, TimeUnit.SECONDS);
        }
        return job;
    }


}
