package org.etsi.osl.metrico.mapper;

import jakarta.validation.Valid;
import org.etsi.osl.metrico.model.Job;
import org.etsi.osl.metrico.model.SupportedDataAccessEndpoints;
import org.etsi.osl.tmf.pm628.model.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class JobMapper {

    private static final Logger logger = LoggerFactory.getLogger(JobMapper.class);

    public static Job measurementCollectionJobMapToJob(MeasurementCollectionJob measurementCollectionJob) {
        Job job = new Job();

        job.setMeasurementCollectionJobRef(UUID.fromString(measurementCollectionJob.getUuid()));

        if(measurementCollectionJob.getDataAccessEndpoint().size()!=1){    // DataAccessEndpoint will provide the Monitoring API IP
            throw new IllegalArgumentException("DataAccessEndpoint should be exactly one");
        }else {
            DataAccessEndpoint dataAccessEndpoint = measurementCollectionJob.getDataAccessEndpoint().get(0);
            job.setDataAccessEndPointRef(UUID.fromString(dataAccessEndpoint.getUuid()));

            if (SupportedDataAccessEndpoints.contains(dataAccessEndpoint.getApiType())) {   // SupportedDataAccessEndpoints is an ENUM with all the supported API types
                job.setApiType(dataAccessEndpoint.getApiType());
            } else if (dataAccessEndpoint.getApiType() == null) {
                throw new IllegalArgumentException("API type needs to be defined");
            } else {
                throw new IllegalArgumentException("API type " + dataAccessEndpoint.getApiType() + " not supported");
            }

            if (dataAccessEndpoint.getUri() != null) {
                job.setDataAccessEndPointUri(dataAccessEndpoint.getUri());
            } else {
                throw new IllegalArgumentException("DataAccessEndpointUri cannot be null");
            }
        }

        if(measurementCollectionJob.getJobCollectionFilter().getMappings().size() == 1){
            DataFilterMap query = measurementCollectionJob.getJobCollectionFilter();
                // String stringQuery = measurementCollectionJob.getJobCollectionFilter().getMappings().get(0).getFilterTemplate().getDescription();
            job.setQuery(query);
        } else {
            throw new IllegalArgumentException("There should be exactly one query");
        }

        job.setStartDateTime(measurementCollectionJob.getScheduleDefinition().get(0).getScheduleDefinitionStartTime());
        job.setEndDateTime(measurementCollectionJob.getScheduleDefinition().get(0).getScheduleDefinitionEndTime());

        @Valid Granularity granularity = measurementCollectionJob.getGranularity();
        job.setExecutionInterval(convertGranularityToSeconds(measurementCollectionJob.getGranularity().getValue()));
        
        logger.atDebug().setMessage("Received MeasurementCollectionJob:\n" + measurementCollectionJob + "\nConverted it to Job:\n" + job).log();
        return job;
    }

    public static int convertGranularityToSeconds(String value) {
        Pattern PATTERN = Pattern.compile("G_(\\d+)(SEC|MN|H|D|M|Y)");
        if (Granularity.contains(value)) {
            Matcher matcher = PATTERN.matcher(value);
            if (matcher.matches()) {
                int amount = Integer.parseInt(matcher.group(1));
                String unit = matcher.group(2);
                if(value.equalsIgnoreCase(Granularity.NA.getValue())){
                    return Integer.parseInt(null);
                }
                return switch (unit) {
                    case "SEC" -> amount;
                    case "MIN" -> amount * 60;
                    case "H" -> amount * 3600;
                    case "D" -> amount * 86400;
                    case "M" -> amount * 2592000; // Approximate value for a month
                    case "Y" -> amount * 31536000; // Approximate value for a year
                    default -> throw new IllegalArgumentException("Unknown Granularity unit: " + unit);
                };
            }
        }
        throw new IllegalArgumentException("Invalid Granularity format: " + value);
    }
}
