package org.etsi.osl.metrico.services;

import org.etsi.osl.metrico.MetricoCommonMethods;
import org.etsi.osl.metrico.model.Job;
import org.etsi.osl.metrico.prometheus.PrometheusQueries;
import org.etsi.osl.tmf.pm628.model.*;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import java.net.URI;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

class MetricoServiceTest {

    private MetricoCommonMethods metricoCommonMethods;
    private PrometheusQueries prometheusQueries;
    private JobService jobService;
    private MetricoService metricoService;

    private MeasurementCollectionJob validMCJ() {
        MeasurementCollectionJob mcj = new MeasurementCollectionJob();
        mcj.setUuid(UUID.randomUUID().toString());

        DataAccessEndpoint dae = new DataAccessEndpoint();
        dae.setUuid(UUID.randomUUID().toString());
        dae.setApiType("Prometheus"); // must be supported
        dae.setUri(URI.create("http://localhost:9090/api/v1/query?query=up"));
        mcj.setDataAccessEndpoint(List.of(dae));

        ScheduleDefinition schedule = new ScheduleDefinition();
        schedule.setScheduleDefinitionStartTime(OffsetDateTime.now().plusSeconds(1));
        schedule.setScheduleDefinitionEndTime(OffsetDateTime.now().plusHours(1));
        mcj.setScheduleDefinition(List.of(schedule));
        mcj.setGranularity(Granularity.G_30SEC);

        return mcj;
    }

    @BeforeEach
    void setUp() {
        metricoCommonMethods = mock(MetricoCommonMethods.class);
        prometheusQueries = mock(PrometheusQueries.class);
        jobService = mock(JobService.class);
        metricoService = new MetricoService(prometheusQueries, metricoCommonMethods, jobService);
    }

    @Test
    void testOnShutdownStopsAllJobs() {
        metricoService.onShutdown();
        verify(jobService).stopAllJobs();
    }

    @Test
    void testRestartPendingOrInProgressJobs_withJobs() {
        ArrayList<MeasurementCollectionJob> jobs = new ArrayList<>();
        MeasurementCollectionJob mcj1 = validMCJ();
        mcj1.setExecutionState(ExecutionStateType.INPROGRESS);
        jobs.add(mcj1);

        when(metricoCommonMethods.listPendingOrInProgressMeasurementCollectionJobs())
                .thenReturn(jobs);
        when(metricoCommonMethods.retrieveMeasurementCollectionJob(mcj1.getUuid()))
                .thenReturn(mcj1);
        when(prometheusQueries.startPeriodicQuery(anyString(), anyString(), any(Job.class), any(MeasurementCollectionJob.class)))
                .thenAnswer(invocation -> {
                    Job job = invocation.getArgument(2);
                    job.setState(ExecutionStateType.INPROGRESS);
                    return job;
                });

        metricoService.restartPendingOrInProgressJobs();

        verify(jobService).stopAllJobs();
        verify(metricoCommonMethods).retrieveMeasurementCollectionJob(mcj1.getUuid());
    }

    @Test
    void testRestartPendingOrInProgressJobs_withNullList() {
        when(metricoCommonMethods.listPendingOrInProgressMeasurementCollectionJobs())
                .thenReturn(null);

        metricoService.restartPendingOrInProgressJobs();

        verify(jobService).stopAllJobs();
    }

    @Test
    void testStartPeriodicQueryToPrometheus_callsPrometheusQueriesAndUpdatesJob() {
        MeasurementCollectionJob mcj = validMCJ();

        Job job = new Job();
        job.setDataAccessEndPointUri(mcj.getDataAccessEndpoint().get(0).getUri());
        job.setMeasurementCollectionJobRef(UUID.fromString(mcj.getUuid()));
        job.setState(org.etsi.osl.tmf.pm628.model.ExecutionStateType.INPROGRESS);

        when(prometheusQueries.startPeriodicQuery(anyString(), anyString(), any(Job.class), eq(mcj)))
                .thenReturn(job);
        when(metricoCommonMethods.updateMeasurementCollectionJobById(eq(mcj.getUuid()), any()))
                .thenReturn(mcj);

        metricoService.startPeriodicQueryToPrometheus(mcj);

        verify(prometheusQueries).startPeriodicQuery(anyString(), anyString(), any(Job.class), eq(mcj));
        verify(metricoCommonMethods).updateMeasurementCollectionJobById(eq(mcj.getUuid()), any());
        verify(metricoCommonMethods).updateRelatedResource(mcj);
    }

    @Test
    void testStartPeriodicQueryToPrometheusRef_delegatesToStartPeriodicQueryToPrometheus() {
        MeasurementCollectionJobRef mcjRef = new MeasurementCollectionJobRef();
        mcjRef.setId("123e4567-e89b-12d3-a456-426614174002");
        MeasurementCollectionJob mcj = validMCJ();
        mcj.setUuid("123e4567-e89b-12d3-a456-426614174002");

        when(prometheusQueries.startPeriodicQuery(anyString(), anyString(), any(Job.class), any(MeasurementCollectionJob.class)))
                .thenAnswer(invocation -> {
                    Job job = invocation.getArgument(2);
                    job.setState(ExecutionStateType.INPROGRESS);
                    return job;
                });
        when(metricoCommonMethods.retrieveMeasurementCollectionJob(mcjRef)).thenReturn(mcj);

        metricoService.startPeriodicQueryToPrometheusRef(mcjRef);

        verify(metricoCommonMethods).retrieveMeasurementCollectionJob(mcjRef);
    }

    @Test
    void testStartPeriodicQueryToPrometheusEvent_withValidJob() {
        MeasurementCollectionJobCreateEvent event = new MeasurementCollectionJobCreateEvent();
        MeasurementCollectionJobCreateEventPayload payload = new MeasurementCollectionJobCreateEventPayload();
        MeasurementCollectionJobRef mcjRef = new MeasurementCollectionJobRef();
        mcjRef.setId("123e4567-e89b-12d3-a456-426614174002");
        MeasurementCollectionJob mcj = validMCJ();
        mcj.setUuid("123e4567-e89b-12d3-a456-426614174002");
        payload.setMeasurementCollectionJob(mcjRef);
        event.setEvent(payload);

        when(prometheusQueries.startPeriodicQuery(anyString(), anyString(), any(Job.class), any(MeasurementCollectionJob.class)))
                .thenAnswer(invocation -> {
                    Job job = invocation.getArgument(2);
                    job.setState(ExecutionStateType.INPROGRESS);
                    return job;
                });
        when(metricoCommonMethods.retrieveMeasurementCollectionJob("123e4567-e89b-12d3-a456-426614174002")).thenReturn(mcj);

        metricoService.startPeriodicQueryToPrometheusEvent(event);

        verify(metricoCommonMethods).retrieveMeasurementCollectionJob("123e4567-e89b-12d3-a456-426614174002");
    }

    @Test
    void testStartPeriodicQueryToPrometheusEvent_withNullJob() {
        MeasurementCollectionJobCreateEvent event = new MeasurementCollectionJobCreateEvent();
        MeasurementCollectionJobCreateEventPayload payload = new MeasurementCollectionJobCreateEventPayload();
        MeasurementCollectionJobRef mcjRef = new MeasurementCollectionJobRef();
        mcjRef.setId("123e4567-e89b-12d3-a456-426614174002");
        payload.setMeasurementCollectionJob(mcjRef);
        event.setEvent(payload);

        when(metricoCommonMethods.retrieveMeasurementCollectionJob("123e4567-e89b-12d3-a456-426614174002")).thenReturn(null);

        metricoService.startPeriodicQueryToPrometheusEvent(event);

        verify(metricoCommonMethods).retrieveMeasurementCollectionJob("123e4567-e89b-12d3-a456-426614174002");
    }


    @Test
    void testRestartPendingOrInProgressJobs_skipsJobWithNoDataAccessEndpoint() {
        MeasurementCollectionJob mcj = validMCJ();
        mcj.setDataAccessEndpoint(null);

        ArrayList <MeasurementCollectionJob> mcjList = new ArrayList<>();
        mcjList.add(mcj);

        when(metricoCommonMethods.listPendingOrInProgressMeasurementCollectionJobs())
                .thenReturn(mcjList);
        when(metricoCommonMethods.retrieveMeasurementCollectionJob(mcj.getUuid()))
                .thenReturn(mcj);

        metricoService.restartPendingOrInProgressJobs();

        verify(metricoCommonMethods).updateMeasurementCollectionJobById(eq(mcj.getUuid()), any(MeasurementCollectionJobMVO.class));
        verify(jobService).stopAllJobs();
    }

    @Test
    void testRestartPendingOrInProgressJobs_skipsJobWithMultipleDataAccessEndpoints() {
        MeasurementCollectionJob mcj = validMCJ();
        DataAccessEndpoint dae2 = new DataAccessEndpoint();
        dae2.setUuid(UUID.randomUUID().toString());
        dae2.setApiType("Prometheus");
        dae2.setUri(URI.create("http://localhost:9091/api/v1/query?query=down"));
        List<DataAccessEndpoint> endpoints = new ArrayList<>(mcj.getDataAccessEndpoint());
        endpoints.add(dae2);
        mcj.setDataAccessEndpoint(endpoints);

        ArrayList <MeasurementCollectionJob> mcjList = new ArrayList<>();
        mcjList.add(mcj);

        when(metricoCommonMethods.listPendingOrInProgressMeasurementCollectionJobs())
                .thenReturn(mcjList);
        when(metricoCommonMethods.retrieveMeasurementCollectionJob(mcj.getUuid()))
                .thenReturn(mcj);

        metricoService.restartPendingOrInProgressJobs();

        verify(metricoCommonMethods).updateMeasurementCollectionJobById(eq(mcj.getUuid()), any(MeasurementCollectionJobMVO.class));
        verify(jobService).stopAllJobs();
    }

    @Test
    void testRestartPendingOrInProgressJobs_skipsJobWithMultipleScheduleDefinitions() {
        MeasurementCollectionJob mcj = validMCJ();
        ScheduleDefinition schedule2 = new ScheduleDefinition();
        schedule2.setScheduleDefinitionStartTime(OffsetDateTime.now().plusSeconds(2));
        schedule2.setScheduleDefinitionEndTime(OffsetDateTime.now().plusHours(2));
        List<ScheduleDefinition> schedules = new ArrayList<>(mcj.getScheduleDefinition());
        schedules.add(schedule2);
        mcj.setScheduleDefinition(schedules);

        ArrayList <MeasurementCollectionJob> mcjList = new ArrayList<>();
        mcjList.add(mcj);

        when(metricoCommonMethods.listPendingOrInProgressMeasurementCollectionJobs())
                .thenReturn(mcjList);
        when(metricoCommonMethods.retrieveMeasurementCollectionJob(mcj.getUuid()))
                .thenReturn(mcj);

        metricoService.restartPendingOrInProgressJobs();

        verify(metricoCommonMethods).updateMeasurementCollectionJobById(eq(mcj.getUuid()), any(MeasurementCollectionJobMVO.class));
        verify(jobService).stopAllJobs();
    }

    @Test
    void testRestartPendingOrInProgressJobs_skipsJobWithEndedScheduleDefinition() {
        MeasurementCollectionJob mcj = validMCJ();
        OffsetDateTime start = OffsetDateTime.now().minusHours(2);
        OffsetDateTime end = OffsetDateTime.now().minusHours(1);
        mcj.getScheduleDefinition().get(0).setScheduleDefinitionStartTime(start);
        mcj.getScheduleDefinition().get(0).setScheduleDefinitionEndTime(end);

        ArrayList <MeasurementCollectionJob> mcjList = new ArrayList<>();
        mcjList.add(mcj);

        when(metricoCommonMethods.listPendingOrInProgressMeasurementCollectionJobs())
                .thenReturn(mcjList);
        when(metricoCommonMethods.retrieveMeasurementCollectionJob(mcj.getUuid()))
                .thenReturn(mcj);

        metricoService.restartPendingOrInProgressJobs();

        verify(metricoCommonMethods).updateMeasurementCollectionJobById(eq(mcj.getUuid()), any(MeasurementCollectionJobMVO.class));
        verify(jobService).stopAllJobs();
    }

    @Test
    void testRestartPendingOrInProgressJobs_jobWithNoScheduleDefinition_addsAndSkipsIfEnded() {
        MeasurementCollectionJob mcj = validMCJ();
        mcj.setScheduleDefinition(new ArrayList<>()); // No schedule definitions
        mcj.setCreationTime(OffsetDateTime.now().minusHours(2));
        ArrayList <MeasurementCollectionJob> mcjList = new ArrayList<>();
        mcjList.add(mcj);

        when(metricoCommonMethods.listPendingOrInProgressMeasurementCollectionJobs())
                .thenReturn(mcjList);
        when(metricoCommonMethods.retrieveMeasurementCollectionJob(mcj.getUuid()))
                .thenReturn(mcj);

        metricoService.restartPendingOrInProgressJobs();

        verify(metricoCommonMethods).updateMeasurementCollectionJobById(eq(mcj.getUuid()), any(MeasurementCollectionJobMVO.class));
        verify(jobService).stopAllJobs();
    }
}