package org.etsi.osl.metrico.services;

import org.etsi.osl.metrico.MetricoCommonMethods;
import org.etsi.osl.metrico.model.Job;
import org.etsi.osl.tmf.pm628.model.ExecutionStateType;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import java.time.OffsetDateTime;
import java.util.UUID;
import java.util.concurrent.ScheduledFuture;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

class JobServiceTest {

    private MetricoCommonMethods metricoCommonMethods;
    private JobService jobService;

    @BeforeEach
    void setUp() {
        metricoCommonMethods = mock(MetricoCommonMethods.class);
        jobService = new JobService(metricoCommonMethods);
    }

    @Test
    void startJob_withValidDates_shouldStartJob() {
        Job job = new Job();
        job.setStartDateTime(OffsetDateTime.now().plusSeconds(1));
        job.setEndDateTime(OffsetDateTime.now().plusHours(1));
        job.setExecutionInterval(10);
        job.setMeasurementCollectionJobRef(UUID.fromString("123e4567-e89b-12d3-a456-426614174002"));

        Runnable task = mock(Runnable.class);

        Job result = jobService.startJob(task, job);

        assertEquals(ExecutionStateType.INPROGRESS, result.getState());
        verify(metricoCommonMethods).updateMeasurementCollectionJobById(eq("123e4567-e89b-12d3-a456-426614174002"), any());
        assertTrue(jobService.getRunningJobs().containsKey("123e4567-e89b-12d3-a456-426614174002"));
    }

    @Test
    void startJob_withEndDateBeforeNow_shouldFailJob() {
        Job job = new Job();
        job.setStartDateTime(OffsetDateTime.now().minusHours(2));
        job.setEndDateTime(OffsetDateTime.now().minusHours(1));
        job.setExecutionInterval(10);
        job.setMeasurementCollectionJobRef(UUID.fromString("123e4567-e89b-12d3-a456-426614174002"));

        Runnable task = mock(Runnable.class);

        Job result = jobService.startJob(task, job);

        assertEquals(ExecutionStateType.FAILED, result.getState());
        verify(metricoCommonMethods).updateMeasurementCollectionJobById(eq("123e4567-e89b-12d3-a456-426614174002"), any());
        assertFalse(jobService.getRunningJobs().containsKey("123e4567-e89b-12d3-a456-426614174002"));
    }

    @Test
    void stopJob_shouldCancelFutureAndSetCancelled() {
        Job job = new Job();
        job.setMeasurementCollectionJobRef(UUID.fromString("123e4567-e89b-12d3-a456-426614174002"));
        job.setState(ExecutionStateType.INPROGRESS);

        ScheduledFuture<?> future = mock(ScheduledFuture.class);
        when(future.cancel(true)).thenReturn(true);
        job.setFuture(future);

        jobService.getRunningJobs().put("123e4567-e89b-12d3-a456-426614174002", job);

        jobService.stopJob(job);

        assertEquals(ExecutionStateType.CANCELLED, job.getState());
        assertFalse(jobService.getRunningJobs().containsKey("123e4567-e89b-12d3-a456-426614174002"));
        verify(metricoCommonMethods).updateMeasurementCollectionJobById(eq("123e4567-e89b-12d3-a456-426614174002"), any());
    }

    @Test
    void stopJob_alreadyCancelled_shouldJustRemove() {
        Job job = new Job();
        job.setMeasurementCollectionJobRef(UUID.fromString("123e4567-e89b-12d3-a456-426614174002"));
        job.setState(ExecutionStateType.CANCELLED);

        jobService.getRunningJobs().put("123e4567-e89b-12d3-a456-426614174002", job);

        jobService.stopJob(job);

        assertFalse(jobService.getRunningJobs().containsKey("123e4567-e89b-12d3-a456-426614174002"));
    }

    @Test
    void stopJob_byUuid_shouldCallStopJob() {
        Job job = new Job();
        job.setMeasurementCollectionJobRef(UUID.fromString("123e4567-e89b-12d3-a456-426614174002"));
        job.setState(ExecutionStateType.INPROGRESS);

        ScheduledFuture<?> future = mock(ScheduledFuture.class);
        when(future.cancel(true)).thenReturn(true);
        job.setFuture(future);

        jobService.getRunningJobs().put("123e4567-e89b-12d3-a456-426614174002", job);

        jobService.stopJob("123e4567-e89b-12d3-a456-426614174002");

        assertEquals(ExecutionStateType.CANCELLED, job.getState());
        assertFalse(jobService.getRunningJobs().containsKey("123e4567-e89b-12d3-a456-426614174002"));
    }

    @Test
    void stopAllJobs_shouldStopAndClearAll() {
        Job job1 = new Job();
        job1.setMeasurementCollectionJobRef(UUID.fromString("123e4567-e89b-12d3-a456-426614174002"));
        job1.setState(ExecutionStateType.INPROGRESS);
        ScheduledFuture<?> future1 = mock(ScheduledFuture.class);
        when(future1.cancel(true)).thenReturn(true);
        job1.setFuture(future1);

        Job job2 = new Job();
        job2.setMeasurementCollectionJobRef(UUID.fromString("123e4567-e89b-12d3-a456-426614174007"));
        job2.setState(ExecutionStateType.INPROGRESS);
        ScheduledFuture<?> future2 = mock(ScheduledFuture.class);
        when(future2.cancel(true)).thenReturn(true);
        job2.setFuture(future2);

        jobService.getRunningJobs().put("123e4567-e89b-12d3-a456-426614174002", job1);
        jobService.getRunningJobs().put("123e4567-e89b-12d3-a456-426614174007", job2);

        jobService.stopAllJobs();

        assertTrue(jobService.getRunningJobs().isEmpty());
        assertEquals(ExecutionStateType.CANCELLED, job1.getState());
        assertEquals(ExecutionStateType.CANCELLED, job2.getState());
    }
}