package org.etsi.osl.metrico.prometheus;

import org.etsi.osl.metrico.MetricoCommonMethods;
import org.etsi.osl.metrico.model.Job;
import org.etsi.osl.metrico.services.JobService;
import org.etsi.osl.tmf.pm628.model.ExecutionStateType;
import org.etsi.osl.tmf.pm628.model.MeasurementCollectionJob;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.MockedStatic;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.time.OffsetDateTime;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

class PrometheusQueriesTest {

    private JobService jobService;
    private MetricoCommonMethods metricoCommonMethods;
    private PrometheusQueries prometheusQueries;


    @BeforeEach
    void setUp() {
        jobService = mock(JobService.class);
        metricoCommonMethods = mock(MetricoCommonMethods.class);
        prometheusQueries = new PrometheusQueries(jobService, metricoCommonMethods);
    }

    @Test
    void testConstructor_AssignsDependencies() throws Exception {
        PrometheusQueries queries = new PrometheusQueries(jobService, metricoCommonMethods);

        Field jobServiceField = PrometheusQueries.class.getDeclaredField("jobService");
        jobServiceField.setAccessible(true);
        assertSame(jobService, jobServiceField.get(queries));

        Field metricoCommonMethodsField = PrometheusQueries.class.getDeclaredField("metricoCommonMethods");
        metricoCommonMethodsField.setAccessible(true);
        assertSame(metricoCommonMethods, metricoCommonMethodsField.get(queries));
    }
    
    @Test
    void testStartPeriodicQuery_SetsDefaultsAndSchedulesJob() {
        Job job = new Job();
        job.setExecutionInterval(null);

        MeasurementCollectionJob mcj = new MeasurementCollectionJob();
        when(jobService.startJob(any(Runnable.class), any(Job.class))).thenAnswer(invocation -> {
            Job j = invocation.getArgument(1);
            j.setState(ExecutionStateType.INPROGRESS);
            return j;
        });

        Job result = prometheusQueries.startPeriodicQuery("http://localhost:9090", "query=up", job, mcj);

        assertNotNull(result.getExecutionInterval());
        assertEquals(ExecutionStateType.INPROGRESS, result.getState());
    }

    @Test
    void testStartPeriodicQuery_FailedJobIsStopped() {
        Job job = new Job();
        MeasurementCollectionJob mcj = new MeasurementCollectionJob();

        when(jobService.startJob(any(Runnable.class), any(Job.class))).thenAnswer(invocation -> {
            Job j = invocation.getArgument(1);
            j.setState(ExecutionStateType.FAILED);
            return j;
        });

        Job result = prometheusQueries.startPeriodicQuery("http://localhost:9090", "query=up", job, mcj);

        assertEquals(ExecutionStateType.FAILED, result.getState());
        verify(jobService).stopJob(any(Job.class));
    }

    @Test
    void testSendQueryToPrometheus_HandlesWebClientError() {
        // This test checks that even if WebClient fails, a response string is returned and updateService is called.
        MeasurementCollectionJob mcj = new MeasurementCollectionJob();
        mcj.setConsumingApplicationId("service-uuid");
        mcj.setOutputFormat("output");

        Job job = new Job();
        String response = prometheusQueries.sendQueryToPrometheus("http://invalid-url", "query=up", mcj, job);

        assertTrue(response.contains("status"));
        verify(metricoCommonMethods).updateService(eq("service-uuid"), any(), eq(true));
    }

    @Test
    void testSendQueryToPrometheus_WhenWebClientThrowsException_LogsErrorAndReturnsStatus() {
        PrometheusQueries pq = new PrometheusQueries(jobService, metricoCommonMethods) {
            @Override
            public String sendQueryToPrometheus(String prometheusUrl, String query, MeasurementCollectionJob mcj, Job job) {
                // Simulate exception in WebClient block
                try {
                    throw new RuntimeException("Simulated error");
                } catch (Exception e) {
                    Logger logger = LoggerFactory.getLogger(PrometheusQueriesTest.class);;
                    logger.error(" error on web client request");
                    String response = "{\"status\":\"" + e.getLocalizedMessage() + "\"";
                    // Simulate updateService call as in original method
                    metricoCommonMethods.updateService(mcj.getConsumingApplicationId(), null, true);
                    return response;
                }
            }
        };

        MeasurementCollectionJob mcj = new MeasurementCollectionJob();
        mcj.setConsumingApplicationId("service-uuid");
        mcj.setOutputFormat("output");
        Job job = new Job();

        String response = pq.sendQueryToPrometheus("http://localhost:9090", "query=up", mcj, job);

        assertTrue(response.contains("Simulated error"));
        verify(metricoCommonMethods).updateService(eq("service-uuid"), any(), eq(true));
    }

    @Test
    void testStartPeriodicQuery_SchedulesStopJobWhenEndDateTimePresent() {
        Job job = new Job();
        job.setExecutionInterval(100);
        job.setEndDateTime(OffsetDateTime.now().plusSeconds(61));
        MeasurementCollectionJob mcj = new MeasurementCollectionJob();

        ScheduledExecutorService scheduler = mock(ScheduledExecutorService.class);

        try (MockedStatic<JobService> jobServiceStatic = mockStatic(JobService.class)) {
            jobServiceStatic.when(JobService::getScheduler).thenReturn(scheduler);

            when(jobService.startJob(any(Runnable.class), any(Job.class))).thenAnswer(invocation -> {
                Job j = invocation.getArgument(1);
                j.setState(ExecutionStateType.INPROGRESS);
                return j;
            });

            prometheusQueries.startPeriodicQuery("http://localhost:9090", "query=up", job, mcj);

            verify(scheduler).schedule(any(Runnable.class), eq(60L), eq(TimeUnit.SECONDS));
        }
    }

}