package org.etsi.osl.metrico.mapper;

import org.etsi.osl.metrico.model.Job;
import org.etsi.osl.tmf.pm628.model.DataAccessEndpoint;
import org.etsi.osl.tmf.pm628.model.Granularity;
import org.etsi.osl.tmf.pm628.model.MeasurementCollectionJob;
import org.etsi.osl.tmf.pm628.model.ScheduleDefinition;
import org.junit.jupiter.api.Test;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import static org.junit.jupiter.api.Assertions.*;

public class JobMapperTest {

    @Test
    public void testMeasurementCollectionJobMapToJob() {
        MeasurementCollectionJob measurementCollectionJob = new MeasurementCollectionJob();
        measurementCollectionJob.setUuid("9f22dc98-f439-4fdd-98e3-f6471cf8ca67");

        DataAccessEndpoint dataAccessEndpoint = new DataAccessEndpoint();
        dataAccessEndpoint.setUuid("123e4567-e89b-12d3-a456-426614174000");
        dataAccessEndpoint.setUri(URI.create("http://example.com"));
        dataAccessEndpoint.setApiType("Prometheus");

        List<DataAccessEndpoint> dataAccessEndpointList = new ArrayList<>();
        dataAccessEndpointList.add(dataAccessEndpoint);
        measurementCollectionJob.setDataAccessEndpoint(dataAccessEndpointList);

        ScheduleDefinition scheduleDefinition = new ScheduleDefinition();
        scheduleDefinition.setScheduleDefinitionStartTime("2023-01-01T00:00:00Z");
        scheduleDefinition.setScheduleDefinitionEndTime("2023-01-02T00:00:00Z");

        List<ScheduleDefinition> scheduleDefinitionList = new ArrayList<>();
        scheduleDefinitionList.add(scheduleDefinition);
        measurementCollectionJob.setScheduleDefinition(scheduleDefinitionList);

        Granularity granularity = Granularity.G_1H;
        measurementCollectionJob.setGranularity(granularity);

        Job job = JobMapper.measurementCollectionJobMapToJob(measurementCollectionJob);

        assertNotNull(job);
        assertEquals(UUID.fromString("9f22dc98-f439-4fdd-98e3-f6471cf8ca67"), job.getMeasurementCollectionJobRef());
        assertEquals(UUID.fromString("123e4567-e89b-12d3-a456-426614174000"), job.getDataAccessEndPointRef());
        assertEquals("Prometheus", job.getApiType());
        assertEquals(URI.create("http://example.com"), job.getDataAccessEndPointUri());
        assertEquals("2023-01-01T00:00Z", job.getStartDateTime().toString());
        assertEquals("2023-01-02T00:00Z", job.getEndDateTime().toString());
        assertEquals(3600, job.getExecutionInterval());
    }

    @Test
    public void testConvertGranularityToSeconds() {
        assertEquals(60, JobMapper.convertGranularityToSeconds("G_1MN"));
        assertEquals(3600, JobMapper.convertGranularityToSeconds("G_1H"));
        assertEquals(2592000, JobMapper.convertGranularityToSeconds("G_1M"));
        assertEquals(31536000, JobMapper.convertGranularityToSeconds("G_1Y"));
    }

    @Test
    public void testConvertGranularityToSecondsInvalidFormat() {
        assertThrows(IllegalArgumentException.class, () -> JobMapper.convertGranularityToSeconds("G_1W"));
    }
}