package org.etsi.osl.metrico.services;

import jakarta.annotation.PreDestroy;
import jakarta.validation.constraints.NotNull;
import org.apache.camel.builder.RouteBuilder;
import org.etsi.osl.metrico.MetricoCommonMethods;
import org.etsi.osl.metrico.mapper.JobMapper;
import org.etsi.osl.metrico.model.Job;
import org.etsi.osl.metrico.prometheus.PrometheusQueries;
import org.etsi.osl.tmf.common.model.ELifecycle;
import org.etsi.osl.tmf.common.model.EValueType;
import org.etsi.osl.tmf.common.model.Notification;
import org.etsi.osl.tmf.common.model.service.Characteristic;
import org.etsi.osl.tmf.common.model.service.ServiceStateType;
import org.etsi.osl.tmf.pm628.model.*;
import org.etsi.osl.tmf.rcm634.model.LogicalResourceSpecification;
import org.etsi.osl.tmf.rcm634.model.ResourceSpecificationCreate;
import org.etsi.osl.tmf.rcm634.model.ResourceSpecificationRef;
import org.etsi.osl.tmf.scm633.model.ServiceSpecification;
import org.etsi.osl.tmf.sim638.model.ServiceAttributeValueChangeNotification;
import org.etsi.osl.tmf.sim638.model.ServiceDeleteNotification;
import org.etsi.osl.tmf.sim638.model.ServiceStateChangeNotification;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.context.event.ApplicationStartedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Service;

import java.net.URI;
import java.net.URISyntaxException;
import java.time.OffsetDateTime;
import java.util.List;

@Service
public class MetricoService extends RouteBuilder {

    public static final String OSL_METRICO_RSPEC_NAME = "METRICO_Resource_Specification";
    public static final String OSL_METRICO_RSPEC_VERSION = "1.0.0";
    public static final String OSL_METRICO_RSPEC_CATEGORY = "metrico.osl.etsi.org/v1";
    public static final String OSL_METRICO_RSPEC_TYPE = "LogicalResourceSpecification";
    public static final String OSL_METRICO_RSPEC_DESCRIPTION = "This Specification is used to describe a generic METRICO job resource";
    private static final Logger logger = LoggerFactory.getLogger(MetricoService.class);
    private static String OSL_METRICO_RSPEC_ID = null;

    private final MetricoCommonMethods metricoCommonMethods;
    private final PrometheusQueries prometheusQueries;
    private final JobService jobService;

    public MetricoService(PrometheusQueries prometheusQueries, MetricoCommonMethods metricoCommonMethods, JobService jobService) {
        this.prometheusQueries = prometheusQueries;
        this.metricoCommonMethods = metricoCommonMethods;
        this.jobService = jobService;
    }

    @Override
    public void configure() throws Exception {
        // TODO Auto-generated method stub
    }

    @EventListener(ApplicationStartedEvent.class)
    public void onApplicationEvent() {
        // When METRICO starts, it checks if a related resource spec exists, and if it not, it creates it.
        registerMetricoResourceSpec();
        // When METRICO starts, it checks if there are any pending or unfinished jobs from previous sessions
        // and tries to resume them.
        restartPendingOrInProgressJobs();

    }

    @PreDestroy
    public void onShutdown() {
        jobService.stopAllJobs();
        logger.info("All running jobs have been stopped.");
    }


    private void registerMetricoResourceSpec() {
        ResourceSpecificationCreate rsc = new ResourceSpecificationCreate();
        rsc.setName(OSL_METRICO_RSPEC_NAME);
        rsc.setCategory(OSL_METRICO_RSPEC_CATEGORY);
        rsc.setVersion(OSL_METRICO_RSPEC_VERSION);
        rsc.setDescription(OSL_METRICO_RSPEC_DESCRIPTION);
        rsc.setType(OSL_METRICO_RSPEC_TYPE);

        rsc.setLifecycleStatus(ELifecycle.ACTIVE.getValue());
        rsc.addResourceSpecificationCharacteristicItemShort("_MT_CHARACTERISTIC_NAME", "", EValueType.TEXT.getValue(), "The characteristic of the service with id _MT_SERVICEUUID that will be updated with monitoring metrics", false);
        rsc.addResourceSpecificationCharacteristicItemShort("_MT_SERVICEUUID", "", EValueType.TEXT.getValue(), "The id of the service to update", false);
        rsc.addResourceSpecificationCharacteristicItemShort("_MT_RECURRING_INTERVAL", "G_1MN", EValueType.TEXT.getValue(), "The polling interval of the monitoring source", false);
        rsc.addResourceSpecificationCharacteristicItemShort("_MT_TYPE", "PROMETHEUS", EValueType.TEXT.getValue(), "The monitoring source type", false);
        rsc.addResourceSpecificationCharacteristicItemShort("_MT_QUERY", "", EValueType.TEXT.getValue(), "The query towards the monitoring source (e.g. query=gnb_service_state)", false);
        rsc.addResourceSpecificationCharacteristicItemShort("_MT_URL", "", EValueType.TEXT.getValue(), "The monitoring source URL (e.g. https://prom.osl.etsi.org:9090)", false);

        LogicalResourceSpecification result = metricoCommonMethods.createOrUpdateResourceSpecByNameCategoryVersion(rsc);

        while (result == null) {
            try {
                logger.info("Cannot get resource for registerMetricoResourceSpec. Retrying in 10 seconds");
                Thread.sleep(10000);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
            result = metricoCommonMethods.createOrUpdateResourceSpecByNameCategoryVersion(rsc);
        }
        OSL_METRICO_RSPEC_ID = result.getId();
    }

    public void restartPendingOrInProgressJobs() {
        logger.info("===== Looking for PENDING or IN_PROGRESS Measurement Collection Jobs from previous sessions =====");
        List<MeasurementCollectionJob> jobsPending = metricoCommonMethods.listPendingOrInProgressMeasurementCollectionJobs();
        jobService.stopAllJobs();
        if (jobsPending != null) {
            logger.info("===== Started resuming PENDING or IN_PROGRESS Measurement Collection Jobs from previous sessions =====");
            for (MeasurementCollectionJob measurementCollectionJob : jobsPending) {
                measurementCollectionJob = metricoCommonMethods.retrieveMeasurementCollectionJob(measurementCollectionJob.getUuid());
                if (measurementCollectionJob.getDataAccessEndpoint() == null || measurementCollectionJob.getDataAccessEndpoint().size() != 1) {
                    logger.warn("MeasurementCollectionJob with uuid: {} has {} dataAccessEndpoint(s), skipping.",
                            measurementCollectionJob.getUuid(),
                            measurementCollectionJob.getDataAccessEndpoint() == null ? 0 : measurementCollectionJob.getDataAccessEndpoint().size());
                    MeasurementCollectionJobMVO mcjMVO = new MeasurementCollectionJobMVO();
                    mcjMVO.setExecutionState(ExecutionStateType.FAILED);
                    metricoCommonMethods.updateMeasurementCollectionJobById(measurementCollectionJob.getUuid(), mcjMVO);
                    continue;
                }
                if (measurementCollectionJob.getScheduleDefinition().size() > 1) {
                    logger.warn("MeasurementCollectionJob with uuid: {} has more than 1 ScheduleDefinitions, skipping.",
                            measurementCollectionJob.getUuid());
                    MeasurementCollectionJobMVO mcjMVO = new MeasurementCollectionJobMVO();
                    mcjMVO.setExecutionState(ExecutionStateType.FAILED);
                    metricoCommonMethods.updateMeasurementCollectionJobById(measurementCollectionJob.getUuid(), mcjMVO);
                    continue;
                } else if (measurementCollectionJob.getScheduleDefinition().size() == 1) {
                    // If the ScheduleDefinitionEndTime is null, set it to 1 hour after the ScheduleDefinitionStartTime to check if it is past due
                    if (measurementCollectionJob.getScheduleDefinition().get(0).getScheduleDefinitionEndTime() == null && measurementCollectionJob.getScheduleDefinition().get(0).getScheduleDefinitionStartTime() != null) {
                        measurementCollectionJob.getScheduleDefinition().get(0).setScheduleDefinitionEndTime(measurementCollectionJob.getScheduleDefinition().get(0).getScheduleDefinitionStartTime().plusHours(1));
                    } else if (measurementCollectionJob.getScheduleDefinition().get(0).getScheduleDefinitionEndTime() == null && measurementCollectionJob.getScheduleDefinition().get(0).getScheduleDefinitionStartTime() == null) {
                        measurementCollectionJob.getScheduleDefinition().get(0).setScheduleDefinitionEndTime(measurementCollectionJob.getCreationTime().plusHours(1));
                    }
                    // If the ScheduleDefinitionEndTime is before the current time, skip the job
                    if (measurementCollectionJob.getScheduleDefinition().get(0).getScheduleDefinitionEndTime().isBefore(OffsetDateTime.now())) {
                        logger.warn("MeasurementCollectionJob with uuid: {} has a ScheduleDefinition that has already ended, skipping.",
                                measurementCollectionJob.getUuid());
                        MeasurementCollectionJobMVO mcjMVO = new MeasurementCollectionJobMVO();
                        mcjMVO.setExecutionState(ExecutionStateType.FAILED);
                        metricoCommonMethods.updateMeasurementCollectionJobById(measurementCollectionJob.getUuid(), mcjMVO);
                        continue;
                    }
                } else {
                    // If there is no ScheduleDefinition, set the end time to 1 hour after the creation time
                    measurementCollectionJob.addScheduleDefinitionItem(new ScheduleDefinition());
                    measurementCollectionJob.getScheduleDefinition().get(0).setScheduleDefinitionStartTime(measurementCollectionJob.getCreationTime());
                    measurementCollectionJob.getScheduleDefinition().get(0).setScheduleDefinitionEndTime(measurementCollectionJob.getCreationTime().plusHours(1));

                    if (measurementCollectionJob.getScheduleDefinition().get(0).getScheduleDefinitionEndTime().isBefore(OffsetDateTime.now())) {
                        logger.warn("MeasurementCollectionJob with uuid: {} has a ScheduleDefinition that has already ended, skipping.",
                                measurementCollectionJob.getUuid());
                        MeasurementCollectionJobMVO mcjMVO = new MeasurementCollectionJobMVO();
                        mcjMVO.setExecutionState(ExecutionStateType.FAILED);
                        metricoCommonMethods.updateMeasurementCollectionJobById(measurementCollectionJob.getUuid(), mcjMVO);
                        continue;
                    }
                }
                logger.info("===== Resuming measurementCollectionJob with uuid: {} =====", measurementCollectionJob.getUuid());
                this.startPeriodicQueryToPrometheus(measurementCollectionJob);
            }
            logger.info("===== Finished resuming PENDING or IN_PROGRESS Measurement Collection Jobs from previous sessions =====");
        } else {
            logger.info("===== No PENDING OF IN_PROGRESS Measurement Collection Jobs from previous sessions to restart =====");
        }
    }


    public void startPeriodicQueryToPrometheus(@NotNull MeasurementCollectionJob givenMCJ) {
        Job job = JobMapper.measurementCollectionJobMapToJob(givenMCJ);
        String promURL = job.getDataAccessEndPointUri().getScheme() + "://" + job.getDataAccessEndPointUri().getAuthority();
        String promQuery = job.getDataAccessEndPointUri().getQuery();
        logger.atInfo().setMessage("Periodic query started, with ID: " + job.getMeasurementCollectionJobRef()).log();
        MeasurementCollectionJobMVO measurementCollectionJobMVO = new MeasurementCollectionJobMVO();
        job = prometheusQueries.startPeriodicQuery(promURL, promQuery, job, givenMCJ);

        if (job.getState() == ExecutionStateType.FAILED) {
            logger.atError().setMessage("Periodic query failed to start due to internal error.").log();
        } else {
            logger.atInfo().setMessage("Periodic query started, with ID: " + job.getMeasurementCollectionJobRef()).log();
        }
        measurementCollectionJobMVO.setExecutionState(job.getState());

        givenMCJ = metricoCommonMethods.updateMeasurementCollectionJobById(givenMCJ.getUuid(), measurementCollectionJobMVO);
        if (givenMCJ != null) {
            metricoCommonMethods.updateRelatedResource(givenMCJ);
        }
    }


    public void startPeriodicQueryToPrometheusRef(@NotNull MeasurementCollectionJobRef mcjRef) {
        MeasurementCollectionJob givenMCJ = metricoCommonMethods.retrieveMeasurementCollectionJob(mcjRef);

        startPeriodicQueryToPrometheus(givenMCJ);
    }


    public void startPeriodicQueryToPrometheusEvent(@NotNull MeasurementCollectionJobCreateEvent mcjevent) {

        MeasurementCollectionJob givenMCJ = metricoCommonMethods.retrieveMeasurementCollectionJob(mcjevent.getEvent().getMeasurementCollectionJob().getId());

        if (givenMCJ != null) {
            startPeriodicQueryToPrometheus(givenMCJ);
        } else {
            logger.error("=======> CANNOT retrieve Measurement Collection Job with mcjId = " + mcjevent.getEvent().getMeasurementCollectionJob().getId() + " from activeMQ");
        }
    }

    public void handleServiceEvent(final Notification n) {

        org.etsi.osl.tmf.sim638.model.Service service = null;
        ServiceSpecification serviceSpec = null;
        MeasurementCollectionJob mcj = null;
        ResourceSpecificationRef rSpec = null;
        boolean isMetricoRFSService = false;
        boolean isRunningJob = false;

        if (n instanceof ServiceStateChangeNotification) {
            service = ((ServiceStateChangeNotification) n).getEvent().getService();
        } else if (n instanceof ServiceDeleteNotification) {
            service = ((ServiceDeleteNotification) n).getEvent().getService();
        } else if (n instanceof ServiceAttributeValueChangeNotification) {
            service = ((ServiceAttributeValueChangeNotification) n).getEvent().getService();
        }

        // Check if it is a METRICO RFS service
        if (service != null && service.getServiceSpecificationRef() != null) {
            serviceSpec = metricoCommonMethods.retrieveServiceSpecificationById(service.getServiceSpecificationRef().getId());
            if (serviceSpec != null && serviceSpec.getResourceSpecification().size() == 1) {
                rSpec = serviceSpec.getResourceSpecification().iterator().next();
                if (rSpec.getId().equalsIgnoreCase(OSL_METRICO_RSPEC_ID)) {
                    isMetricoRFSService = true;
                }
            }
        } else {
            logger.debug("Service {} is not a METRICO service, skipping.", service.getId());
            return;
        }

        if (isMetricoRFSService) {
            String mcjRefId = service.getServiceCharacteristicByName("_MT_MCJ_REFID").getValue().getValue();
            mcj = metricoCommonMethods.retrieveMeasurementCollectionJob(mcjRefId);

            if(mcj.getExecutionState() !=null) {
                if (mcj.getExecutionState().equals(ExecutionStateType.INPROGRESS) || mcj.getExecutionState().equals(ExecutionStateType.PENDING)) {
                    isRunningJob = true;
                }
            }

            if (n instanceof ServiceStateChangeNotification && service.getState() == ServiceStateType.TERMINATED) {
                logger.debug("Service {} is a METRICO service with state TERMINATED. Terminating related job.", service.getId());
                jobService.stopJob(String.valueOf(service.getServiceCharacteristicByName("_MT_MCJ_REFID").getValue().getValue()));
            } else if (n instanceof ServiceDeleteNotification) {
                logger.debug("Service {} is a METRICO service that was deleted. Terminating related job.", service.getId());
                jobService.stopJob(String.valueOf(service.getServiceCharacteristicByName("_MT_MCJ_REFID").getValue().getValue()));
            } else if (n instanceof ServiceAttributeValueChangeNotification && isRunningJob) {
                logger.debug("Service {} is a METRICO service that was updated. Updating related job.", service.getId());

                MeasurementCollectionJobMVO mcjMVO = new MeasurementCollectionJobMVO();

                mcjMVO.setExecutionState(ExecutionStateType.PENDING);

                MeasurementCollectionJob mcjOld = metricoCommonMethods.retrieveMeasurementCollectionJob(mcjRefId);
                Characteristic serviceCharacteristic;

                if (!mcjOld.getProducingApplicationId().equals(service.getId())) {
                    mcjMVO.setProducingApplicationId(service.getServiceSpecificationRef().getId());
                }
                serviceCharacteristic = service.getServiceCharacteristicByName("_MT_CHARACTERISTIC_NAME");
                String characteristicName = String.valueOf(serviceCharacteristic.getValue().getValue());
                if (metricoCommonMethods.notNullOrEmpty(characteristicName) && !mcjOld.getOutputFormat().equals(characteristicName)) {
                    mcjMVO.setOutputFormat(characteristicName);
                }
                serviceCharacteristic = service.getServiceCharacteristicByName("_MT_SERVICEUUID");
                String cfs_id = String.valueOf(serviceCharacteristic.getValue().getValue());
                if (metricoCommonMethods.notNullOrEmpty(cfs_id) && !mcjOld.getConsumingApplicationId().equals(cfs_id)) {
                    mcjMVO.setConsumingApplicationId(cfs_id);
                }
                ScheduleDefinitionMVO scheduleDefinitionNew = new ScheduleDefinitionMVO();
                ScheduleDefinition scheduleDefinitionOld = mcjOld.getScheduleDefinition().get(0);
                boolean scheduleDefinitionChanged = false;
                if (!scheduleDefinitionOld.getScheduleDefinitionStartTime().equals(service.getStartDate())) {
                    scheduleDefinitionNew.setScheduleDefinitionStartTime(service.getStartDate());
                    scheduleDefinitionChanged = true;
                }
                if (!scheduleDefinitionOld.getScheduleDefinitionEndTime().equals(service.getEndDate())) {
                    scheduleDefinitionNew.setScheduleDefinitionEndTime(service.getEndDate());
                    scheduleDefinitionChanged = true;
                }
                if (scheduleDefinitionChanged) {
                    mcjMVO.addScheduleDefinitionItem(scheduleDefinitionNew);
                }

                serviceCharacteristic = service.getServiceCharacteristicByName("_MT_RECURRING_INTERVAL");
                String recurringIntervalString = String.valueOf(serviceCharacteristic.getValue().getValue());
                if (metricoCommonMethods.notNullOrEmpty(recurringIntervalString) && !mcjOld.getGranularity().toString().equalsIgnoreCase(recurringIntervalString)) {
                    if (Granularity.contains(recurringIntervalString)) {
                        mcjMVO.setGranularity(Granularity.valueOf(recurringIntervalString));
                    }
                }

                boolean monitoringTypeChanged = false;
                DataAccessEndpointMVO daeMVO = new DataAccessEndpointMVO();
                DataAccessEndpoint daeOld = mcjOld.getDataAccessEndpoint().get(0);
                serviceCharacteristic = service.getServiceCharacteristicByName("_MT_TYPE");
                String monitoringType = String.valueOf(serviceCharacteristic.getValue().getValue());
                if (metricoCommonMethods.notNullOrEmpty(monitoringType) && !daeOld.getApiType().equalsIgnoreCase(monitoringType)) {
                    daeMVO.setApiType(monitoringType);
                    monitoringTypeChanged = true;
                }

                boolean uriChanged = false;
                serviceCharacteristic = service.getServiceCharacteristicByName("_MT_QUERY");
                String monitoringQuery = String.valueOf(serviceCharacteristic.getValue().getValue());
                serviceCharacteristic = service.getServiceCharacteristicByName("_MT_URL");
                String monitoringURL = String.valueOf(serviceCharacteristic.getValue().getValue());
                URI newUri = null;
                try {
                    newUri = new URI(monitoringURL + "?" + monitoringQuery);
                } catch (URISyntaxException e) {
                    throw new RuntimeException(e);
                }
                if (!daeOld.getUri().equals(newUri)) {
                    daeMVO.setUri(newUri);
                    uriChanged = true;
                }
                // This is a workaround because the APIType and URI are not mutually exclusive in the MVO.
                // If the monitoring type has changed, we keep the old URI, and if the URI has changed, we keep the old APIType.
                if(monitoringTypeChanged && !uriChanged) {
                    daeMVO.setUri(daeOld.getUri());
                } else if ((!monitoringTypeChanged) && uriChanged) {
                    daeMVO.setApiType(daeOld.getApiType());
                }

                if (uriChanged || monitoringTypeChanged) {
                    mcjMVO.addDataAccessEndpointItem(daeMVO);
                }

                jobService.stopJob(String.valueOf(service.getServiceCharacteristicByName("_MT_MCJ_REFID").getValue().getValue()));

                MeasurementCollectionJob updatedMCJ = metricoCommonMethods.updateMeasurementCollectionJobById(
                        mcjOld.getUuid(), mcjMVO);

                startPeriodicQueryToPrometheus(updatedMCJ);
            } else {
                logger.debug("Service {} is a METRICO service with state {}. No action taken.", service.getId(), service.getState());
            }
        } else {
            logger.debug("Service {} is not a METRICO RFS service, skipping.", service.getId());
        }
    }
}


