package org.etsi.osl.metrico.services;

import lombok.Getter;
import org.etsi.osl.metrico.MetricoCommonMethods;
import org.etsi.osl.metrico.model.Job;
import org.etsi.osl.tmf.pm628.model.ExecutionStateType;
import org.etsi.osl.tmf.pm628.model.MeasurementCollectionJobMVO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.time.OffsetDateTime;
import java.util.concurrent.*;

@Service
public class JobService {

    private static final Logger logger = LoggerFactory.getLogger(JobService.class);

    @Value("${METRICO_THREAD_POOL_SIZE}")
    private static int threadPoolSize;
    @Getter
    private static final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(threadPoolSize);
    @Getter
    private final ConcurrentMap<String, Job> runningJobs = new ConcurrentHashMap<>();


    private final MetricoCommonMethods metricoCommonMethods;
    public JobService(MetricoCommonMethods metricoCommonMethods) {
        this.metricoCommonMethods = metricoCommonMethods;
    }

    /**
     * Starts a job for the given MeasurementCollectionJob.
     *
     * @param task the task to be executed
     * @param job  the job to be started
     * @return the updated job
     * @throws NullPointerException       if the job or task is null
     * @throws IllegalArgumentException   if the job's start or end date is invalid
     * @throws RejectedExecutionException if the job could not be scheduled
     */
    public synchronized Job startJob(Runnable task, Job job) {
        logger.debug("Starting Job for MeasurementCollectionJob with ID {}.", job.getMeasurementCollectionJobRef());
        MeasurementCollectionJobMVO mcjMVO = new MeasurementCollectionJobMVO();

        if (job.getEndDateTime().isBefore(OffsetDateTime.now())) {
            job.setState(ExecutionStateType.FAILED);
            mcjMVO.setExecutionState(ExecutionStateType.FAILED);
            logger.error("Job for MeasurementCollectionJob with ID {} End Date is before now, the MeasurementCollectionJob will not be scheduled.", job.getMeasurementCollectionJobRef().toString());
        } else {
            long initialDelay = Duration.between(OffsetDateTime.now(), job.getStartDateTime()).getSeconds();
            if (initialDelay < 0) {
                initialDelay = 0;
            }
            try {
                ScheduledFuture<?> future = scheduler.scheduleAtFixedRate(
                        task,
                        initialDelay,
                        job.getExecutionInterval(),
                        TimeUnit.SECONDS
                );

                job.setFuture(future);
                job.setState(ExecutionStateType.INPROGRESS);
                mcjMVO.setExecutionState(ExecutionStateType.INPROGRESS);
                runningJobs.put(job.getMeasurementCollectionJobRef().toString(), job);
                logger.info("Job for MeasurementCollectionJob with ID {} started successfully.", job.getMeasurementCollectionJobRef().toString());
            } catch (RejectedExecutionException e) {
                job.setState(ExecutionStateType.FAILED);
                mcjMVO.setExecutionState(ExecutionStateType.FAILED);
                logger.error("Job for MeasurementCollectionJob with ID {} could not be scheduled.", job.getMeasurementCollectionJobRef().toString(), e);
            }
        }
        metricoCommonMethods.updateMeasurementCollectionJobById(job.getMeasurementCollectionJobRef().toString(), mcjMVO);
        return job;
    }

    /**
     * Stops the given job.
     *
     * <p>This method retrieves the associated MeasurementCollectionJob from the database and updates its state.
     * If the job is already cancelled or completed, it removes the job from the runningJobs map and logs the status.
     * If the job is still running, it attempts to cancel the job's future task and updates the job's state accordingly.
     * The job is then removed from the runningJobs map and the MeasurementCollectionJob is updated in the database.</p>
     *
     * @param job the job to be stopped
     */
    public synchronized void stopJob(Job job) {

        MeasurementCollectionJobMVO mcjMVO = new MeasurementCollectionJobMVO();

        if (job.getState() == ExecutionStateType.CANCELLED) {
            logger.debug("Job for MeasurementCollectionJob with ID {} is already CANCELED.", job.getMeasurementCollectionJobRef().toString());
            runningJobs.remove(job.getMeasurementCollectionJobRef().toString());
            return;
        } else if (job.getState() == ExecutionStateType.COMPLETED) {
            logger.debug("Job for MeasurementCollectionJob with ID {} is already COMPLETED.", job.getMeasurementCollectionJobRef().toString());
            runningJobs.remove(job.getMeasurementCollectionJobRef().toString());
            return;
        }
        if (job.getFuture() != null) {
            boolean wasCancelled = job.getFuture().cancel(true);
            if (wasCancelled) {
                job.setState(ExecutionStateType.CANCELLED);
                mcjMVO.setExecutionState(ExecutionStateType.CANCELLED);
                runningJobs.remove(job.getMeasurementCollectionJobRef().toString());
                logger.debug("Job for MeasurementCollectionJob with ID {} stopped successfully.", job.getMeasurementCollectionJobRef().toString());
            } else {
                job.setState(ExecutionStateType.PENDING);
                mcjMVO.setExecutionState(ExecutionStateType.PENDING);
                logger.debug("Job for MeasurementCollectionJob with ID {} could not be stopped because it has already completed, has been cancelled, or could not be cancelled for some other reason.",
                        job.getMeasurementCollectionJobRef().toString());
            }
            metricoCommonMethods.updateMeasurementCollectionJobById(job.getMeasurementCollectionJobRef().toString(), mcjMVO);
        }
    }

    /**
     * Stops the job associated with the given Measurement Collection Job UUID.
     *
     * <p>This method retrieves the job from the runningJobs map using the provided UUID.
     * If the job is found, it calls the existing stopJob method to stop the job.
     * If the job is not found, it logs a warning message.</p>
     *
     * @param measurementCollectionJobUuid the UUID of the Measurement Collection Job
     */
    public synchronized void stopJob(String measurementCollectionJobUuid) {
        Job job = runningJobs.get(measurementCollectionJobUuid);
        if (job != null) {
            stopJob(job);
        } else {
            logger.warn("No job found with associated with MeasurementCollectionJob with UUID: {}", measurementCollectionJobUuid);
        }
    }

    /**
     * Stops all running jobs.
     *
     * <p>This method iterates over all jobs in the runningJobs map, stops each job, and then clears the map.</p>
     */
    public synchronized void stopAllJobs() {
        for (Job job : runningJobs.values()) {
            stopJob(job);
        }
        runningJobs.clear();
    }

}