package org.etsi.osl.metrico;

import com.fasterxml.jackson.databind.ObjectMapper;
import jakarta.validation.constraints.NotNull;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.etsi.osl.tmf.common.model.Any;
import org.etsi.osl.tmf.common.model.service.ResourceRef;
import org.etsi.osl.tmf.pm628.model.ExecutionStateType;
import org.etsi.osl.tmf.pm628.model.MeasurementCollectionJob;
import org.etsi.osl.tmf.pm628.model.MeasurementCollectionJobMVO;
import org.etsi.osl.tmf.pm628.model.MeasurementCollectionJobRef;
import org.etsi.osl.tmf.rcm634.model.LogicalResourceSpecification;
import org.etsi.osl.tmf.rcm634.model.ResourceSpecificationCreate;
import org.etsi.osl.tmf.ri639.model.*;
import org.etsi.osl.tmf.scm633.model.ServiceSpecification;
import org.etsi.osl.tmf.sim638.model.Service;
import org.etsi.osl.tmf.sim638.model.ServiceUpdate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Component
public class MetricoCommonMethods extends RouteBuilder {
    private static final Logger logger = LoggerFactory.getLogger(MetricoCommonMethods.class);

    @Value("${PM_MEASUREMENT_COLLECTION_JOB_UPDATE}")
    private String PM_MEASUREMENT_COLLECTION_JOB_UPDATE = "";
    @Value("${PM_MEASUREMENT_COLLECTION_GET_JOB_BY_ID}")
    private String PM_MEASUREMENT_COLLECTION_GET_JOB_BY_ID = "";
    @Value("${PM_MEASUREMENT_COLLECTION_JOBS_GET}")
    private String PM_MEASUREMENT_COLLECTION_JOBS_GET = "";
    @Value("${CATALOG_GET_SERVICE_BY_ID}")
    private String CATALOG_GET_SERVICE_BY_ID = "";
    @Value("${CATALOG_UPD_SERVICE}")
    private String CATALOG_UPD_SERVICE = "";
    @Value("${CATALOG_GET_RESOURCE_BY_ID}")
    private String CATALOG_GET_RESOURCE_BY_ID = "";
    @Value("${CATALOG_UPD_RESOURCE}")
    private String CATALOG_UPD_RESOURCE = "";
    @Value("${CATALOG_UPDADD_RESOURCESPEC}")
    private String CATALOG_UPDADD_RESOURCESPEC = "";
    @Value("${PM_MEASUREMENT_COLLECTION_JOB_GET_INPROGRESS_OR_PENDING}")
    private String PM_MEASUREMENT_COLLECTION_JOB_GET_INPROGRESS_OR_PENDING;
    @Value("${CATALOG_GET_SERVICESPEC_BY_ID}")
    private String CATALOG_GET_SERVICESPEC_BY_ID = "";


    private ProducerTemplate template;

    public MetricoCommonMethods(ProducerTemplate template) {
        this.template = template;
    }

    @Override
    public void configure() throws Exception {
        // TODO Auto-generated method stub
    }

    /**
     * Retrieves a Measurement Collection Job from the database using the given job ID.
     *
     * @param mcjId the ID of the Measurement Collection Job to retrieve
     * @return the retrieved Measurement Collection Job, or null if the job could not be retrieved
     */
    public MeasurementCollectionJob retrieveMeasurementCollectionJob(String mcjId) {

        logger.debug("will retrieve Measurement Collection Job with mcjId = {}.", mcjId);
        try {
            Object response = template.requestBody(PM_MEASUREMENT_COLLECTION_GET_JOB_BY_ID, mcjId);
            if (!(response instanceof String)) {
                logger.error("Measurement Collection Job object is wrong.");
                return null;
            }
            logger.debug("retrieveMeasurementCollectionJobById response is: " + response);
            MeasurementCollectionJob mcj = JsonUtil.toJsonObj((String) response, MeasurementCollectionJob.class);
            return mcj;
        } catch (Exception e) {
            logger.error("Cannot retrieve Measurement Collection Job details from database. " + e.toString());
        }
        return null;
    }

    /**
     * Retrieves a Measurement Collection Job from the database using the given job reference.
     *
     * @param mcjRef the reference of the Measurement Collection Job to retrieve
     * @return the retrieved Measurement Collection Job, or null if the job could not be retrieved
     */
    public MeasurementCollectionJob retrieveMeasurementCollectionJob(MeasurementCollectionJobRef mcjRef) {
        String mcjId = mcjRef.getId();

        return retrieveMeasurementCollectionJob(mcjId);
    }

    /**
     * Retrieves a list of all Measurement Collection Jobs from the database.
     *
     * @return a list of all Measurement Collection Jobs, or null if the jobs could not be retrieved
     */
    public List<MeasurementCollectionJob> listMeasurementCollectionJob() {

        logger.debug("will retrieve all Measurement Collection Jobs from database");
        try {
            Object response = template.requestBody(PM_MEASUREMENT_COLLECTION_JOBS_GET, "");
            if (!(response instanceof String)) {
                logger.error("List<MeasurementCollectionJob> object is wrong.");
                return null;
            }
            logger.debug("listMeasurementCollectionJob response is: " + response);
            //List<MeasurementCollectionJob> allMcjs = JsonUtil.toJsonObj((String) response, List<MeasurementCollectionJob.class);
            ObjectMapper mapper = new ObjectMapper();
            List<MeasurementCollectionJob> allMcjs = mapper.readValue((String) response, mapper.getTypeFactory().constructCollectionType(List.class, MeasurementCollectionJob.class));
            return allMcjs;

        } catch (Exception e) {
            logger.error("Cannot retrieve Measurement Collection Job details from database: {}. ", e.toString());
        }
        return null;
    }

    /**
     * Retrieves a list of Measurement Collection Jobs from the database that have an execution state of either Pending or InProgress.
     *
     * @return a list of unfinished Measurement Collection Jobs, or null if no jobs match the criteria or if an error occurs
     */
    public ArrayList<MeasurementCollectionJob> listPendingOrInProgressMeasurementCollectionJobs() {
        logger.debug("===== Will try to retrieve Measurement Collection Jobs with ExecutionStateType Pending or InProgress from the database ====");

        ArrayList<MeasurementCollectionJob> pendingOrInProgressMeasurementCollectionJobs = null;

        try {
            Object response = template.requestBody(PM_MEASUREMENT_COLLECTION_JOB_GET_INPROGRESS_OR_PENDING, "");

            logger.debug("will retrieve Measurement Collection Jobs with Execution State  " + ExecutionStateType.PENDING.toString() + " from catalog response: " + response.getClass());
            if (!(response instanceof String)) {
                logger.error("===== List<MeasurementCollectionJob> object is wrong ====");
                return null;
            } else if (((String) response).isEmpty() || ((String) response).equals("[]")) {
                logger.debug("===== No PENDING OR IN_PROGRESS Measurement Collection Jobs from previous sessions =====");
                return null;
            } else {
                logger.debug("listPendingOrInProgressMeasurementCollectionJobs response is: " + response);

                ObjectMapper objectMapper = new ObjectMapper();
                pendingOrInProgressMeasurementCollectionJobs = objectMapper.readValue(
                        (String) response,
                        objectMapper.getTypeFactory().constructCollectionType(ArrayList.class, MeasurementCollectionJob.class)
                );

                logger.info("===== Found PENDING or IN_PROGRESS Measurement Collection Jobs from previous sessions =====");
            }
        } catch (Exception e) {
            logger.error("Cannot retrieve/filter Measurement Collection Job details from the database. {}", e.toString());
        }
        return pendingOrInProgressMeasurementCollectionJobs;
    }


    /**
     * Updates the related resource state based on the given Measurement Collection Job.
     *
     * @param givenMCJ the Measurement Collection Job containing the related Service UUID and execution state
     */
    public void updateRelatedResource(@NotNull MeasurementCollectionJob givenMCJ) {
        if(givenMCJ.getExecutionState()==null){
            givenMCJ.setExecutionState(ExecutionStateType.PENDING);
        }
        String servUUID = givenMCJ.getProducingApplicationId();//this contains the related ServiceUUID with this MCJ
        logger.debug("Update Resource related to Service with ServiceId {}.", servUUID);
        Service aService = retrieveService(servUUID);

        if (aService != null && aService.getSupportingResource().size() > 0) {
            ResourceRef resRef = aService.getSupportingResource().stream().findFirst().get();
            logger.debug("Update Resource with ResourceId {}.", resRef.getId());

            ResourceUpdate rup = new ResourceUpdate();
            Characteristic resCharacteristicItem = new Characteristic();
            resCharacteristicItem.setName("_MT_MCJ_REF");
            resCharacteristicItem.setValueType("TEXT");
            Any val = new Any();
            val.setValue(givenMCJ.getUuid());
            val.setAlias("");
            resCharacteristicItem.setValue(val);
            rup.addResourceCharacteristicItem(resCharacteristicItem);

            if (givenMCJ.getExecutionState().equals(ExecutionStateType.FAILED)) {
                rup.setResourceStatus(ResourceStatusType.SUSPENDED);
            } else {
                rup.setResourceStatus(ResourceStatusType.AVAILABLE);
            }
            updateResourceById(resRef.getId(), rup);
        } else {
            logger.error("No related Resource found for Service with Service Id {}.", servUUID);
        }
    }


    /**
     * Creates or updates a LogicalResourceSpecification in the catalog using the given name, category, and version.
     *
     * @param s the ResourceSpecificationCreate object containing the specification details
     * @return the created or updated LogicalResourceSpecification instance, or null if the operation failed
     */
    public LogicalResourceSpecification createOrUpdateResourceSpecByNameCategoryVersion(ResourceSpecificationCreate s) {
        logger.info("==== Will try to Create or Update a Resource Specification for METRICO ====");
        try {
            Map<String, Object> map = new HashMap<>();
            map.put("aname", s.getName());
            map.put("aversion", s.getVersion());
            map.put("acategory", s.getCategory());
            logger.info("will createOrUpdateResourceSpecByNameCategoryVersion ");
            Object response = template.requestBodyAndHeaders(CATALOG_UPDADD_RESOURCESPEC, JsonUtil.toJsonString(s), map);
            assert response instanceof String : "ResourceSpecification  object is wrong.";
            LogicalResourceSpecification rs = JsonUtil.toJsonObj((String) response, LogicalResourceSpecification.class);
            logger.info("==== Created or Updated a Resource Specification for METRICO ====");
            return rs;
        } catch (Exception e) {
            logger.error("Cannot create ResourceSpecification");
            e.printStackTrace();
        }
        return null;
    }


    /**
     * Updates a Service in the catalog using the given service ID and update information.
     *
     * @param serviceId                 the ID of the service to update
     * @param serviceUpdate             the update information for the service
     * @param triggerServiceActionQueue flag to trigger the service action queue
     * @return the updated Service instance, or null if the service could not be updated
     */
    public Service updateService(String serviceId, ServiceUpdate serviceUpdate, boolean triggerServiceActionQueue) {
        logger.info("will update Service : " + serviceId);
        try {
            Map<String, Object> map = new HashMap<>();
            map.put("serviceid", serviceId);
            map.put("triggerServiceActionQueue", triggerServiceActionQueue);

            Object response = template.requestBodyAndHeaders(CATALOG_UPD_SERVICE, JsonUtil.toJsonString(serviceUpdate), map);

            assert response instanceof String : "Service Instance object is wrong.";
            Service serviceInstance = JsonUtil.toJsonObj((String) response, Service.class);
            return serviceInstance;


        } catch (Exception e) {
            logger.error("Cannot update Service with serviceId {} : {}", serviceId, e.toString());
        }
        return null;

    }

    /**
     * Updates a Resource in the catalog using the given resource ID and update information.
     *
     * @param oslResourceId the ID of the resource to update
     * @param rs            the update information for the resource
     * @return the updated LogicalResource instance, or null if the resource could not be updated
     */
    public Resource updateResourceById(String oslResourceId, ResourceUpdate rs) {
        logger.debug("will update Resource with id: " + oslResourceId);
        try {
            Map<String, Object> map = new HashMap<>();
            map.put("resourceId", oslResourceId);
            map.put("triggerServiceActionQueue", false);

            Object response = template.requestBodyAndHeaders(CATALOG_UPD_RESOURCE, JsonUtil.toJsonString(rs), map);
            assert response instanceof String : "Service Instance object is wrong.";
            LogicalResource resourceInstance = JsonUtil.toJsonObj((String) response, LogicalResource.class);
            return resourceInstance;
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("Cannot update Service with id {} : {} ", oslResourceId, e.toString());
        }
        return null;
    }


    /**
     * Retrieves a Service instance from the catalog using the given service ID.
     *
     * @param serviceID the ID of the service to retrieve
     * @return the retrieved Service instance, or null if the service could not be retrieved
     */
    public Service retrieveService(String serviceID) {
        logger.info("will retrieve Service instance from catalog serviceID=" + serviceID);
        try {
            Object response = template.requestBody(CATALOG_GET_SERVICE_BY_ID, serviceID);

            if (!(response instanceof String)) {
                logger.error("Service object is wrong.");
                return null;
            }
            return JsonUtil.toJsonObj((String) response, Service.class);

        } catch (Exception e) {
            logger.error("Cannot retrieve Service details from catalog. " + e.toString());
        }
        return null;
    }


    /**
     * Retrieves a Resource instance from the catalog using the given resource ID.
     *
     * @param resourceID the ID of the resource to retrieve
     * @return the retrieved LogicalResource instance, or null if the resource could not be retrieved
     */
    public Resource retrieveResource(@NotNull String resourceID) {
        logger.info("will retrieve Resource instance from catalog: resourceID = {}", resourceID);
        try {
            Object response = template.requestBody(CATALOG_GET_RESOURCE_BY_ID, resourceID);

            if (!(response instanceof String)) {
                logger.error("resource object is wrong.");
                return null;
            }
            LogicalResource resourceInstance = JsonUtil.toJsonObj((String) response, LogicalResource.class);

            return resourceInstance;

        } catch (Exception e) {
            logger.error("Cannot retrieve LogicalResource details from catalog. {}", e.toString());
        }
        return null;

    }


    /**
     * Updates a Measurement Collection Job in the database using the given job ID and update information.
     *
     * @param mcjId  the ID of the Measurement Collection Job to update
     * @param mcjMVO the update information for the Measurement Collection Job
     * @return the updated Measurement Collection Job, or null if the job could not be updated
     */
    public MeasurementCollectionJob updateMeasurementCollectionJobById(String mcjId, MeasurementCollectionJobMVO mcjMVO) {
        MeasurementCollectionJob measurementCollectionJob;
        logger.debug("will update MeasurementCollectionJob with id {} and \nMeasurementCollectionJobMVO.toString():\n{}", mcjId, JsonUtil.toJsonString(mcjMVO));

        try {
            Map<String, Object> map = new HashMap<>();
            map.put("mcjid", mcjId);

            Object response = template.requestBodyAndHeaders(PM_MEASUREMENT_COLLECTION_JOB_UPDATE, JsonUtil.toJsonString(mcjMVO), map);
            logger.debug("JsonUtil.toJsonString(mcjMVO): \n{}", JsonUtil.toJsonString(mcjMVO));
            if (!(response instanceof String)) {
                logger.error("MeasurementCollectionJob object is wrong.");
                return null;
            }
            measurementCollectionJob = JsonUtil.toJsonObj((String) response, MeasurementCollectionJob.class);
            logger.info("response from PM_MEASUREMENT_COLLECTION_JOB_UPDATE: \n{}", response);
            return measurementCollectionJob;
        } catch (Exception e) {
            e.printStackTrace();
            logger.error("Cannot update MeasurementCollectionJob with id {} : {}", mcjId, e.toString());
        }
        return null;
    }

    public ServiceSpecification retrieveServiceSpecificationById(String serviceSpecId) {

        logger.debug("will retrieve Service Specification with id = {}.", serviceSpecId);
        try {
            Object response = template.requestBody(CATALOG_GET_SERVICESPEC_BY_ID, serviceSpecId);
            if (!(response instanceof String)) {
                logger.error("Service Specification object is wrong.");
                return null;
            }
            logger.debug("retrieveServiceSpecificationById response is: {}", response);
            ServiceSpecification serviceSpec = JsonUtil.toJsonObj((String) response, ServiceSpecification.class);
            return serviceSpec;
        } catch (Exception e) {
            logger.error("Cannot retrieve Service Specification details from database. " + e.toString());
        }
        return null;
    }

    public boolean notNullOrEmpty(String str) {
        return str != null && !str.isEmpty();
    }

}
