package org.etsi.osl.metrico.services;

import jakarta.validation.constraints.NotNull;
import org.apache.camel.ProducerTemplate;
import org.apache.camel.builder.RouteBuilder;
import org.etsi.osl.metrico.JsonUtil;
import org.etsi.osl.metrico.mapper.DataAccessEndpointMapper;
import org.etsi.osl.metrico.mapper.JobMapper;
import org.etsi.osl.metrico.model.Job;
import org.etsi.osl.metrico.prometheus.PrometheusQueries;
import org.etsi.osl.metrico.repo.JobRepository;
import org.etsi.osl.metrico.reposervices.JobRepoService;
import org.etsi.osl.tmf.common.model.Any;
import org.etsi.osl.tmf.common.model.ELifecycle;
import org.etsi.osl.tmf.common.model.EValueType;
import org.etsi.osl.tmf.common.model.service.ResourceRef;
import org.etsi.osl.tmf.pm628.model.*;
import org.etsi.osl.tmf.rcm634.model.ResourceSpecificationCreate;
import org.etsi.osl.tmf.ri639.model.LogicalResource;
import org.etsi.osl.tmf.ri639.model.Resource;
import org.etsi.osl.tmf.ri639.model.ResourceUpdate;
import org.etsi.osl.tmf.so641.model.ServiceOrder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.context.event.ApplicationStartedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;

@Service
public class MetricoService extends RouteBuilder {

    private static final Logger logger = LoggerFactory.getLogger(JobService.class);


    public static final String OSL_METRICO_RSPEC_NAME = "metrico.osl.etsi.org";
    public static final String OSL_METRICO_RSPEC_VERSION = "0.0.1";
    public static final String OSL_METRICO_RSPEC_CATEGORY = "metrico.osl.etsi.org/v1";
    public static final String OSL_METRICO_RESOURCE_CATEGORY = "metrico.osl.etsi.org/v1";
    public static final String OSL_METRICO_RSPEC_TYPE = "LogicalResourceSpecification";
    public static final String OSL_METRICO_RSPEC_DESCRIPTION = "This Specification is used to describe a generic KubernetesCRD";

    @Value("${PM_MEASUREMENT_COLLECTION_JOB_UPDATE}")
    private String PM_MEASUREMENT_COLLECTION_JOB_UPDATE = "";

    @Value("${PM_MEASUREMENT_COLLECTION_GET_JOB_BY_ID}")
    private String PM_MEASUREMENT_COLLECTION_GET_JOB_BY_ID = "";   
    
    @Autowired
    private JobRepoService jobRepoService;
    

    @Autowired
    private JobRepository jobRepository;
    

    @Autowired
    private final PrometheusQueries prometheusQueries;
    private final ProducerTemplate producerTemplate;

    public MetricoService(PrometheusQueries prometheusQueries, ProducerTemplate producerTemplate) {
        this.prometheusQueries = prometheusQueries;
        this.producerTemplate = producerTemplate;
    }
    
    
    /**
     * This one is executed when the cridge service application starts
     * @param event
     */
    @EventListener(ApplicationStartedEvent.class)
    public void onApplicationEvent() {
      
      registerMetricoResourceSpec();
      
      List<Job> jobsPending = jobRepoService.findAll();
      logger.info("===== Pending jobs from previous sessions =====");
      for (Job job : jobsPending) {
        logger.info("try to resume jobuuid: {}, state:{}, start: {}, end:{}, mcjid: {} ", job.getUuid(), job.getState(), job.getStartDateTime(), job.getEndDateTime(), job.getMeasurementCollectionJobRef()  );

        
        //delete previous job from DB. a new one will be created
        jobRepoService.softDeleteJob(job);
        
        MeasurementCollectionJobRef mjref = new MeasurementCollectionJobRef();
        mjref.setId( job.getMeasurementCollectionJobRef().toString() );
        this.startPeriodicQueryToPrometheusRef(mjref );
      }

      logger.info("===== Pending jobs from previous sessions done =====");
      
    }


    private void registerMetricoResourceSpec() {
      logger.info("===== Pending jobs from previous sessions =====");
      
      
      ResourceSpecificationCreate rsc = new ResourceSpecificationCreate();
      rsc.setName( "METRICO_RESOURCE_SPECIFICATION" );
      rsc.setCategory( OSL_METRICO_RSPEC_CATEGORY );
      rsc.setVersion( OSL_METRICO_RSPEC_VERSION );
      rsc.setDescription( OSL_METRICO_RSPEC_DESCRIPTION );
      rsc.setType( OSL_METRICO_RSPEC_TYPE );
      
      rsc.setLifecycleStatus( ELifecycle.ACTIVE.getValue() );
      rsc.addResourceSpecificationCharacteristicItemShort( "_MT_CHARACTERISTIC_NAME", "", EValueType.TEXT.getValue(), "Used for providing the characteristic of service _MT_SERVICEUUID to update", false);
      rsc.addResourceSpecificationCharacteristicItemShort( "_MT_SERVICEUUID", "", EValueType.TEXT.getValue(), "Used for providing the id of the service to update", false);
      rsc.addResourceSpecificationCharacteristicItemShort( "_MT_END_TIME", "", EValueType.TEXT.getValue(), "Used for providing the end time", false);
      rsc.addResourceSpecificationCharacteristicItemShort( "_MT_START_TIME", "", EValueType.TEXT.getValue(), "Used for providing start time", false);
      rsc.addResourceSpecificationCharacteristicItemShort( "_MT_RECURRING_INTERVAL", "G_1MN", EValueType.TEXT.getValue(), "Used for providing the polling interval of prometheus", false);
      rsc.addResourceSpecificationCharacteristicItemShort( "_MT_TYPE", "PROMETHEUS", EValueType.TEXT.getValue(), "Used for providing ", false);
      rsc.addResourceSpecificationCharacteristicItemShort( "_MT_QUERY", "", EValueType.TEXT.getValue(), "Used for providing the query towards prometheus", false);
      rsc.addResourceSpecificationCharacteristicItemShort( "_MT_URL", "", EValueType.TEXT.getValue(), "Used for providing the prometheus URL", false);
      
      
      prometheusQueries.createOrUpdateResourceSpecByNameCategoryVersion(rsc);
      
    }


    @Autowired
    private ProducerTemplate template;

    public String[] queryToPrometheus(@NotNull MeasurementCollectionJob givenMCJ){
        DataAccessEndpoint givenDataAccessEndpoint = givenMCJ.getDataAccessEndpoint().get(0);
        Job job = JobMapper.measurementCollectionJobMapToJob(givenMCJ);
        String promURL = job.getDataAccessEndPointUri().getScheme() + "://" + job.getDataAccessEndPointUri().getAuthority();
        String promQuery = job.getDataAccessEndPointUri().getQuery();
        promQuery = promQuery.replace("query=", "");

        String [] promResponse = prometheusQueries.sendQueryToPrometheus(promURL, promQuery, givenMCJ, job).split("\n");

        DataFilterTemplate filterTemplate = new DataFilterTemplate();
        filterTemplate.setName(promQuery);
        DataFilterAttributeStringArray stringArray = new DataFilterAttributeStringArray();
        stringArray.setValue(List.of(promResponse));
        DataFilterMapItem dataFilterMapItem = new DataFilterMapItem();
        dataFilterMapItem.setFilterTemplate(filterTemplate);
        dataFilterMapItem.setStringArray(stringArray);
        DataFilterMap dataFilterMap = new DataFilterMap();
        dataFilterMap.addMappingsItem(dataFilterMapItem);
        givenDataAccessEndpoint.setUriQueryFilter(dataFilterMap);

        List<DataAccessEndpointMVO> newDataAccessEndpointMVO = new ArrayList<>();
        newDataAccessEndpointMVO.add(DataAccessEndpointMapper.INSTANCE.toDataAccessEndpointMVO(givenDataAccessEndpoint));

//        MeasurementCollectionJobMVO promResponseMCJ = new MeasurementCollectionJobMVO();

//        promResponseMCJ.setDataAccessEndpoint(newDataAccessEndpointMVO);
//        producerTemplate.sendBodyAndHeader(PM_MEASUREMENT_COLLECTION_JOB_UPDATE, JsonUtil.toJsonString(promResponseMCJ),"mcjid",givenMCJ.getUuid() );

        return promResponse;
    }

    public void startPeriodicQueryToPrometheus(@NotNull MeasurementCollectionJob givenMCJ){
        Job job = JobMapper.measurementCollectionJobMapToJob(givenMCJ);
        String promURL = job.getDataAccessEndPointUri().getScheme() + "://" + job.getDataAccessEndPointUri().getAuthority();
        String promQuery = job.getDataAccessEndPointUri().getQuery();

        job =  prometheusQueries.startPeriodicQuery(promURL, promQuery, job, givenMCJ );

        if (job.getState() == ExecutionStateType.FAILED) {
            logger.atError().setMessage("Periodic query failed to start due to internal error.").log();
            
        } else {
            logger.atInfo().setMessage("Periodic query started, with ID: " + job.getUuid()).log();
        }

        givenMCJ.setExecutionState(job.getState());

        producerTemplate.sendBodyAndHeader( PM_MEASUREMENT_COLLECTION_JOB_UPDATE, JsonUtil.toJsonString(givenMCJ), "mcjid", givenMCJ.getUuid() );
        updateRelatedResource(givenMCJ);
    }



    public void startPeriodicQueryToPrometheusRef(@NotNull MeasurementCollectionJobRef mcjRef){
        MeasurementCollectionJob givenMCJ = retrieveMeasurementCollectionJob(mcjRef);

        startPeriodicQueryToPrometheus(givenMCJ);
    }
    
    
    public void startPeriodicQueryToPrometheusEvent(@NotNull MeasurementCollectionJobCreateEvent mcjevent){
      
      MeasurementCollectionJob givenMCJ = retrieveMeasurementCollectionJob( mcjevent.getEvent().getMeasurementCollectionJob().getId() );

      if ( givenMCJ != null) {
        startPeriodicQueryToPrometheus(givenMCJ);        
      } else {

        logger.error("=======> CANNOT retrieve Measurement Collection Job with mcjId = " + mcjevent.getEvent().getMeasurementCollectionJob().getId()  +" from activeMQ");
      }
  }

    public MeasurementCollectionJob retrieveMeasurementCollectionJob(String mcjId) {

        logger.debug("will retrieve Measurement Collection Job with mcjId = " + mcjId +" from database");
        try {
            Object response = template.
                    requestBody( PM_MEASUREMENT_COLLECTION_GET_JOB_BY_ID, mcjId);
            if ( !(response instanceof String)) {
                logger.error("Measurement Collection Job object is wrong.");
                return null;
            }
            logger.debug("retrieveMeasurementCollectionJobById response is: " + response);
            MeasurementCollectionJob mcj = JsonUtil.toJsonObj( (String)response, MeasurementCollectionJob.class);
            return mcj;
        }catch (Exception e) {
            logger.error("Cannot retrieve Measurement Collection Job details from database. " + e.toString());
        }
        return null;
    }

    public MeasurementCollectionJob retrieveMeasurementCollectionJob(MeasurementCollectionJobRef mcjRef) {
        String mcjId = mcjRef.getId();

        return retrieveMeasurementCollectionJob(mcjId);
    }
    
    
    private void updateRelatedResource(@NotNull MeasurementCollectionJob givenMCJ) {
      //retrieve related service from inventory
      //retrieve related resource from inventory
      //patch resource state
      String servUUID = givenMCJ.getProducingApplicationId();//this contains the related ServiceUUID with this MCJ

      logger.debug("updateRelatedResource servUUID = " + servUUID );
      org.etsi.osl.tmf.sim638.model.Service aService = prometheusQueries.retrieveService(servUUID);
      
      if ( aService!=null && aService.getSupportingResource().size()>0) {        

        ResourceRef resRef = aService.getSupportingResource().stream().findFirst().get();
        
        //Resource aResource = retrieveResource(resRef.getId());
        logger.debug("updateRelatedResource resRef = " + resRef.getId() );
        
        ResourceUpdate rup = new ResourceUpdate();
        org.etsi.osl.tmf.ri639.model.Characteristic resCharacteristicItem =  new org.etsi.osl.tmf.ri639.model.Characteristic();
        resCharacteristicItem.setName( "_MT_MCJ_REF" );
        resCharacteristicItem.setValueType( "TEXT" );
        Any val = new Any();
        val.setValue( givenMCJ.getUuid() );
        val.setAlias( "" );
        resCharacteristicItem.setValue( val );
        rup.addResourceCharacteristicItem(  resCharacteristicItem );
        
        if ( givenMCJ.getExecutionState().equals( ExecutionStateType.FAILED  ) ) {
          rup.setResourceStatus( org.etsi.osl.tmf.ri639.model.ResourceStatusType.SUSPENDED );          
        } else {
          rup.setResourceStatus( org.etsi.osl.tmf.ri639.model.ResourceStatusType.AVAILABLE );
        }
        
        prometheusQueries.updateResourceById(resRef.getId(), rup );
        
      }
      
    }

    @Override
    public void configure() throws Exception {
      // TODO Auto-generated method stub
      
    }
    
    
 

}
