package org.etsi.osl.metrico.model;

import org.etsi.osl.tmf.pm628.model.ExecutionStateType;
import org.junit.jupiter.api.Test;

import java.time.OffsetDateTime;
import java.time.temporal.ChronoUnit;
import java.util.UUID;

import static org.junit.jupiter.api.Assertions.*;

class JobTest {

    @Test
    void testDefaultConstructor() {
        Job job = new Job();
        assertEquals(ExecutionStateType.PENDING, job.getState());
        assertEquals(10, job.getExecutionInterval());
        assertNotNull(job.getStartDateTime());
        assertTrue(ChronoUnit.SECONDS.between(OffsetDateTime.now(), job.getStartDateTime()) <= 1);
    }

    @Test
    void testConstructorWithStopDateTime() {
        OffsetDateTime stopDateTime = OffsetDateTime.now().plusDays(1);
        Job job = new Job(stopDateTime);
        assertEquals(stopDateTime, job.getEndDateTime());
        assertEquals(ExecutionStateType.PENDING, job.getState());
        assertEquals(10, job.getExecutionInterval());
    }

    @Test
    void testConstructorWithStartAndStopDateTime() {
        OffsetDateTime startDateTime = OffsetDateTime.now();
        OffsetDateTime stopDateTime = startDateTime.plusDays(1);
        Job job = new Job(startDateTime, stopDateTime);
        assertEquals(startDateTime, job.getStartDateTime());
        assertEquals(stopDateTime, job.getEndDateTime());
        assertEquals(ExecutionStateType.PENDING, job.getState());
        assertEquals(10, job.getExecutionInterval());
    }

    @Test
    void testConstructorWithAllParams() {
        OffsetDateTime startDateTime = OffsetDateTime.now();
        OffsetDateTime stopDateTime = startDateTime.plusDays(1);
        Integer executionInterval = 15;
        Job job = new Job(startDateTime, stopDateTime, executionInterval);
        assertEquals(startDateTime, job.getStartDateTime());
        assertEquals(stopDateTime, job.getEndDateTime());
        assertEquals(executionInterval, job.getExecutionInterval());
        assertEquals(ExecutionStateType.PENDING, job.getState());
    }

    @Test
    void testToString() {
        Job job = new Job();
        job.setUuid(UUID.fromString("123e4567-e89b-12d3-a456-426614174000"));
        job.setState(ExecutionStateType.PENDING);
        job.setStartDateTime(OffsetDateTime.parse("2023-01-01T10:15:30+01:00"));
        job.setEndDateTime(OffsetDateTime.parse("2023-01-02T10:15:30+01:00"));
        job.setExecutionInterval(10);
        job.setDataAccessEndPointRef(UUID.fromString("123e4567-e89b-12d3-a456-426614174001"));
        job.setScheduleDefinitionRef(UUID.fromString("123e4567-e89b-12d3-a456-426614174002"));
        job.setMeasurementCollectionJobRef(UUID.fromString("123e4567-e89b-12d3-a456-426614174003"));
        job.setDeleted(false);

        String expected = "Job{uuid=123e4567-e89b-12d3-a456-426614174000, state=pending, startDateTime=2023-01-01T10:15:30+01:00, endDateTime=2023-01-02T10:15:30+01:00, executionInterval=10, dataAccessEndPointRef=123e4567-e89b-12d3-a456-426614174001, scheduleDefinitionRef=123e4567-e89b-12d3-a456-426614174002, measurementCollectionJobRef=123e4567-e89b-12d3-a456-426614174003, deleted=false}";
        assertEquals(expected, job.toString());
    }

}