package org.etsi.osl.metrico.prometheus;

import java.io.IOException;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import javax.net.ssl.SSLException;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.camel.ProducerTemplate;
import org.etsi.osl.metrico.model.Job;
import org.etsi.osl.metrico.services.JobService;
import org.etsi.osl.tmf.common.model.Any;
import org.etsi.osl.tmf.common.model.service.Characteristic;
import org.etsi.osl.tmf.pm628.model.ExecutionStateType;
import org.etsi.osl.tmf.pm628.model.MeasurementCollectionJob;
import org.etsi.osl.tmf.ri639.model.LogicalResource;
import org.etsi.osl.tmf.ri639.model.Resource;
import org.etsi.osl.tmf.ri639.model.ResourceUpdate;
import org.etsi.osl.tmf.sim638.model.ServiceUpdate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpStatusCode;
import org.springframework.stereotype.Component;
import org.springframework.web.reactive.function.client.WebClient;
import org.springframework.web.util.UriComponentsBuilder;
import jakarta.validation.constraints.NotNull;
import reactor.core.publisher.Mono;

@Component
public class PrometheusQueries {

    private static final Logger logger = LoggerFactory.getLogger(PrometheusQueries.class);
    private final JobService jobService;
    

    @Value("${CATALOG_GET_SERVICE_BY_ID}")
    private String CATALOG_GET_SERVICE_BY_ID = "";
    

    @Value("${CATALOG_GET_RESOURCE_BY_ID}")
    private String CATALOG_GET_RESOURCE_BY_ID = "";
    

    @Value("${CATALOG_UPD_RESOURCE}")
    private String CATALOG_UPD_RESOURCE = "";
    

    @Value("${CATALOG_UPD_SERVICE}")
    private String CATALOG_UPD_SERVICE = "";
    

    @Autowired
    private ProducerTemplate template;
    

    public PrometheusQueries(JobService jobService) {
        this.jobService = jobService;
    }

    public String sendQueryToPrometheus(String prometheusUrl, String query, MeasurementCollectionJob mcj, Job job) {
        //RestTemplate restTemplate = new RestTemplate();

        UriComponentsBuilder builder = UriComponentsBuilder.fromHttpUrl(prometheusUrl)
                .path("/api/v1/query")
                .query(query);
        logger.atInfo().log("Sent query at prometheus with URL: " + prometheusUrl + " with query: " + query);
        logger.atInfo().log("Sent query  builder.toUriString(): " + builder.toUriString() );

        //ResponseEntity<String> response = restTemplate.getForEntity(builder.toUriString(), String.class);
        
        WebClient webclient = WebClient.create();

        String response=""; 
        
        if ( webclient!=null ) {
          
          try {

            String url = builder.toUriString();
              
            response = webclient.get()
                    .uri(url)
                      //.header("Authorization", "Basic " + encodedClientData)
                        //.attributes( ServletOAuth2AuthorizedClientExchangeFilterFunction.clientRegistrationId("authOpensliceProvider"))
                        .retrieve()
                        .onStatus(HttpStatusCode::is4xxClientError, r -> {
                            logger.error("4xx eror");
                            return Mono.error(new RuntimeException("4xx"));
                          })
                          .onStatus(HttpStatusCode::is5xxServerError, r -> {
                              logger.error("5xx eror");
                            return Mono.error(new RuntimeException("5xx"));
                          })
                      .bodyToMono( new ParameterizedTypeReference<String>() {})
                      .block();
        
            logger.atDebug().log("Received " + response);
            
            String [] promResponse =response.split("\n");
            
            
            
            }catch (Exception e) {
                logger.error(" error on web client request");
                response = "{\"status\":\""+ e.getLocalizedMessage() +"\"";
                e.printStackTrace();
            }
           
      } else  {
          logger.error("WebClient is null. Cannot be created.");
          response = "{\"status\":\"Cannot connect\"";
      }
        
        

      ServiceUpdate su = new ServiceUpdate();

      //patch TMF service
      String serviceUUID = mcj.getConsumingApplicationId();
      logger.atDebug().log("serviceUUID= " +  serviceUUID);
      Characteristic serviceCharacteristicItem =  new Characteristic();
      serviceCharacteristicItem.setName( mcj.getOutputFormat() );
      serviceCharacteristicItem.setValueType( "TEXT" );                    
      Any val = new Any();
      val.setValue( response );
      val.setAlias( "");        
      serviceCharacteristicItem.setValue( val );
      su.addServiceCharacteristicItem(serviceCharacteristicItem);
      updateService(serviceUUID, su , true);
      
       return response;
    }
    
   

    public Job startPeriodicQuery(String prometheusUrl, String query, final Job ajob, MeasurementCollectionJob mcj) {

      Job job = ajob;
      if (job.getStartDateTime() == null){
        job.setStartDateTime(OffsetDateTime.now());
      }
      if (job.getEndDateTime() == null){
          job.setEndDateTime(job.getStartDateTime().plusHours(1));
      }
      if(job.getExecutionInterval() == null){
          job.setExecutionInterval(180);
      }
    
        final Runnable queryHandler = () -> sendQueryToPrometheus(prometheusUrl, query, mcj, ajob);

        job = jobService.startJob(queryHandler, job);
        if (job.getState() == ExecutionStateType.FAILED) {
            jobService.stopJob(ajob);
            return null;
        }

        if (job.getEndDateTime() != null) {
            long stopAfterSeconds = Duration.between(OffsetDateTime.now(), job.getEndDateTime() ).getSeconds();

            JobService.getScheduler().schedule(() -> {
                jobService.stopJob(ajob);

            }, stopAfterSeconds, TimeUnit.SECONDS);
        }

        return job;
    }
    
    
    
    public Resource retrieveResource(@NotNull String resourceID) {
      logger.info("will retrieve Resource instance from catalog resourceID=" + resourceID   );
      try {
          Object response = template.
                  requestBody( CATALOG_GET_RESOURCE_BY_ID, resourceID);

          if ( !(response instanceof String)) {
              logger.error("resource object is wrong.");
              return null;
          }
          LogicalResource rInstance = toJsonObj( (String)response, LogicalResource.class);
          
          return rInstance;
          
      }catch (Exception e) {
          logger.error("Cannot retrieve LogicalResource details from catalog. " + e.toString());
      }
      return null;
      
    }
    
    /**
     * Ger service instance via bus
     * @param serviceID
     * @return
     */
    public org.etsi.osl.tmf.sim638.model.Service retrieveService(String serviceID) {
        logger.info("will retrieve Service instance from catalog serviceID=" + serviceID   );
        try {
            Object response = template.
                    requestBody( CATALOG_GET_SERVICE_BY_ID, serviceID);

            if ( !(response instanceof String)) {
                logger.error("Service object is wrong.");
                return null;
            }
            org.etsi.osl.tmf.sim638.model.Service serviceInstance = toJsonObj( (String)response, org.etsi.osl.tmf.sim638.model.Service.class); 
            //logger.debug("retrieveService response is: " + response);
            return serviceInstance;
            
        }catch (Exception e) {
            logger.error("Cannot retrieve Service details from catalog. " + e.toString());
        }
        return null;
    }
    
    
    public Resource updateResourceById(String oslResourceId, ResourceUpdate rs) {


      logger.debug("will update Resource : " + oslResourceId );
      try {
        Map<String, Object> map = new HashMap<>();
        map.put("resourceId", oslResourceId );
        map.put("triggerServiceActionQueue", false );

        Object response = template.requestBodyAndHeaders( CATALOG_UPD_RESOURCE, toJsonString(rs), map);

        if ( !(response instanceof String)) {
          logger.error("Service Instance object is wrong.");
        }

        LogicalResource resourceInstance = toJsonObj( (String)response, LogicalResource.class); 
        //logger.debug("createService response is: " + response);
        return resourceInstance;


      }catch (Exception e) {
        e.printStackTrace();
        logger.error("Cannot update Service: " + oslResourceId + ": " + e.toString());
      }
      return null;
    }

    
    /**
     * @param serviceId
     * @param s
     * @param triggerServiceActionQueue is a cryptic thing. However it is used as follows: if FALSE, to just update the service status in catalog without further taking any action.
     * if TRUE then the ServiceUpdate will trigger a ServiceActionQueue to further process the update. So this is needed to avoid these kinds of deadlocks
     * @return
     */
    public org.etsi.osl.tmf.sim638.model.Service updateService(String serviceId, ServiceUpdate s, boolean triggerServiceActionQueue) {
        logger.info("will update Service : " + serviceId );
        try {
            Map<String, Object> map = new HashMap<>();
            map.put("serviceid", serviceId );
            map.put("triggerServiceActionQueue", triggerServiceActionQueue );
            
            Object response = template.requestBodyAndHeaders( CATALOG_UPD_SERVICE, toJsonString(s), map);

            if ( !(response instanceof String)) {
                logger.error("Service Instance object is wrong.");
            }

            org.etsi.osl.tmf.sim638.model.Service serviceInstance = toJsonObj( (String)response, org.etsi.osl.tmf.sim638.model.Service.class); 
            //logger.debug("createService response is: " + response);
            return serviceInstance;
            
            
        }catch (Exception e) {
            logger.error("Cannot update Service: " + serviceId + ": " + e.toString());
        }
        return null;
        
    }

    
    
    
    
    static <T> T toJsonObj(String content, Class<T> valueType)  throws IOException {
        ObjectMapper mapper = new ObjectMapper();
        mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        return mapper.readValue( content, valueType);
    }

     static String toJsonString(Object object) throws IOException {
            ObjectMapper mapper = new ObjectMapper();
            mapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
            return mapper.writeValueAsString(object);
        }


}


