package org.etsi.osl.metrico.services;

import jakarta.validation.constraints.NotNull;
import org.apache.camel.ProducerTemplate;
import org.etsi.osl.metrico.mapper.JobMapper;
import org.etsi.osl.metrico.model.Job;
import org.etsi.osl.metrico.prometheus.PrometheusQueries;
import org.etsi.osl.tmf.pm628.model.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.time.OffsetDateTime;
import java.util.ArrayList;
import java.util.List;

@Service
public class MetricoService {

    private static final Logger logger = LoggerFactory.getLogger(JobService.class);

    @Value("{PM_MEASUREMENT_COLLECTION_JOB_UPDATE}")
    private String PM_MEASUREMENT_COLLECTION_JOB_UPDATE = "";

    private final PrometheusQueries prometheusQueries;
    private final ProducerTemplate producerTemplate;

    public MetricoService(PrometheusQueries prometheusQueries, ProducerTemplate producerTemplate) {
        this.prometheusQueries = prometheusQueries;
        this.producerTemplate = producerTemplate;
    }

    public String sendQueryToPrometheus(String promURL, String promQuery) {
        return prometheusQueries.sendQueryToPrometheus(promURL, promQuery);
    }

    public String[] queryToPrometheus(@NotNull MeasurementCollectionJob givenMCJ){
        DataAccessEndpoint givenDataAccessEndpoint = givenMCJ.getDataAccessEndpoint().get(0);
        Job job = JobMapper.measurementCollectionJobMapToJob(givenMCJ);
        String promURL = job.getDataAccessEndPointUri().getScheme() + "://" + job.getDataAccessEndPointUri().getAuthority();
        String promQuery = job.getDataAccessEndPointUri().getQuery();
        promQuery = promQuery.replace("query=", "");

        String [] promResponse = sendQueryToPrometheus(promURL, promQuery).split("\n");

        DataFilterTemplate filterTemplate = new DataFilterTemplate();
        filterTemplate.setName(promQuery);
        DataFilterAttributeStringArray stringArray = new DataFilterAttributeStringArray();
        stringArray.setValue(List.of(promResponse));
        DataFilterMapItem dataFilterMapItem = new DataFilterMapItem();
        dataFilterMapItem.setFilterTemplate(filterTemplate);
        dataFilterMapItem.setStringArray(stringArray);
        DataFilterMap dataFilterMap = new DataFilterMap();
        dataFilterMap.addMappingsItem(dataFilterMapItem);
        givenDataAccessEndpoint.setUriQueryFilter(dataFilterMap);

        List<DataAccessEndpoint> newDataAccessEndpoint = new ArrayList<>();
        newDataAccessEndpoint.add(givenDataAccessEndpoint);

        givenMCJ.setDataAccessEndpoint(newDataAccessEndpoint);

        producerTemplate.sendBody(PM_MEASUREMENT_COLLECTION_JOB_UPDATE, givenMCJ);

        return promResponse;
    }

    public String startPeriodicQueryToPrometheus(@NotNull MeasurementCollectionJob givenMCJ){
        Job job = JobMapper.measurementCollectionJobMapToJob(givenMCJ);
        String promURL = job.getDataAccessEndPointUri().getScheme() + "://" + job.getDataAccessEndPointUri().getAuthority();
        String promQuery = job.getDataAccessEndPointUri().getQuery();
        promQuery = promQuery.replace("query=", "");

        if (job.getStartDateTime() == null){
            job.setStartDateTime(OffsetDateTime.now());
        }
        if (job.getEndDateTime() == null){
            job.setEndDateTime(job.getStartDateTime().plusHours(1));
        }
        if(job.getExecutionInterval() == null){
            job.setExecutionInterval(180);
        }

        Job newPeriodicQuery =  prometheusQueries.startPeriodicQuery(promURL, promQuery,job.getStartDateTime(), job.getEndDateTime(), job.getExecutionInterval());

        if (newPeriodicQuery.getState() == ExecutionStateType.FAILED) {
            logger.atError().setMessage("Periodic query failed to start due to internal error.").log();
            return "Periodic query failed to start.";
        } else {
            logger.atInfo().setMessage("Periodic query started, with ID: " + newPeriodicQuery.getUuid()).log();
            return "Periodic query started with ID: " + newPeriodicQuery.getUuid();
        }
    }

}
