package org.etsi.osl.metrico.services;

import lombok.Getter;
import org.etsi.osl.metrico.model.Job;
import org.etsi.osl.metrico.reposervices.JobRepoService;
import org.etsi.osl.tmf.pm628.model.ExecutionStateType;
import org.etsi.osl.tmf.pm628.model.MeasurementCollectionJobRef;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.context.event.ApplicationStartedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.time.OffsetDateTime;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.*;

@Service
public class JobService {

    private static final Logger logger = LoggerFactory.getLogger(JobService.class);

    @Value("${METRICO_THREAD_POOL_SIZE}")
    private static int threadPoolSize;
    @Getter
    private static final ScheduledExecutorService scheduler = Executors.newScheduledThreadPool(threadPoolSize);

    private final JobRepoService jobRepoService;

    public JobService(JobRepoService jobRepoService) {
        this.jobRepoService = jobRepoService;
    }
    

    
    

    /**
     * Schedules a new job to be executed periodically based on the provided parameters.
     * <p>
     * This method initializes a new {@link Job} instance with the specified start and end date times, and execution interval.
     * It calculates the initial delay as the difference in seconds between the current time and the start date time.
     * The job is initially set to a PENDING state. It then attempts to schedule the job to run at a fixed rate,
     * starting after the initial delay and subsequently with the specified execution interval.
     * If the job is successfully scheduled, its state is updated to INPROGRESS, and it is logged.
     * In case of a scheduling failure due to a {@link RejectedExecutionException}, the job's state is set to FAILED,
     * and the error is logged.
     * </p>
     *
     * @param task the {@link Runnable} task that the job will execute
     * @param startDateTime the {@link OffsetDateTime} specifying when the job should start
     * @param endDateTime the {@link OffsetDateTime} specifying when the job should end
     * @param executionInterval the interval, in seconds, between successive executions of the job
     * @return the newly created and scheduled {@link Job} instance
     * @throws RejectedExecutionException if the task cannot be scheduled for execution
     */
    public Job startJob(Runnable task,  Job job) {
      
      
      if ( job.getEndDateTime().isBefore( OffsetDateTime.now() )) {
        job.setState(ExecutionStateType.FAILED);
        logger.error("Job with ID {} End Date is before now, the job will not be scheduled.", job.getUuid());
        return job;        
      }
      
        long initialDelay = Duration.between(OffsetDateTime.now(), job.getStartDateTime()).getSeconds();
        if (initialDelay<0) {
          initialDelay = 0;
        }
        job = jobRepoService.createAndSaveJob(job);
        
        try {
            ScheduledFuture<?> future = scheduler.scheduleAtFixedRate(task, initialDelay, job.getExecutionInterval(), TimeUnit.SECONDS);
            job.setFuture(future);
            job.setState(ExecutionStateType.INPROGRESS);
            logger.info("Job with ID {} started successfully.", job.getUuid());
        } catch (RejectedExecutionException e) {
            job.setState(ExecutionStateType.FAILED);
            logger.error("Job with ID {} could not be scheduled.", job.getUuid(), e);
        }
        

        job = jobRepoService.updateJob( job.getUuid(), job.getState());
        
        return job;
    }

    /**
     * Attempts to stop a job identified by its UUID.
     * <p>
     * This method checks if the job exists and its current state. If the job is already in a
     * {@link ExecutionStateType#CANCELLED} or {@link ExecutionStateType#COMPLETED} state, it logs
     * the status and returns. Otherwise, it attempts to cancel the job's future execution.
     * If the cancellation is successful, the job's state is set to {@link ExecutionStateType#CANCELLED},
     * and it logs the successful stop. If the job cannot be stopped (e.g., already completed or cancelled),
     * its state is set to {@link ExecutionStateType#PENDING}, and a warning is logged. If the job does not
     * exist, it logs a warning.
     * </p>
     *
     * @param jobId the UUID of the job to stop
     */
    public void stopJob(Job job) {

        if (job != null) {
            
            if (job.getState() == ExecutionStateType.CANCELLED ) {
                logger.info("Job with ID {} is already CANCELED.", job.getUuid());
                jobRepoService.softDeleteJob( job );
                return;
            } else if (job.getState() == ExecutionStateType.COMPLETED) {
                logger.info("Job with ID {} is already COMPLETED.", job.getUuid());
                jobRepoService.softDeleteJob( job );
                return;
            }
            if (job.getFuture() != null) {
                boolean wasCancelled = job.getFuture().cancel(true);
                if (wasCancelled) {
                    job.setState(ExecutionStateType.CANCELLED);
                    jobRepoService.updateJob(job.getUuid(), job.getState());
                    logger.info("Job with ID {} stopped successfully.", job.getUuid());
                } else {
                    job.setState(ExecutionStateType.PENDING);
                    jobRepoService.updateJob( job.getUuid(), job.getState());
                    logger.warn("Job with ID {} could not be stopped because it has already completed, has been cancelled, or could not be cancelled for some other reason.", job.getUuid());
                }
            }
        } else {
            logger.warn("Job does not exist.");
        }
    }

}