package org.etsi.osl.cridge;

import java.util.Date;
import java.util.concurrent.TimeUnit;
import org.apache.camel.LoggingLevel;
import org.apache.camel.builder.RouteBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import io.fabric8.kubernetes.api.model.Namespace;
import io.fabric8.kubernetes.api.model.NamespaceBuilder;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClientBuilder;
import lombok.Getter;

@Service
@Getter
public class CRRouteBuilder  extends RouteBuilder{

	@Value("${CRD_DEPLOY_CR_REQ}")
	private String CRD_DEPLOY_CR_REQ = "";


    @Value("${CRD_PATCH_CR_REQ}")
    private String CRD_PATCH_CR_REQ = "";
    
    @Value("${CRD_DELETE_CR_REQ}")
    private String CRD_DELETE_CR_REQ = "";
	

	@Autowired
	private KubernetesClientResource kubernetesClientResource;
	
	
	private static final Logger logger = LoggerFactory.getLogger( CRRouteBuilder.class.getSimpleName());

	@Override
	public void configure() throws Exception {
		
		from( CRD_DEPLOY_CR_REQ )
		.log(LoggingLevel.INFO, log, CRD_DEPLOY_CR_REQ + " message received!")
		.to("log:DEBUG?showBody=true&showHeaders=true")
		.bean( kubernetesClientResource, "deployCR(${headers}, ${body})")
		.convertBodyTo( String.class );
		
		from( CRD_DELETE_CR_REQ )
        .log(LoggingLevel.INFO, log, CRD_DELETE_CR_REQ + " message received!")
        .to("log:DEBUG?showBody=true&showHeaders=true")
        .bean( kubernetesClientResource, "deleteCR(${headers}, ${body})")
        .convertBodyTo( String.class );
        
        from( CRD_PATCH_CR_REQ )
        .log(LoggingLevel.INFO, log, CRD_PATCH_CR_REQ + " message received!")
        .to("log:DEBUG?showBody=true&showHeaders=true")
        .bean( kubernetesClientResource, "patchCR(${headers}, ${body})")
        .convertBodyTo( String.class );
        
        // every  120000 ms (2 minutes)
        from( "timer://processUpdateResources?period=120000" )
        .log(LoggingLevel.INFO, log, " process nameSpacesTobeDeleted!")
        .to("log:DEBUG?showBody=true&showHeaders=true")
        .bean( CRRouteBuilder.class , "processNameSpacesTobeDeleted()");
		
	}
	
	 public void processNameSpacesTobeDeleted() {
	   
	   kubernetesClientResource.getNameSpacesTobeDeleted().forEach( (nameSpaceName, datetime) -> {
	     
	     Date currentTime = new Date(); //get current time
	     long diffInMillies = Math.abs(currentTime .getTime() - datetime.getTime());
	     
	     long diff = TimeUnit.MINUTES.convert(diffInMillies, TimeUnit.MILLISECONDS);
	     if (diff<=2) { //after 2 minutes this namespace will be deleted
	       return;
	     }
	     
	     try (final KubernetesClient k8s = new KubernetesClientBuilder().build()) {
	       logger.info("Trying to delete namespace {}", nameSpaceName);
	       Namespace ns = new NamespaceBuilder()
	            .withNewMetadata()
	            .withName( nameSpaceName )
	            .endMetadata().build();
	        k8s.namespaces().resource(ns).delete();     
	        
	        kubernetesClientResource.getNameSpacesTobeDeleted().remove(nameSpaceName);
	     }catch (Exception e) {
	        e.printStackTrace();
	      }
	    });
	   
	 }

}
