package org.etsi.osl.cridge;

import static io.fabric8.kubernetes.client.dsl.base.CustomResourceDefinitionContext.v1CRDFromCustomResourceType;
import static org.junit.Assert.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertEquals;
import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.net.HttpURLConnection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.apache.camel.CamelContext;
import org.apache.camel.RoutesBuilder;
import org.apache.camel.builder.RouteBuilder;
import org.apache.camel.model.dataformat.JsonLibrary;
import org.etsi.osl.domain.model.kubernetes.KubernetesCRDV1;
import org.etsi.osl.domain.model.kubernetes.KubernetesCRV1;
import org.etsi.osl.domain.model.kubernetes.KubernetesSecret;
import org.etsi.osl.tmf.ri639.model.ResourceCreate;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.event.ApplicationStartedEvent;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.context.event.EventListener;
import org.springframework.test.context.event.RecordApplicationEvents;
import io.fabric8.kubernetes.api.model.Condition;
import io.fabric8.kubernetes.api.model.GenericKubernetesResource;
import io.fabric8.kubernetes.api.model.KubernetesResource;
import io.fabric8.kubernetes.api.model.KubernetesResourceList;
import io.fabric8.kubernetes.api.model.ListMeta;
import io.fabric8.kubernetes.api.model.ListMetaBuilder;
import io.fabric8.kubernetes.api.model.NamedContext;
import io.fabric8.kubernetes.api.model.NamedContextBuilder;
import io.fabric8.kubernetes.api.model.ObjectMetaBuilder;
import io.fabric8.kubernetes.api.model.Secret;
import io.fabric8.kubernetes.api.model.SecretBuilder;
import io.fabric8.kubernetes.api.model.WatchEvent;
import io.fabric8.kubernetes.api.model.WatchEventBuilder;
import io.fabric8.kubernetes.api.model.apiextensions.v1.CustomResourceDefinition;
import io.fabric8.kubernetes.api.model.apiextensions.v1.CustomResourceDefinitionList;
import io.fabric8.kubernetes.api.model.apiextensions.v1.CustomResourceDefinitionListBuilder;
import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClientBuilder;
import io.fabric8.kubernetes.client.Watch;
import io.fabric8.kubernetes.client.Watcher;
import io.fabric8.kubernetes.client.WatcherException;
import io.fabric8.kubernetes.client.dsl.MixedOperation;
import io.fabric8.kubernetes.client.dsl.Resource;
import io.fabric8.kubernetes.client.dsl.base.ResourceDefinitionContext;
import io.fabric8.kubernetes.client.server.mock.EnableKubernetesMockClient;
import io.fabric8.kubernetes.client.server.mock.KubernetesMockServer;
import io.fabric8.kubernetes.client.utils.Serialization;

@RecordApplicationEvents
//@RunWith(SpringRunner.class)
@SpringBootTest(properties = {

    "kubernetes.master = test_kube",
    "spring.activemq.brokerUrl = vm://embedded?broker.persistent=false,useShutdownHook=false",
    "CATALOG_GET_RESOURCESPEC_BY_ID = direct:get_rspec_byid",
    "CATALOG_GET_RESOURCESPEC_BY_NAME_CATEGORY = direct:get_rspec_byNameCategory",
    "CATALOG_UPDADD_RESOURCE = direct:updadd_resource",})


@EnableKubernetesMockClient
public class CridgeIntegrationTest {
  private static final Logger logger = LoggerFactory.getLogger("org.etsi.osl.cridge");

  private static KubernetesMockServer server;

  @Autowired
  KubernetesClientResource kubernetesClientResource;

  @Autowired
  CatalogClient catalogClient;


  /**
   * it is used to override beans and setup the environment before application context starts
   */
  @TestConfiguration
  static class routesPreparation {

    @Autowired
    private CamelContext camelContext;

    private SCMocked scmocked = new SCMocked();

    private static KubernetesClient kubernetesClient;

    @Autowired
    KubernetesClientResource kubernetesClientResource;

    RoutesBuilder builder = new RouteBuilder() {
      @Override
      public void configure() {
        from("direct:get_rspec_byid").bean(scmocked, "getRspecID");
        from("direct:get_rspec_byNameCategory").bean(scmocked,
            "get_rspec_byNameCategory(${header.aname}, ${header.acategory}, ${header.aversion})");
        from("direct:updadd_resource").unmarshal()
            .json(JsonLibrary.Jackson, ResourceCreate.class, true).bean(scmocked,
                "updadd_resource(${header.aname},${header.acategory}, ${header.aversion}, ${body})");

      };
    };


    @EventListener(ApplicationStartedEvent.class)
    public void preparedForTheTest() {
      logger.info("=============== Created routes for the test and initialise =============== ");

      NamedContext nctx = new NamedContextBuilder().withNewContext().withCluster("testCluster")
          .and().withName("tstContext").build();

      // KubernetesClientBuilderCustomizer zzz = new KubernetesClientBuilderCustomizer();
      Config config = Config.autoConfigure(null);
      config.setNamespace("testNamespace");
      KubernetesClientBuilder xxx = new KubernetesClientBuilder().withConfig(config);
      // zzz.accept(xxx);
      kubernetesClient = server.createClient();
      kubernetesClient.getConfiguration().setCurrentContext(nctx);


      kubernetesClientResource.setKubernetesClient(kubernetesClient);
      logger.info("Starting preparedForTheTest kubernetesClient.toString() {} ",
          kubernetesClient.toString());
      logger.info("Starting preparedForTheTest for cluster getContexts {} ",
          kubernetesClient.getConfiguration().getContexts().toString());


      ListMeta metada = new ListMetaBuilder().build();
      // Given
      CustomResourceDefinitionList list =
          new CustomResourceDefinitionListBuilder().withMetadata(metada).build();


      // the following expects are for the sharedIndexInformers in the beginning
      server.expect().get()
          .withPath("/apis/apiextensions.k8s.io/v1/customresourcedefinitions?resourceVersion=0")
          .andReturn(HttpURLConnection.HTTP_OK, list).once();

      server.expect().get().withPath(
          "/apis/apiextensions.k8s.io/v1/customresourcedefinitions?allowWatchBookmarks=true&timeoutSeconds=600&watch=true")
          .andReturn(HttpURLConnection.HTTP_OK, list).once();

      try {
        camelContext.addRoutes(builder);
      } catch (Exception e) {
        e.printStackTrace();
      }
    }
  }


  @Test
  public void testCRDRegister() throws Exception {

    logger.info("===============TEST testCRDRegister =============================");
    CustomResourceDefinition crdMyCalc = v1CRDFromCustomResourceType(MyCalculator.class).build();
    server.expect().post().withPath("/apis/apiextensions.k8s.io/v1/customresourcedefinitions")
        .andReturn(HttpURLConnection.HTTP_OK, crdMyCalc).once();
    // When
    CustomResourceDefinition createdCronTabCrd = routesPreparation.kubernetesClient.apiextensions()
        .v1().customResourceDefinitions().resource(crdMyCalc).create();

    // Then
    assertNotNull(createdCronTabCrd);
  }


  @Test
  public void testCRDEPLOY() throws Exception {
    logger.info("===============TEST testCRDEPLOY =============================");


    // CustomResourceDefinition cronTabCrd = routesPreparation.kubernetesClient.apiextensions().v1()
    // .customResourceDefinitions()
    // .load(new BufferedInputStream(new FileInputStream("src/test/resources/crontab-crd.yaml")))
    // .item();


    // prepare CR_SPEC request
    CustomResourceDefinition crdMyCalc = v1CRDFromCustomResourceType(MyCalculator.class).build();



    Map<String, Object> map = new HashMap<>();
    map.put("currentContextCluster", "testCluster");
    map.put("clusterMasterURL", server.getHostName());
    map.put("org.etsi.osl.serviceId", "sid-xxx-xxx-xxx");
    map.put("org.etsi.osl.resourceId", "rid-xxx-xxx-xxx");
    map.put("org.etsi.osl.prefixName", "crrid12345");
    map.put("org.etsi.osl.serviceOrderId", "orderid-xxx-xxx-xxx");
    map.put("org.etsi.osl.namespace", "orderid-xxx-xxx-xxx");
    map.put("org.etsi.osl.statusCheckFieldName", "_CR_CHECK_FIELD");
    map.put("org.etsi.osl.statusCheckValueStandby", "_CR_CHECKVAL_STANDBY");
    map.put("org.etsi.osl.statusCheckValueAlarm", "_CR_CHECKVAL_ALARM");
    map.put("org.etsi.osl.statusCheckValueAvailable", "_CR_CHECKVAL_AVAILABLE");
    map.put("org.etsi.osl.statusCheckValueReserved", "_CR_CHECKVAL_RESERVED");
    map.put("org.etsi.osl.statusCheckValueUnknown", "_CR_CHECKVAL_UNKNOWN");
    map.put("org.etsi.osl.statusCheckValueSuspended", "_CR_CHECKVAL_SUSPENDED");


    String _CR_SPEC = Serialization.asYaml(getMyCalculator("test-resource"));

    // First check for an invalid context cluster
    String response = kubernetesClientResource.deployCR(map, _CR_SPEC);
    assertEquals("SEE OTHER", response);


    // Now try for the correct context cluster
    map.put("currentContextCluster",
        kubernetesClientResource.getKubernetesContextDefinition().getCurrentContextCluster());
    map.put("clusterMasterURL",
        kubernetesClientResource.getKubernetesContextDefinition().getMasterURL());


    // server api expectations
    server.expect().get()
        .withPath("/api/v1/namespaces/orderid-xxx-xxx-xxx/secrets?resourceVersion=0")
        .andReturn(HttpURLConnection.HTTP_OK, crdMyCalc).once();

    server.expect().get().withPath(
        "/api/v1/namespaces/orderid-xxx-xxx-xxx/secrets?allowWatchBookmarks=true&timeoutSeconds=600&watch=true")
        .andReturn(HttpURLConnection.HTTP_OK, crdMyCalc).once();

    server.expect().get()
        .withPath("/apis/stable.example.com/v1/namespaces/orderid-xxx-xxx-xxx/mycalculators")
        .andReturn(HttpURLConnection.HTTP_OK, crdMyCalc).once();

    server.expect().get().withPath(
        "/apis/stable.example.com/v1/namespaces/orderid-xxx-xxx-xxx/mycalculators/amycalculator")
        .andReturn(HttpURLConnection.HTTP_OK, crdMyCalc).once();

    server.expect().post().withPath("/api/v1/namespaces/orderid-xxx-xxx-xxx/mycalculators")
        .andReturn(HttpURLConnection.HTTP_CREATED, crdMyCalc).once();

    server.expect().post()
        .withPath(
            "/apis/examples.osl.etsi.org/v1alpha1/namespaces/orderid-xxx-xxx-xxx/mycalculators")
        .andReturn(HttpURLConnection.HTTP_CREATED, crdMyCalc).once();

    // register the resource handler
    KubernetesClient client = routesPreparation.kubernetesClient;

    client.getKubernetesSerialization().registerKubernetesResource("examples.osl.etsi.org/v1alpha1",
        "MyCalculator", MyCalculator.class);

    // make the deployment test!
    response = kubernetesClientResource.deployCR(map, _CR_SPEC);

    assertEquals("OK", response);

  }

  @Test
  public void testCRWithWatch() throws Exception {
    // useful blog
    // https://itnext.io/mock-kubernetes-api-server-in-java-using-fabric8-kubernetes-mock-server-81a75cf6c47c

    // Given
    KubernetesClient client = routesPreparation.kubernetesClient;

    server.expect().withPath(
        "/apis/examples.osl.etsi.org/v1alpha1/namespaces/orderid-xxx-xxx-xxx/mycalculators?allowWatchBookmarks=true&watch=true")
        .andUpgradeToWebSocket().open().waitFor(10L)
        .andEmit(new WatchEvent(getMyCalculator("amycalculator"), "ADDED")).waitFor(20L)
        .andEmit(new WatchEventBuilder().withNewStatusObject()
            .withMessage("410 - the event requested is outdated")
            .withCode(HttpURLConnection.HTTP_GONE).endStatusObject().build())
        .done().always();

    MixedOperation<MyCalculator, KubernetesResourceList<MyCalculator>, Resource<MyCalculator>> userAclClient =
        client.resources(MyCalculator.class);


    // When
    CountDownLatch eventRecieved = new CountDownLatch(1);
    client.getKubernetesSerialization().registerKubernetesResource("examples.osl.etsi.org/v1alpha1",
        "MyCalculator", MyCalculator.class);

    Watch watch =
        userAclClient.inNamespace("orderid-xxx-xxx-xxx").watch(new Watcher<MyCalculator>() {
          @Override
          public void eventReceived(Action action, MyCalculator calc) {
            if (action.name().contains("ADDED")) {
              eventRecieved.countDown();
              logger.info(
                  "===============CountDownLatch eventRecieved.countDown ADDED {}=============================",
                  Serialization.asYaml(calc));
            }
          }

          @Override
          public void onClose(WatcherException e) {}
        });

    // Then
    eventRecieved.await(30, TimeUnit.SECONDS);
    Assertions.assertEquals(0, eventRecieved.getCount());
    watch.close();

  }


  private KubernetesResource getMyCalculator(String resourceName) {
    MyCalculatorSpec spec = new MyCalculatorSpec();
    spec.setAction("amycalculator");

    MyCalculator createdMyCalcL = new MyCalculator();
    createdMyCalcL.setMetadata(new ObjectMetaBuilder().withName(resourceName).build());
    createdMyCalcL.setSpec(spec);

    Condition condition = new Condition();
    condition.setMessage("Last reconciliation succeeded");
    condition.setReason("Successful");
    condition.setStatus("True");
    condition.setType("Successful");
    MyCalculatorStatus status = new MyCalculatorStatus();
    status.setCondition(new Condition[] {condition});
    createdMyCalcL.setStatus(status);

    return createdMyCalcL;
  }


  @Test
  public void testKubernetesCRD2OpensliceCRD() throws FileNotFoundException {

    logger.info("===============TEST testCRDRegister =============================");
    // CustomResourceDefinition crdMyCalc = v1CRDFromCustomResourceType(MyCalculator.class).build();
    CustomResourceDefinition cronTabCrd =
        routesPreparation.kubernetesClient.apiextensions().v1().customResourceDefinitions()
            .load(
                new BufferedInputStream(new FileInputStream("src/test/resources/crontab-crd.yaml")))
            .item();

    List<KubernetesCRDV1> value = kubernetesClientResource.KubernetesCRD2OpensliceCRD(cronTabCrd);

    assertEquals(1, value.size());
    assertEquals("CronTab", value.get(0).getKind());
    assertEquals("v1", value.get(0).getVersion());
    assertEquals("stable.example.com", value.get(0).getApiGroup());


  }

  @Test
  public void testKubernetesCRDOpensliceCR() throws FileNotFoundException {


    logger.info("===============TEST testCRRegister =============================");

    ResourceDefinitionContext resourceDefinitionContext = new ResourceDefinitionContext.Builder()
        .withGroup("stable.example.com").withVersion("v1").withPlural("crontabs").build();

    GenericKubernetesResource cronTabCr =
        routesPreparation.kubernetesClient.genericKubernetesResources(resourceDefinitionContext)
            .load(new FileInputStream("src/test/resources/crontab-cr.yaml")).item();


    KubernetesCRV1 value = kubernetesClientResource.KubernetesCR2OpensliceCR(cronTabCr);

    assertEquals("CronTab", value.getKind());


  }

  @Test
  public void testKubernetesSecret2OpensliceResource() throws FileNotFoundException {


    logger.info(
        "===============TEST KubernetesSecret2OpensliceResource =============================");


    Secret secret1 = new SecretBuilder().withNewMetadata().withName("secret1").endMetadata()
        .addToData("username", "guccifer").addToData("password", "shadowgovernment").build();


    server.expect().post().withPath("/api/v1/namespaces/default/secrets")
        .andReturn(HttpURLConnection.HTTP_OK, secret1).once();

    Secret secretCreated = routesPreparation.kubernetesClient.secrets().inNamespace("default")
        .resource(secret1).create();


    KubernetesSecret value =
        kubernetesClientResource.KubernetesSecret2OpensliceResource(secretCreated);

    assertEquals("Secret", value.getKind());
    assertEquals("v1", value.getVersion());


  }


//  @Test
//  public void testCRDelete() throws Exception {
//    logger.info("===============TEST testCRDelete =============================");
//
//
//    // CustomResourceDefinition cronTabCrd = routesPreparation.kubernetesClient.apiextensions().v1()
//    // .customResourceDefinitions()
//    // .load(new BufferedInputStream(new FileInputStream("src/test/resources/crontab-crd.yaml")))
//    // .item();
//
//
//    // prepare CR_SPEC request
//    CustomResourceDefinition crdMyCalc = v1CRDFromCustomResourceType(MyCalculator.class).build();
//
//
//
//    Map<String, Object> map = new HashMap<>();
//    map.put("currentContextCluster", "testCluster");
//    map.put("clusterMasterURL", server.getHostName());
//    map.put("org.etsi.osl.serviceId", "sid-xxx-xxx-xxx");
//    map.put("org.etsi.osl.resourceId", "rid-xxx-xxx-xxx");
//    map.put("org.etsi.osl.prefixName", "crrid12345");
//    map.put("org.etsi.osl.serviceOrderId", "orderid-xxx-xxx-xxx");
//    map.put("org.etsi.osl.namespace", "orderid-xxx-xxx-xxx");
//    map.put("org.etsi.osl.statusCheckFieldName", "_CR_CHECK_FIELD");
//    map.put("org.etsi.osl.statusCheckValueStandby", "_CR_CHECKVAL_STANDBY");
//    map.put("org.etsi.osl.statusCheckValueAlarm", "_CR_CHECKVAL_ALARM");
//    map.put("org.etsi.osl.statusCheckValueAvailable", "_CR_CHECKVAL_AVAILABLE");
//    map.put("org.etsi.osl.statusCheckValueReserved", "_CR_CHECKVAL_RESERVED");
//    map.put("org.etsi.osl.statusCheckValueUnknown", "_CR_CHECKVAL_UNKNOWN");
//    map.put("org.etsi.osl.statusCheckValueSuspended", "_CR_CHECKVAL_SUSPENDED");
//
//
//    String _CR_SPEC = Serialization.asYaml(getMyCalculator("test-resource"));
//
//    // First check for an invalid context cluster
//    String response = kubernetesClientResource.deployCR(map, _CR_SPEC);
//    assertEquals("SEE OTHER", response);
//
//
//    // Now try for the correct context cluster
//    map.put("currentContextCluster",
//        kubernetesClientResource.getKubernetesContextDefinition().getCurrentContextCluster());
//    map.put("clusterMasterURL",
//        kubernetesClientResource.getKubernetesContextDefinition().getMasterURL());
//
//
//    // server api expectations
//    server.expect().get()
//        .withPath("/api/v1/namespaces/orderid-xxx-xxx-xxx/secrets?resourceVersion=0")
//        .andReturn(HttpURLConnection.HTTP_OK, crdMyCalc).once();
//
//    server.expect().get().withPath(
//        "/api/v1/namespaces/orderid-xxx-xxx-xxx/secrets?allowWatchBookmarks=true&timeoutSeconds=600&watch=true")
//        .andReturn(HttpURLConnection.HTTP_OK, crdMyCalc).once();
//
//    server.expect().get()
//        .withPath("/apis/stable.example.com/v1/namespaces/orderid-xxx-xxx-xxx/mycalculators")
//        .andReturn(HttpURLConnection.HTTP_OK, crdMyCalc).once();
//
//    server.expect().get().withPath(
//        "/apis/stable.example.com/v1/namespaces/orderid-xxx-xxx-xxx/mycalculators/amycalculator")
//        .andReturn(HttpURLConnection.HTTP_OK, crdMyCalc).once();
//
//    server.expect().post().withPath("/api/v1/namespaces/orderid-xxx-xxx-xxx/mycalculators")
//        .andReturn(HttpURLConnection.HTTP_CREATED, crdMyCalc).once();
//
//    server.expect().post()
//        .withPath(
//            "/apis/examples.osl.etsi.org/v1alpha1/namespaces/orderid-xxx-xxx-xxx/mycalculators")
//        .andReturn(HttpURLConnection.HTTP_CREATED, crdMyCalc).once();
//
//    // register the resource handler
//    KubernetesClient client = routesPreparation.kubernetesClient;
//
//    client.getKubernetesSerialization().registerKubernetesResource("examples.osl.etsi.org/v1alpha1",
//        "MyCalculator", MyCalculator.class);
//
//    // make the deployment test!
//    response = kubernetesClientResource.deleteCR(map, _CR_SPEC);
//
//    assertEquals("OK", response);
//
//  }



//  @Test
//  public void testCPatch() throws Exception {
//    logger.info("===============TEST testCRDEPLOY =============================");
//
//
//    // CustomResourceDefinition cronTabCrd = routesPreparation.kubernetesClient.apiextensions().v1()
//    // .customResourceDefinitions()
//    // .load(new BufferedInputStream(new FileInputStream("src/test/resources/crontab-crd.yaml")))
//    // .item();
//
//
//    // prepare CR_SPEC request
//    CustomResourceDefinition crdMyCalc = v1CRDFromCustomResourceType(MyCalculator.class).build();
//
//
//
//    Map<String, Object> map = new HashMap<>();
//    map.put("currentContextCluster", "testCluster");
//    map.put("clusterMasterURL", server.getHostName());
//    map.put("org.etsi.osl.serviceId", "sid-xxx-xxx-xxx");
//    map.put("org.etsi.osl.resourceId", "rid-xxx-xxx-xxx");
//    map.put("org.etsi.osl.prefixName", "crrid12345");
//    map.put("org.etsi.osl.serviceOrderId", "orderid-xxx-xxx-xxx");
//    map.put("org.etsi.osl.namespace", "orderid-xxx-xxx-xxx");
//    map.put("org.etsi.osl.statusCheckFieldName", "_CR_CHECK_FIELD");
//    map.put("org.etsi.osl.statusCheckValueStandby", "_CR_CHECKVAL_STANDBY");
//    map.put("org.etsi.osl.statusCheckValueAlarm", "_CR_CHECKVAL_ALARM");
//    map.put("org.etsi.osl.statusCheckValueAvailable", "_CR_CHECKVAL_AVAILABLE");
//    map.put("org.etsi.osl.statusCheckValueReserved", "_CR_CHECKVAL_RESERVED");
//    map.put("org.etsi.osl.statusCheckValueUnknown", "_CR_CHECKVAL_UNKNOWN");
//    map.put("org.etsi.osl.statusCheckValueSuspended", "_CR_CHECKVAL_SUSPENDED");
//
//
//    String _CR_SPEC = Serialization.asYaml(getMyCalculator("test-resource"));
//
//    // First check for an invalid context cluster
//    String response = kubernetesClientResource.deployCR(map, _CR_SPEC);
//    assertEquals("SEE OTHER", response);
//
//
//    // Now try for the correct context cluster
//    map.put("currentContextCluster",
//        kubernetesClientResource.getKubernetesContextDefinition().getCurrentContextCluster());
//    map.put("clusterMasterURL",
//        kubernetesClientResource.getKubernetesContextDefinition().getMasterURL());
//
//
//    // server api expectations
//    server.expect().get()
//        .withPath("/api/v1/namespaces/orderid-xxx-xxx-xxx/secrets?resourceVersion=0")
//        .andReturn(HttpURLConnection.HTTP_OK, crdMyCalc).once();
//
//    server.expect().get().withPath(
//        "/api/v1/namespaces/orderid-xxx-xxx-xxx/secrets?allowWatchBookmarks=true&timeoutSeconds=600&watch=true")
//        .andReturn(HttpURLConnection.HTTP_OK, crdMyCalc).once();
//
//    server.expect().get()
//        .withPath("/apis/stable.example.com/v1/namespaces/orderid-xxx-xxx-xxx/mycalculators")
//        .andReturn(HttpURLConnection.HTTP_OK, crdMyCalc).once();
//
//    server.expect().get().withPath(
//        "/apis/stable.example.com/v1/namespaces/orderid-xxx-xxx-xxx/mycalculators/amycalculator")
//        .andReturn(HttpURLConnection.HTTP_OK, crdMyCalc).once();
//
//    server.expect().post().withPath("/api/v1/namespaces/orderid-xxx-xxx-xxx/mycalculators")
//        .andReturn(HttpURLConnection.HTTP_CREATED, crdMyCalc).once();
//
//    server.expect().post()
//        .withPath(
//            "/apis/examples.osl.etsi.org/v1alpha1/namespaces/orderid-xxx-xxx-xxx/mycalculators")
//        .andReturn(HttpURLConnection.HTTP_CREATED, crdMyCalc).once();
//
//    // register the resource handler
//    KubernetesClient client = routesPreparation.kubernetesClient;
//
//    client.getKubernetesSerialization().registerKubernetesResource("examples.osl.etsi.org/v1alpha1",
//        "MyCalculator", MyCalculator.class);
//    
//    server.expect().get()
//    .withPath(
//        "/apis/examples.osl.etsi.org/v1alpha1/namespaces/orderid-xxx-xxx-xxx/mycalculators/crrid12345")
//        .andReturn(HttpURLConnection.HTTP_OK, null).once();
//    server.expect().patch()
//    .withPath(
//        "/apis/examples.osl.etsi.org/v1alpha1/namespaces/orderid-xxx-xxx-xxx/mycalculators/crrid12345")
//        .andReturn(HttpURLConnection.HTTP_OK, null).once();
//
//
//    // make the deployment test!
//    response = kubernetesClientResource.patchCR(map, _CR_SPEC);
//
//    assertEquals("OK", response);
//
//  }


}
