/*-
 * ========================LICENSE_START=================================
 * org.etsi.osl.controllers.sylva
 * %%
 * Copyright (C) 2024 osl.etsi.org
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =========================LICENSE_END==================================
 */
package org.etsi.osl.controllers.sylva;

import io.fabric8.kubernetes.api.model.ConfigMap;
import io.fabric8.kubernetes.api.model.KubernetesResourceList;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.dsl.MixedOperation;
import io.fabric8.kubernetes.client.dsl.Resource;
import io.fabric8.kubernetes.client.informers.ResourceEventHandler;
import io.fabric8.kubernetes.client.informers.SharedIndexInformer;
import lombok.Data;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.yaml.snakeyaml.DumperOptions;
import org.yaml.snakeyaml.Yaml;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@Service
public class SylvaMDResourceOperator {

  private static final Logger log = LoggerFactory.getLogger(SylvaMDResourceOperator.class);

  private static final String WORKINGDIR_PATH = "/opt/sylva-core"; // Change working directory to
                                                                   // /opt/sylva-core

  private static final String SYLVA_BIN_PATH = "/opt/sylva-core/bin/"; // Change working directory to
                                                                   // /opt/sylva-core

  private static final String WORKLOAD_CLUSTERS_DIR =
      "/opt/sylva-core/environment-values/workload-clusters/";

  @Autowired
  private KubernetesClient kubernetesClient;

  @Data
  private static class ExecResult {
    public ExecResult(int i, String s) {
      this.exitCode = i;
      this.lastLine = s;
    }

    private String lastLine;
    private int exitCode;
  }


  @Autowired
  public SylvaMDResourceOperator(KubernetesClient kubernetesClient) {
    this.kubernetesClient = kubernetesClient;
    // Start watching for resource events
    watchResources();
  }

  // Watch for events on SylvaMDResource
  private void watchResources() {


    MixedOperation<SylvaMDResource, KubernetesResourceList<SylvaMDResource>, Resource<SylvaMDResource>> resourceClient =
        kubernetesClient.resources(SylvaMDResource.class);

    SharedIndexInformer<SylvaMDResource> sharedIndexInformer =
        resourceClient.inAnyNamespace().inform(new ResourceEventHandler<SylvaMDResource>() {

          @Override
          public void onAdd(SylvaMDResource resource) {
            log.info("===== onAdd action =====");
            reconcile(resource, null, kubernetesClient);

          }

          @Override
          public void onUpdate(SylvaMDResource oldResource, SylvaMDResource resource) {
            log.info("===== onUpdate action =====");
            reconcile(resource, oldResource, kubernetesClient);

          }

          @Override
          public void onDelete(SylvaMDResource resource, boolean deletedFinalStateUnknown) {
            //deletion is handled by reconcile due to finalizers
            // handleDelete(resource, kubernetesClient);
            log.info("===== onDelete action =====");
            log.info("Removed resource from system SylvaMDResource: {}",
                resource.getMetadata().getName());
          }


        });

    // try {
    // Thread.sleep(60 * 1000L);
    // } catch (InterruptedException e) {
    // // TODO Auto-generated catch block
    // e.printStackTrace();
    // }
    // log.info("SharedIndexInformer open for 6000 seconds");

    log.info("SharedIndexInformer running");
    sharedIndexInformer.run();


    // resourceClient.inAnyNamespace().watch(new Watcher<SylvaMDResource>() {
    // @Override
    // public void eventReceived(Action action, SylvaMDResource resource) {
    // switch (action) {
    // case ADDED:
    //
    // // here concider to go with a thread
    // // Thread reconcileThread = new Thread(this::reconcile);
    // // reconcileThread.start(); // Start the thread
    // reconcile(action, resource, kubernetesClient);
    //
    //
    // break;
    // case MODIFIED:
    //
    //
    // reconcile(action, resource, kubernetesClient);
    // break;
    // case DELETED:
    // handleDelete(resource, kubernetesClient);
    // break;
    // default:
    // log.warn("Unknown action: {}", action);
    // break;
    // }
    // }
    //
    // @Override
    // public void onClose(WatcherException cause) {
    // if (cause != null) {
    // log.error("Watcher closed due to: {}", cause.getMessage());
    // restartWatcher();
    // }
    // }
    // });


  }


  // Handle the deletion of the SylvaMDResource
  private void handleDelete(SylvaMDResource resource, KubernetesClient client) {
    log.info("Handling deletion of SylvaMDResource: {}", resource.getMetadata().getName());
    String deploymentFolder = WORKLOAD_CLUSTERS_DIR + resource.getMetadata().getName();
    SylvaMDResource resourceClone = performSylvaDeleteWCActions(resource, client);
    deleteDeploymentFolder(deploymentFolder);
  }


  /**
   * 
   * will perform some CLI actions to (hopefully) remove the cluster the actions are from the page
   * https://sylva-projects.gitlab.io/operations/procedures/lcm/cluster-operations/remove-workload-cluster
   * 
   * @param resource
   * @param client
   * @return
   */
  private SylvaMDResource performSylvaDeleteWCActions(SylvaMDResource resource,
      KubernetesClient client) {

    String cmd = "";
    int exitCode = 0;
    String wcName = resource.getMetadata().getName();
    log.info("The resource is in {} state. Will proceed to execute commands towards Sylva mgmt cluster.", resource.getStatus().getState());

    SylvaMDResource resourceClone = updateResourceStatus(resource,
        SylvaMDResourceState.DELETE_IN_PROGRESS, "start deletion", client);


    log.info("Suspend all kustomization and helmrealeses for workload cluster namespace");

//    cmd = String.format(
//        "source bin/env",
//        wcName);
//    exitCode += execCLICommand(cmd).getExitCode();
    
    
    cmd = String.format(
        SYLVA_BIN_PATH+"flux suspend -n %s --all kustomization --kubeconfig management-cluster-kubeconfig",
        wcName);
    exitCode += execCLICommand(cmd).getExitCode();


    cmd = String.format(
        SYLVA_BIN_PATH+"flux suspend -n %s --all helmrelease --kubeconfig management-cluster-kubeconfig", wcName);
    exitCode += execCLICommand(cmd).getExitCode();


    log.info("Remove resources (VMs) from cluster");
    cmd = String.format(
        SYLVA_BIN_PATH+"kubectl delete -n %s clusters.cluster.x-k8s.io %s --kubeconfig management-cluster-kubeconfig",
        wcName, wcName);
    exitCode += execCLICommand(cmd).getExitCode();

    // this is applicable only for rke2-capo deployments - if this is not done before ns deletion,
    // the ns will remain in terminating mode
    log.info("Remove the capo heatstack before removing the controller");
    cmd = String.format(
        SYLVA_BIN_PATH+"kubectl delete -n %s heatstacks heatstack-capo-cluster-resources --kubeconfig management-cluster-kubeconfig",
        wcName);
    exitCode += execCLICommand(cmd).getExitCode();

    log.info("Deletes a job in a different namespace kube-job");
    cmd = String.format(
        SYLVA_BIN_PATH+"kubectl delete -n kube-job job create-image-info-%s --kubeconfig management-cluster-kubeconfig",
        wcName);
    exitCode += execCLICommand(cmd).getExitCode();

    log.info("Delete namespace and remove all resources in the workload cluster namespace");

    cmd = String.format(SYLVA_BIN_PATH+"kubectl delete namespace %s --kubeconfig management-cluster-kubeconfig",
        wcName);
    exitCode += execCLICommand(cmd).getExitCode();

    if (exitCode == 0) {
      log.info("All delete commands executed successfully in {} for cluster {}", WORKINGDIR_PATH,
          wcName);
      resourceClone = updateResourceStatus(resourceClone, SylvaMDResourceState.DELETED,
          "executing apply-workload-cluster.sh success ", client);
    } else {
      log.info("Some delete commands failed execution in {} for cluster {}. Exit code: {}",
          WORKINGDIR_PATH, wcName, exitCode);

      resourceClone = updateResourceStatus(resourceClone, SylvaMDResourceState.DELETE_FAILED,
          "executing apply-workload-cluster.sh failed", client);

    }

    return resourceClone;

  }

  // Delete the deployment folder
  private void deleteDeploymentFolder(String path) {


    log.info("Will not delete deployment folder: {}", path);

    // File folder = new File(path);
    // if (folder.exists()) {
    // boolean deleted = folder.delete();
    // boolean deleted = false;
    // if (deleted) {
    // log.info("Deleted deployment folder: {}", path);
    // } else {
    // log.error("Failed to delete deployment folder: {}", path);
    // }
    // }



  }

  // // Restart the watcher if it closes unexpectedly
  // private void restartWatcher() {
  // log.info("Restarting the watcher...");
  // watchResources();
  // }

  public SylvaMDResource reconcile(SylvaMDResource resource, SylvaMDResource oldResource,
      KubernetesClient client) {

    log.info("reconciling");


    // Handle resource deletion
    if (resource.getMetadata().getDeletionTimestamp() != null) {
      if (resource.getStatus().getState().equals(SylvaMDResourceState.DELETE_IN_PROGRESS) 
          || resource.getStatus().getState().equals(SylvaMDResourceState.DELETE_FAILED) 
          || resource.getStatus().getState().equals(SylvaMDResourceState.DELETED) ) {
        log.info("The resource is in {} state. Reconcile deletion action is ignored.",
            resource.getStatus().getState());
        return resource;
      }
      // Perform cleanup logic
      handleDelete(resource, client);

      removeFinalizer(resource, client);
      return null;
    }
    
    

    if (oldResource == null) {
      if (resource.getStatus() != null) {
        log.info(
            "Old resource is NULL. Current resource is in {} state. Reconcile action is ignored.",
            resource.getStatus().getState());
        return resource;
      }
    } else {

      log.info("oldResource.spec = {}", oldResource.getSpec().toString());
      log.info("newresource.spec = {}", resource.getSpec().toString());

      if (resource.getStatus() == null) {
        log.info("The resource has NULL status.");
        return resource;
      }
      if (!resource.getStatus().getState().equals(SylvaMDResourceState.ACTIVE)
          && !resource.getStatus().getState().equals(SylvaMDResourceState.FAILED)) {
        log.info("The resource is in {} state. Reconcile action is ignored.",
            resource.getStatus().getState());
        return resource;
      }


      // Compare old and new resource state ( compare spec, status, etc.)
      if (resource.getSpec().toString().equals(oldResource.getSpec().toString())) {
        log.info("No significant changes detected, skipping reconciliation.");
        return resource; // Skip reconciliation if there are no significant changes
      }
    }
    

    //uncomment to short-circuit and put it immediatelly active
//    SylvaMDResource resourceClone = addFinalizers(resource, client);
//    return updateResourceStatus(resourceClone, SylvaMDResourceState.ACTIVE,
//        "executing apply-workload-cluster.sh success ", client);
    
    
    


    SylvaMDResource resourceClone =
        updateResourceStatus(resource, SylvaMDResourceState.IN_PROGRESS, "reconciling", client);
    resourceClone = addFinalizers(resourceClone, client);

    String resourceName = resource.getMetadata().getName();
    String namespace = resource.getMetadata().getNamespace();
    String deploymentFolder = WORKLOAD_CLUSTERS_DIR + resourceName;



    // Step 1: Create a new deployment folder
    createDeploymentFolder(deploymentFolder);

    // Step 2: Create empty kustomization.yaml, values.yaml, secrets.yaml files
    createEmptyFile(deploymentFolder + "/kustomization.yaml");
    createEmptyFile(deploymentFolder + "/values.yaml");
    createEmptyFile(deploymentFolder + "/secrets.yaml");

    // Step 3: Modify values.yaml using SylvaMDResourceSpec
    
    if ( resourceClone.getSpec().getValuesyaml() != null ) {
      populateFileFromText(namespace, resourceClone.getSpec().getValuesyaml(),
          deploymentFolder + "/values.yaml");      
    }else {
      populateAndModifyValuesYaml(namespace, "values.sylvamd.configmaps.osl.etsi.org",
          deploymentFolder + "/values.yaml", resourceClone.getSpec());      
    }
    

    // Step 4: Populate kustomization.yaml and secrets.yaml from ConfigMaps
    populateFileFromConfigMap(namespace, "kustomization.sylvamd.configmaps.osl.etsi.org",
        deploymentFolder + "/kustomization.yaml");
    populateFileFromConfigMap(namespace, "secrets.sylvamd.configmaps.osl.etsi.org",
        deploymentFolder + "/secrets.yaml");

    // Step 5: Execute the shell script
    resourceClone = executeShellScriptApplyWC(deploymentFolder, resourceClone, client);
    resourceClone = executeShellScriptGetWCKubeconfig(deploymentFolder, resourceClone, client);
    return resourceClone;
  }



  // Create the deployment folder
  private void createDeploymentFolder(String path) {
    File folder = new File(path);
    if (!folder.exists()) {
      boolean created = folder.mkdirs();
      if (created) {
        log.info("Created deployment folder: {}", path);
      } else {
        log.error("Failed to create deployment folder: {}", path);
      }
    }
  }

  // Create empty files (kustomization.yaml, values.yaml, secrets.yaml)
  private void createEmptyFile(String filePath) {
    try {
      File file = new File(filePath);
      if (!file.exists()) {
        boolean created = file.createNewFile();
        if (created) {
          log.info("Created file: {}", filePath);
        }
      }
    } catch (IOException e) {
      log.error("Error creating file: {}", filePath, e);
    }
  }

  // Populate values.yaml and replace the control_plane_replicas and md0.replicas
  private void populateAndModifyValuesYaml(String namespace, String configMapName, String filePath,
      SylvaMDResourceSpec spec) {
    try {
      // Retrieve the ConfigMap
      ConfigMap configMap =
          kubernetesClient.configMaps().inNamespace(namespace).withName(configMapName).get();

      if (configMap != null && configMap.getData() != null) {
        // Fetch the content of the file from the ConfigMap
        String content = configMap.getData().values().stream().findFirst().orElse("");

        // Modify the content using SylvaMDResourceSpec
        content = replaceYamlValues(content, spec);

        // Write the modified content to the specified file
        Files.write(Paths.get(filePath), content.getBytes());
        log.info("Populated and modified values.yaml from ConfigMap {}", configMapName);
      } else {
        log.error("Failed to find or read ConfigMap: {}", configMapName);
      }
    } catch (IOException e) {
      log.error("Error writing to file: {}", filePath, e);
    }
  }

  // Replace control_plane_replicas and machine_deployments.md0.replicas using SnakeYAML
  private String replaceYamlValues(String yamlContent, SylvaMDResourceSpec spec) {
    Yaml yaml = new Yaml();
    Map<String, Object> yamlData = yaml.load(yamlContent);

    // Update control_plane_replicas
    Map<String, Object> cluster = (Map<String, Object>) yamlData.get("cluster");
    cluster.put("control_plane_replicas", spec.getClusterControlPlaneReplicas());

    // Update machine_deployments.md0.replicas
    Map<String, Object> machineDeployments =
        (Map<String, Object>) cluster.get("machine_deployments");
    Map<String, Object> md0 = (Map<String, Object>) machineDeployments.get("md0");
    md0.put("replicas", spec.getClusterMd0Replicas());

    // Serialize the modified YAML back to a string
    Yaml yamlOutput = new Yaml(getYamlDumperOptions());
    return yamlOutput.dump(yamlData);
  }

  // Configure the YAML DumperOptions to maintain the original formatting as much as possible
  private DumperOptions getYamlDumperOptions() {
    DumperOptions options = new DumperOptions();
    options.setDefaultFlowStyle(DumperOptions.FlowStyle.BLOCK);
    options.setPrettyFlow(true);
    options.setIndent(2);
    return options;
  }

  private void populateFileFromConfigMap(String namespace, String configMapName, String filePath) {
    try {
      ConfigMap configMap =
          kubernetesClient.configMaps().inNamespace(namespace).withName(configMapName).get();

      if (configMap != null && configMap.getData() != null) {
        String content = configMap.getData().values().stream().findFirst().orElse("");
        Files.write(Paths.get(filePath), content.getBytes());
        log.info("Populated file {} from ConfigMap {}", filePath, configMapName);
      } else {
        log.error("Failed to find or read ConfigMap: {}", configMapName);
      }
    } catch (IOException e) {
      log.error("Error writing to file: {}", filePath, e);
    }
  }
  

  private void populateFileFromText(String namespace, String text, String filePath) {
    try {

      if (text != null ) {
        String content = text;
        Files.write(Paths.get(filePath), content.getBytes());
        log.info("Populated file {} from text {}", filePath, text);
      } else {
        log.error("Failed to write text: {}", text);
      }
    } catch (IOException e) {
      log.error("Error writing to file: {}", filePath, e);
    }
  }

  // Execute the shell script to apply the workload with the working directory set to
  // /opt/sylva-core
  private SylvaMDResource executeShellScriptApplyWC(String deploymentFolderPath,
      SylvaMDResource resource, KubernetesClient client) {

    File workingDir = new File(WORKINGDIR_PATH);
    File scriptFile = new File(workingDir, "apply-workload-cluster.sh");


    SylvaMDResource resourceClone = updateResourceStatus(resource, SylvaMDResourceState.IN_PROGRESS,
        "executing script", client);

    // Check if the script file exists before executing it
    if (!scriptFile.exists() || !scriptFile.isFile()) {
      log.error("Shell script {} does not exist in working directory {}", scriptFile.getName(),
          WORKINGDIR_PATH);
      return resourceClone;
    }

    String command = "./apply-workload-cluster.sh " + deploymentFolderPath;
    int exitCode = execCLICommand(command).getExitCode();


    if (exitCode == 0) {
      log.info("Shell script executed successfully in {} for {}", WORKINGDIR_PATH,
          deploymentFolderPath);
      resourceClone = updateResourceStatus(resource, SylvaMDResourceState.ACTIVE,
          "executing apply-workload-cluster.sh success ", client);
    } else {
      log.error("Shell script execution failed in {} for {}. Exit code: {}", WORKINGDIR_PATH,
          deploymentFolderPath, exitCode);

      resourceClone = updateResourceStatus(resource, SylvaMDResourceState.FAILED,
          "executing apply-workload-cluster.sh failed", client);

    }

    return resource;
  }


  private SylvaMDResource executeShellScriptGetWCKubeconfig(String deploymentFolderPath,
      SylvaMDResource resource, KubernetesClient client) {

    String wcName = resource.getMetadata().getName();

    String cmd = String.format(
        SYLVA_BIN_PATH+"kubectl get secret -n %s %s-kubeconfig --kubeconfig management-cluster-kubeconfig -o template={{.data.value}}",
        wcName, wcName);



    ExecResult res = execCLICommand(cmd);
    String secret = res.getLastLine();
    int exitCode = res.getExitCode();
    log.info("Workload cluster name:{} , Secret value(BASE64):{}", wcName, secret);

    if (exitCode == 0) {
      log.info("Shell script executed successfully in {}:  {}", WORKINGDIR_PATH, cmd);

      try {
        resource.getStatus().setSecret(secret);
        resource = client.resource(resource).patch();

        return resource;
      } catch (Exception e) {
        return null;
      }


    } else {
      log.error("Shell script execution failed in {} for {}. Exit code: {}, Command: {}",
          WORKINGDIR_PATH, deploymentFolderPath, exitCode, cmd);

    }

    return resource;
  }



  private ExecResult execCLICommand(String command) {


    File workingDir = new File(WORKINGDIR_PATH);

    ExecResult execResult = new ExecResult(-1, null);

    ProcessBuilder processBuilder = new ProcessBuilder(command.split(" "));

    // Set the working directory to WORKINGDIR_PATH /opt/sylva-core
    processBuilder.directory(workingDir);

    try {

      log.info("Will process command: {}", command);

      Process process = processBuilder.start(); // Start the process

      // Create a separate thread to capture the standard error
      Thread errorThread = new Thread(() -> {
        try {
          BufferedReader errorReader =
              new BufferedReader(new InputStreamReader(process.getErrorStream()));
          StringBuilder errors = new StringBuilder();
          String errorLine;
          while ((errorLine = errorReader.readLine()) != null) {
            errors.append(errorLine).append("\n");
            log.info("==e> {}", errorLine);
            execResult.setLastLine(errorLine);
          }
        } catch (IOException e) {
          e.printStackTrace();
        }
      });

      // Start the error capturing thread
      errorThread.start();



      // Capture the standard output
      BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
      String line;
      while ((line = reader.readLine()) != null) {
        log.info("===> {}", line);
        execResult.setLastLine(line);
      }



      int exitCode = process.waitFor(); // Wait for the process to finish
      // Wait for the error thread to finish
      errorThread.join();



      if (exitCode == 0) {
        log.info("execCLICommand executed successfully in {}: {}", WORKINGDIR_PATH, command);
      } else {
        log.info("execCLICommand execution failed in {} (exitCode= {}): {}", WORKINGDIR_PATH,
            exitCode, command);
      }

      execResult.setExitCode(exitCode);
      return execResult;
    } catch (IOException | InterruptedException e) {
      log.error("Error executing command {} in working dir {}", command, WORKINGDIR_PATH, e);
      log.error("StackTrace {}", e.getLocalizedMessage());
      e.printStackTrace();
    }

    return execResult;
  }

  /**
   * @param resource
   * @param i
   * @param string
   * @param client
   * @return
   */
  private SylvaMDResource updateResourceStatus(SylvaMDResource resource, SylvaMDResourceState state,
      String infoText, KubernetesClient client) {

    try {
      SylvaMDResourceStatus resStatus = resource.getStatus();
      if (resStatus == null) {
        resStatus = new SylvaMDResourceStatus();
      }
      resStatus.setState(state);
      resStatus.setStateText(state.getValue());
      resStatus.setInfoText(infoText);
      resource.setStatus(resStatus);

      resource = client.resource(resource).patch();

      return resource;
    } catch (Exception e) {
      return null;
    }


  }

  private SylvaMDResource addFinalizers(SylvaMDResource resource, KubernetesClient client) {

    List<String> finalizers = resource.getMetadata().getFinalizers();
    if (finalizers == null) {
      finalizers = new ArrayList<>();
    }
    finalizers.clear();
    finalizers.add("sylva.osl.etsi.org");
    resource.getMetadata().setFinalizers(finalizers);

    // resource.getMetadata().getFinalizers().clear();
    // resource.getMetadata().getFinalizers().add("sylva.osl.etsi.org");
    resource = client.resource(resource).patch();
    return resource;
  }

  // Remove the finalizer after cleanup
  private SylvaMDResource removeFinalizer(SylvaMDResource resource, KubernetesClient client) {
    List<String> finalizers = resource.getMetadata().getFinalizers();
    if (finalizers != null) {
      finalizers.clear();
      resource.getMetadata().setFinalizers(finalizers);

      resource = client.resource(resource).patch();
      log.info("Finalizer removed from resource: " + resource.getMetadata().getName());
      return resource;
    }
    return resource;
  }

}
