/*-
 * ========================LICENSE_START=================================
 * org.etsi.osl.controllers.kcj
 * %%
 * Copyright (C) 2024 osl.etsi.org
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =========================LICENSE_END==================================
 */
package org.etsi.osl.controllers.kcj;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.yaml.snakeyaml.DumperOptions;
import org.yaml.snakeyaml.Yaml;
import io.fabric8.kubernetes.api.model.HasMetadata;
import io.fabric8.kubernetes.api.model.KubernetesResourceList;
import io.fabric8.kubernetes.api.model.Namespace;
import io.fabric8.kubernetes.api.model.NamespaceBuilder;
import io.fabric8.kubernetes.api.model.Secret;
import io.fabric8.kubernetes.api.model.SecretBuilder;
import io.fabric8.kubernetes.api.model.StatusDetails;
import io.fabric8.kubernetes.client.Config;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClientBuilder;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.dsl.MixedOperation;
import io.fabric8.kubernetes.client.dsl.Resource;
import io.fabric8.kubernetes.client.informers.ResourceEventHandler;
import io.fabric8.kubernetes.client.informers.SharedIndexInformer;

@Service
public class KCJResourceOperator {

  private static final Logger log = LoggerFactory.getLogger(KCJResourceOperator.class);
  private static final String REMOTARGO_URL = "https://raw.githubusercontent.com/argoproj/argo-cd/stable/manifests/install.yaml";
  private static final String ARGOCD_NAMESPACE = "argocd";
  @Autowired
  private KubernetesClient localKubernetesClient;




  @Autowired
  public KCJResourceOperator(KubernetesClient kubernetesClient) {
    this.localKubernetesClient = kubernetesClient;
    // Start watching for resource events
    watchResources();
  }

  // Watch for events on KCJResource
  private void watchResources() {
    
    


    MixedOperation<KCJResource, KubernetesResourceList<KCJResource>, Resource<KCJResource>> resourceClient =
        localKubernetesClient.resources(KCJResource.class);

    SharedIndexInformer<KCJResource> sharedIndexInformer =
        resourceClient.inAnyNamespace().inform(new ResourceEventHandler<KCJResource>() {
          
          public void onAdd(KCJResource resource) {
            log.info("===== onAdd action =====");
            reconcile(resource, null, localKubernetesClient);

          }

          public void onUpdate(KCJResource oldResource, KCJResource resource) {
            log.info("===== onUpdate action =====");
            reconcile(resource, oldResource, localKubernetesClient);

          }

          public void onDelete(KCJResource resource, boolean deletedFinalStateUnknown) {
            //deletion is handled by reconcile due to finalizers
            // handleDelete(resource, kubernetesClient);
            log.info("===== onDelete action =====");
            log.info("Removed resource from system KCJResource: {}",
                resource.getMetadata().getName());
          }

         


        });


    log.info("SharedIndexInformer running");
    sharedIndexInformer.run();




  }


  // Handle the deletion of the KCJResource
  private void handleDelete(KCJResource resource, KubernetesClient localKubClient, KubernetesClient remoteKubClient) {
    log.info("Handling deletion of KCJResource: {} in {} state", resource.getMetadata().getName(), resource.getStatus().getState());
    if ( resource.getSpec().isInstallCridgeLocal() ) { //install cridge on local cluster via local argoCD
      installCRIDGELocallyviaLocalArgoCD( resource, localKubClient, remoteKubClient, "NONE", true);      
    }
    
    if ( resource.getSpec().isInstallCridgeRemote() ) { //install cridge on remote cluster via remote cridge      
       if ( resource.getSpec().isAddClusterToLocalArgoCD() ) {
        installCRIDGERemotellyviaLocalArgoCD( resource, localKubClient, remoteKubClient, "NONE", true);     
      } else  if ( resource.getSpec().isInstallArgoCDRemote()  ) {
          installCRIDGERemotellyviaRemoteArgoCD( resource, localKubClient, remoteKubClient, "NONE", true);
      }
      
    }
    
  }



  public KCJResource reconcile(KCJResource resource, KCJResource oldResource,
      KubernetesClient localKubClient) {

    log.info("reconciling");

    
    String resourceName = resource.getMetadata().getName();
    String namespace = resource.getMetadata().getNamespace();
    String targetKubeConfig = "";
    KCJResource resourceClone;
    
    if ( resource.getSpec().getClusterConfigBase64() != null ) {
      String secretBase64 = resource.getSpec().getClusterConfigBase64();
      // Decode the Base64 string to a regular string
      byte[] decodedBytes = Base64.getDecoder().decode( secretBase64 );
      targetKubeConfig = new String(decodedBytes);
    }else {
      targetKubeConfig = resource.getSpec().getClusterConfig();
    }
    
    log.info("targetKubeConfig = " + targetKubeConfig);
    
    KubernetesClient remoteKubClient;
    // Load kubeconfig from string
    try {
      remoteKubClient = new KubernetesClientBuilder().withConfig(Config.fromKubeconfig(targetKubeConfig)).build();     
    } catch (KubernetesClientException e) {
        e.printStackTrace();
        
        log.error("Failed to create Kubernetes client: " + e.getMessage());
        resourceClone =
            updateResourceStatus(resource, KCJResourceState.FAILED, "FAILED", "FAILED", "FAILED", localKubClient);
        return resourceClone;
    }
    
    
    
    
    

    // Handle resource deletion
    if (resource.getMetadata().getDeletionTimestamp() != null) {
      if (resource.getStatus().getState().equals(KCJResourceState.DELETE_FAILED) 
          || resource.getStatus().getState().equals(KCJResourceState.DELETED) ) {
        log.info("The resource is in {} state. Reconcile deletion action is ignored.",
            resource.getStatus().getState());
        return resource;
      }
      // Perform cleanup logic
      handleDelete(resource, localKubClient, remoteKubClient);

      removeFinalizer(resource, localKubClient);
      return null;
    }
    
    

    if (oldResource == null) {
      if (resource.getStatus() != null) {
        log.info(
            "Old resource is NULL. Current resource is in {} state. Reconcile action is ignored.",
            resource.getStatus().getState());
        return resource;
      }
    } else {

      log.info("oldResource.spec = {}", oldResource.getSpec().toString());
      log.info("newresource.spec = {}", resource.getSpec().toString());

      if (resource.getStatus() == null) {
        log.info("The resource has NULL status.");
        return resource;
      }
      if (!resource.getStatus().getState().equals(KCJResourceState.ACTIVE)
          && !resource.getStatus().getState().equals(KCJResourceState.FAILED)) {
        log.info("The resource is in {} state. Reconcile action is ignored.",
            resource.getStatus().getState());
        return resource;
      }


      // Compare old and new resource state ( compare spec, status, etc.)
      if (resource.getSpec().toString().equals(oldResource.getSpec().toString())) {
        log.info("No significant changes detected, skipping reconciliation.");
        return resource; // Skip reconciliation if there are no significant changes
      }
    }
    


    resourceClone =
        updateResourceStatus(resource, KCJResourceState.IN_PROGRESS, "reconciling", "reconciling", "reconciling", localKubClient);
    resourceClone = addFinalizers(resourceClone, localKubClient);


    
    
    if ( resource.getSpec().isAddClusterToLocalArgoCD() ) {
      resourceClone = execStrategyEnablelocalArgo(resourceClone, localKubClient, remoteKubClient);
    }
    
    if ( resource.getSpec().isInstallArgoCDRemote()  ) {
      resourceClone = execStrategyInstallRemoteArgo(resourceClone, localKubClient, remoteKubClient);
    }
    
    
    if ( resource.getSpec().isInstallCridgeLocal() ) { //install cridge on local cluster via local argoCD
      resourceClone = installCRIDGELocallyviaLocalArgoCD(resourceClone, localKubClient, remoteKubClient, targetKubeConfig, false);
    } else if ( resource.getSpec().isInstallCridgeRemote() ) { //install cridge on remote cluster via remote cridge
      
      if ( resource.getSpec().isInstallArgoCDRemote()  ) {
        resourceClone = installCRIDGERemotellyviaRemoteArgoCD(resourceClone, localKubClient, remoteKubClient, targetKubeConfig, false);
      }else if ( resource.getSpec().isAddClusterToLocalArgoCD() ) {
        resourceClone = installCRIDGERemotellyviaLocalArgoCD(resourceClone, localKubClient, remoteKubClient, targetKubeConfig, false);  
        
      }      
      
    }
    

    return resourceClone;
  }






  private KCJResource installCRIDGERemotellyviaRemoteArgoCD(KCJResource resource, KubernetesClient localKubClient,
      KubernetesClient remoteKubClient, String targetKubeConfig, Boolean isDelete) {
    
    log.info("In installCRIDGERemotellyviaLocalArgoCD");
    Config remoteConfig = remoteKubClient.getConfiguration();
    String appName = "cridge-"+remoteConfig.getCurrentContext().getContext().getCluster();
    String activemq_brokerUrl = resource.getSpec().getCridgeAMQEndPoint() ;
    String activemq_user = resource.getSpec().getCridgeAMQUsername();
    String activemq_password = resource.getSpec().getCridgeAMQPassword();
    
    
    String kubeconfig = targetKubeConfig ;
    String destination_server = "https://kubernetes.default.svc";
    String destination_server_name = "";
    String destination_namespace = "oslcridge";
    
    
    String yamlToapply = prepareCRIDGEyaml(appName, activemq_brokerUrl,  activemq_user, activemq_password,  
        destination_server, destination_server_name, destination_namespace, kubeconfig);
    
     // Convert the YAML string to an InputStream
     InputStream yamlInputStream = new java.io.ByteArrayInputStream(yamlToapply.getBytes());

     // Parse the YAML file into Kubernetes resources
     try {
       if (isDelete) {

         List<StatusDetails> response = remoteKubClient.load(yamlInputStream).delete();
         log.info("response delete=" + response.toString());
         return resource;
       }else {
         // Define the namespace with labels
         Namespace namespace = new NamespaceBuilder()
                 .withNewMetadata()
                     .withName(destination_namespace) // Set the namespace name
                     .addToLabels("pod-security.kubernetes.io/enforce", "baseline")  // Add labels
                     .addToLabels("pod-security.kubernetes.io/audit", "baseline")  // Add labels
                     .addToLabels("pod-security.kubernetes.io/warn", "baseline")  // Add labels
                 .endMetadata()
                 .build();
         
         // Create the namespace
         remoteKubClient.namespaces().createOrReplace(namespace);
         
         
         List<HasMetadata> response = remoteKubClient.load(yamlInputStream).createOrReplace(); 
         log.info("response=" + response.toString());
         return updateResourceStatus(resource, KCJResourceState.ACTIVE, null, "CRIDGE successfully configured remotelly via Remote ArgoCD",  null, localKubClient);
       }
       
       
     }catch (Exception e) {
       log.info("Error in installCRIDGELocallyviaLocalArgoCD: " + e.getMessage());
       return updateResourceStatus(resource, KCJResourceState.ACTIVE, null, "Failed to configure CRIDGE remotelly via Remote ArgoCD", null, localKubClient);   
    }
    
    
  }

  private KCJResource installCRIDGELocallyviaLocalArgoCD(KCJResource resource, KubernetesClient localKubClient,
      KubernetesClient remoteKubClient, String targetKubeConfig, Boolean isDelete) {

    log.info("In installCRIDGELocallyviaLocalArgoCD");
    
    Config remoteConfig = remoteKubClient.getConfiguration();
    String appName = "cridge-local-"+remoteConfig.getCurrentContext().getContext().getCluster();
    String activemq_brokerUrl = resource.getSpec().getCridgeAMQEndPoint() ;
    String activemq_user = resource.getSpec().getCridgeAMQUsername();
    String activemq_password = resource.getSpec().getCridgeAMQPassword();
    
    
    String kubeconfig = targetKubeConfig ;
    String destination_server = "https://kubernetes.default.svc";
    String destination_server_name = "";
    String destination_namespace = appName;
    
    String yamlToapply = prepareCRIDGEyaml(appName, activemq_brokerUrl,  activemq_user, activemq_password,  
        destination_server, destination_server_name, destination_namespace, kubeconfig);
    
     // Convert the YAML string to an InputStream
     InputStream yamlInputStream = new java.io.ByteArrayInputStream(yamlToapply.getBytes());

     // Parse the YAML file into Kubernetes resources
     try {
       if (isDelete) {

         List<StatusDetails> response = localKubClient.load(yamlInputStream).delete();
         log.info("response delete=" + response.toString());
         return resource;
       }else {
         Namespace namespace = new NamespaceBuilder()
             .withNewMetadata()
                 .withName(destination_namespace) // Set the namespace name
                 .addToLabels("pod-security.kubernetes.io/enforce", "baseline")  // Add labels
                 .addToLabels("pod-security.kubernetes.io/audit", "baseline")  // Add labels
                 .addToLabels("pod-security.kubernetes.io/warn", "baseline")  // Add labels
             .endMetadata()
             .build();
     
     // Create the namespace
     remoteKubClient.namespaces().createOrReplace(namespace);
     
         List<HasMetadata> response = localKubClient.load(yamlInputStream).createOrReplace(); 
         log.info("response=" + response.toString());
         return updateResourceStatus(resource, KCJResourceState.ACTIVE, null, "CRIDGE successfully configured locally via Local ArgoCD",  null, localKubClient);
       }
       
       
     }catch (Exception e) {
       log.info("Error in installCRIDGELocallyviaLocalArgoCD: " + e.getMessage());
       return updateResourceStatus(resource, KCJResourceState.ACTIVE, null, "Failed to configure CRIDGE Locally via Local ArgoCD",  null, localKubClient);
    }
    
  }
  
  
  private KCJResource installCRIDGERemotellyviaLocalArgoCD(KCJResource resource,
      KubernetesClient localKubClient, KubernetesClient remoteKubClient, String targetKubeConfig, Boolean isDelete) {

    log.info("In installCRIDGERemotellyviaLocalArgoCD");
    
    Config remoteConfig = remoteKubClient.getConfiguration();
    String appName = "cridge-remote-"+remoteConfig.getCurrentContext().getContext().getCluster();
    String activemq_brokerUrl = resource.getSpec().getCridgeAMQEndPoint() ;
    String activemq_user = resource.getSpec().getCridgeAMQUsername();
    String activemq_password = resource.getSpec().getCridgeAMQPassword();
    
    
    String kubeconfig = targetKubeConfig ;
    //String destination_server = remoteConfig.getMasterUrl();
    String destination_server = "";
    String destination_server_name =  remoteConfig.getCurrentContext().getContext().getCluster();

    log.info("destination_server_name = " + destination_server_name);
    String destination_namespace = appName;
    
    String yamlToapply = prepareCRIDGEyaml(appName, activemq_brokerUrl,  activemq_user, activemq_password,  
        destination_server, destination_server_name, destination_namespace, kubeconfig);
    
     // Convert the YAML string to an InputStream
     InputStream yamlInputStream = new java.io.ByteArrayInputStream(yamlToapply.getBytes());

     // Parse the YAML file into Kubernetes resources
     try {
       if (isDelete) {

         List<StatusDetails> response = localKubClient.load(yamlInputStream).delete();
         log.info("response delete=" + response.toString());
         return resource;
       }else {

         Namespace namespace = new NamespaceBuilder()
             .withNewMetadata()
                 .withName(destination_namespace) // Set the namespace name
                 .addToLabels("pod-security.kubernetes.io/enforce", "baseline")  // Add labels
                 .addToLabels("pod-security.kubernetes.io/audit", "baseline")  // Add labels
                 .addToLabels("pod-security.kubernetes.io/warn", "baseline")  // Add labels
             .endMetadata()
             .build();
     
     // Create the namespace
     remoteKubClient.namespaces().createOrReplace(namespace);
         List<HasMetadata> response = localKubClient.load(yamlInputStream).createOrReplace(); 
         log.info("response=" + response.toString());
         return updateResourceStatus(resource, KCJResourceState.ACTIVE, null, "CRIDGE successfully configured remotelly via Local ArgoCD",  null, localKubClient);
       }
       
       
     }catch (Exception e) {
       log.info("Error in installCRIDGELocallyviaLocalArgoCD: " + e.getMessage());
       return updateResourceStatus(resource, KCJResourceState.ACTIVE, null, "Failed to configure CRIDGE remotelly via Local ArgoCD", null, localKubClient);
       
    }
  }
  
  private String prepareCRIDGEyaml(String appName, String activemq_brokerUrl,  String activemq_user, String activemq_password,  
      String destination_server, String destination_server_name, String destination_namespace, String kubeconfig) {
    String yamlToapply =  
        """
    apiVersion: argoproj.io/v1alpha1
    kind: Application
    metadata:
      finalizers:
      - resources-finalizer.argocd.argoproj.io
      name: %s
      namespace: argocd
    spec:
      project: default
      source:
        repoURL: https://labs.etsi.org/rep/osl/code/org.etsi.osl.cridge.git
        path: helm/cridge
        targetRevision: helm
        helm:
          parameters:
            - name: oscreds.activemq.brokerUrl
              value: %s
            - name: oscreds.activemq.user
              value: %s
            - name: oscreds.activemq.password
              value: %s
          values: |
            kubeconfig: |
              XXX
      destination:
        server: %s
        name: %s
        namespace: %s
      syncPolicy:
        automated:
          prune: true
          selfHeal: true
        syncOptions:
          - CreateNamespace=true
        """;
        

         
         
         yamlToapply = String.format(yamlToapply, appName, activemq_brokerUrl,  activemq_user, activemq_password,  
             destination_server, destination_server_name, destination_namespace);
         

         // Step 2: Parse the YAML string into a Map
         Yaml yaml = new Yaml();
         Map<String, Object> yamlMap = yaml.load(yamlToapply);

         // Step 3: Modify the node (e.g., change the image field)
         Map<String, Object> firstContainer = (Map<String, Object>) ((Map<String, Object>) yamlMap.get("spec")).get("source") ;
         firstContainer = (Map<String, Object>) ((Map<String, Object>) firstContainer.get("helm"));
         String  val = (String) ( firstContainer.get("values"));

         kubeconfig = kubeconfig.replace("\n", "\n  ");
         firstContainer.put("values", "kubeconfig: |\n  " + kubeconfig);  // Modify the image field
         // Step 4: Convert the modified Map back to a YAML string
         DumperOptions options = new DumperOptions();
         options.setDefaultFlowStyle(DumperOptions.FlowStyle.BLOCK);  // Set the style to block for readability
         Yaml yamlDumper = new Yaml(options);
         yamlToapply = yamlDumper.dump(yamlMap);
         

         log.info("yamlToapply=" + yamlToapply);
         return yamlToapply;
  }

  private KCJResource execStrategyEnablelocalArgo(KCJResource resource, 
      KubernetesClient localKubClient, 
      KubernetesClient remoteKubClient) {
    
    //STEP 1: exec to remote cluster the RBC service accounts creation
    
    try (InputStream inputStream = KCJResourceOperator.class.getResourceAsStream("/remoteArgoCDRBAC.yaml")) {
      if (inputStream == null) {
          log.error("File not found!");
          return resource;
      }
      // Parse the YAML file into Kubernetes resources
      remoteKubClient.load(inputStream).serverSideApply()  ;

    } catch (Exception e) {
        e.printStackTrace();
    }
    
    // STEP 2 Create the Secret
    Secret secret = new SecretBuilder()
        .withNewMetadata()
            .withName("argocd-manager-token")
            .withNamespace("kube-system")
            .addToAnnotations("kubernetes.io/service-account.name", "argocd-manager")
        .endMetadata()
        .withType("kubernetes.io/service-account-token")
        .build();


    // Apply the Secret to the Kubernetes cluster
    remoteKubClient.secrets().inNamespace("kube-system").createOrReplace(secret);

    
    Secret aSecret = remoteKubClient.secrets().inNamespace("kube-system").withName("argocd-manager-token").get();

    log.info("Secret created or replaced successfully!");
    log.info("Secret:\n{}", aSecret);
    
    
    // Extract the Base64 encoded token from the Secret's data
    String base64EncodedToken = aSecret.getData().get("token");
    String decodedToken ="";
    if (base64EncodedToken != null) {
      // Decode the Base64 token
      byte[] decodedBytes = Base64.getDecoder().decode(base64EncodedToken);
      decodedToken = new String(decodedBytes);

      // Print the decoded token
      log.info("Decoded token: " + decodedToken);
    } else {
        log.error("Token not found in the Secret data.");
    }

    // Extract the Base64 encoded token from the Secret's data
    String ca = aSecret.getData().get("ca.crt");
    log.info("ca: " + ca);
    
    
    //STEP 3 create th secret in the local cluster
    
    
    Config remoteConfig = remoteKubClient.getConfiguration();

    // Retrieve the server URL
    String serverUrl = remoteConfig.getMasterUrl();
    
    Secret localSecret = new SecretBuilder()
        .withNewMetadata()
            .withName( remoteConfig.getCurrentContext().getContext().getCluster() +  "-secret")
            .addToLabels("argocd.argoproj.io/secret-type", "cluster")
        .endMetadata()
        .withType("Opaque")
        .addToStringData("name", remoteConfig.getCurrentContext().getContext().getCluster() )
        .addToStringData("server",  serverUrl   )
        .addToStringData("config", "{\n" +
                "  \"bearerToken\": \""+decodedToken+"\",\n" +
                "  \"tlsClientConfig\": {\n" +
                "    \"insecure\": false,\n" +
                "    \"caData\": \""+ ca +"\"\n" +
                "  }\n" +
                "}")
        .build();

    // Apply the Secret to the Kubernetes cluster
    localKubClient.secrets().inNamespace("argocd").createOrReplace(localSecret);

    System.out.println("Secret created or replaced successfully!");
    
    
    
    
    return updateResourceStatus(resource, KCJResourceState.ACTIVE, null, null, "Failed to configure local ArgoCD", localKubClient);
    
  }
  
  
  public static String convertInputStreamToString(InputStream inputStream) throws IOException {
    // Use try-with-resources to ensure InputStream is closed after usage
    try (BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream))) {
        // Use stream to collect the lines into a single string
        return reader.lines().collect(Collectors.joining("\n"));
    }
}
  
  private KCJResource execStrategyInstallRemoteArgo(KCJResource resource, KubernetesClient localKubClient, KubernetesClient remoteKubClient) {
    
    
    // Define the namespace
    Namespace namespace = new NamespaceBuilder()
        .withNewMetadata()
        .withName(ARGOCD_NAMESPACE)  // Replace with your desired namespace name
        .endMetadata()
        .build();

    // Use .resource(namespace) and then .createOrReplace() to create or replace the namespace
    remoteKubClient.resource(namespace).createOr( (operation) -> {      
      try{
        operation.create();
      }catch (Exception e) {
        log.info("NAMESPACE EXISTS. WILL UPDATE IT: " + e.getMessage());
      }
      
      return operation.update();
      }) ;
    
    
    
    try {
      // Download the remote YAML file
      InputStream yamlStream = new URL(REMOTARGO_URL).openStream();
      String yamlString = convertInputStreamToString(yamlStream);
      //System.out.println("Converted String: \n" + yamlString);
      // Convert the YAML string to an InputStream
      InputStream yamlInputStream = new ByteArrayInputStream(yamlString.getBytes(StandardCharsets.UTF_8));

      // Parse the YAML file into Kubernetes resources
      remoteKubClient.load(yamlInputStream).inNamespace(ARGOCD_NAMESPACE).serverSideApply()  ;


      log.info("Applied the remote YAML to namespace: " + namespace);
      return updateResourceStatus(resource, KCJResourceState.ACTIVE, "ArgoCD installed remotelly", null, null, localKubClient);
      
    } catch (IOException e) {
      e.printStackTrace();
      return updateResourceStatus(resource, KCJResourceState.FAILED, "Failed to install remotely ArgoCD", null, null, localKubClient);
    }


    
  }

  /**
   * @param resource
   * @param i
   * @param string
   * @param client
   * @return
   */
  private KCJResource updateResourceStatus(KCJResource resource, KCJResourceState state,
      String infoInstallArgoCD, 
      String infoInstallCridge, 
      String infoEnableLocalArgoCD, KubernetesClient client) {

    try {
      KCJResourceStatus resStatus = resource.getStatus();
      if (resStatus == null) {
        resStatus = new KCJResourceStatus();
      }
      if ( state != null)
        resStatus.setState(state);
      if ( infoInstallArgoCD!= null)
        resStatus.setInfoInstallArgoCD(infoInstallArgoCD);
      if ( infoInstallCridge!= null)
        resStatus.setInfoInstallCridge(infoInstallCridge);
      if ( infoEnableLocalArgoCD!= null)
        resStatus.setInfoEnableLocalArgoCD(infoEnableLocalArgoCD);
      resource.setStatus(resStatus);

      resource = client.resource(resource).patch();

      return resource;
    } catch (Exception e) {
      return null;
    }


  }

  private KCJResource addFinalizers(KCJResource resource, KubernetesClient client) {

    List<String> finalizers = resource.getMetadata().getFinalizers();
    if (finalizers == null) {
      finalizers = new ArrayList<String>();
    }
    finalizers.clear();
    finalizers.add("kcj.controller.osl.etsi.org");
    resource.getMetadata().setFinalizers(finalizers);

    resource = client.resource(resource).patch();
    return resource;
  }

  // Remove the finalizer after cleanup
  private KCJResource removeFinalizer(KCJResource resource, KubernetesClient client) {
    List<String> finalizers = resource.getMetadata().getFinalizers();
    if (finalizers != null) {
      finalizers.clear();
      resource.getMetadata().setFinalizers(finalizers);

      resource = client.resource(resource).patch();
      log.info("Finalizer removed from resource: " + resource.getMetadata().getName());
      return resource;
    }
    return resource;
  }

}
