package org.etsi.osl.controllers.capif.invoker;

import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.X509v3CertificateBuilder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.jcajce.JcaX509v3CertificateBuilder;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMKeyConverter;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import java.io.*;
import java.math.BigInteger;
import java.security.*;
import java.security.cert.X509Certificate;
import java.util.Date;

public class SelfSignedCertFromCSR {

    static {
        Security.addProvider(new BouncyCastleProvider());
    }

    // Load Private Key from PEM
    public static PrivateKey loadPrivateKey(String privateKeyPem) throws Exception {
        PEMParser pemParser = new PEMParser(new StringReader(privateKeyPem));
        Object object = pemParser.readObject();
        pemParser.close();
        return new JcaPEMKeyConverter().getPrivateKey((org.bouncycastle.asn1.pkcs.PrivateKeyInfo) object);
    }

    // Load CSR from PEM
    public static PKCS10CertificationRequest loadCSR(String csrPem) throws IOException {
        PEMParser pemParser = new PEMParser(new StringReader(csrPem));
        Object object = pemParser.readObject();
        pemParser.close();
        return (PKCS10CertificationRequest) object;
    }

    // Generate self-signed certificate from CSR and Private Key
    public static X509Certificate generateSelfSignedCert(PrivateKey privateKey, PKCS10CertificationRequest csr) throws Exception {
        X500Name issuer = csr.getSubject();
        BigInteger serial = new BigInteger(Long.toString(System.currentTimeMillis()));

        Date notBefore = new Date();
        Date notAfter = new Date(System.currentTimeMillis() + (365L * 24 * 60 * 60 * 1000)); // 1 year validity

        X509v3CertificateBuilder certBuilder = new JcaX509v3CertificateBuilder(
                issuer, serial, notBefore, notAfter, csr.getSubject(), csr.getSubjectPublicKeyInfo());

        ContentSigner signer = new JcaContentSignerBuilder("SHA256withRSA")
                .setProvider("BC")
                .build(privateKey);

        X509CertificateHolder holder = certBuilder.build(signer);

        return new JcaX509CertificateConverter()
                .setProvider("BC")
                .getCertificate(holder);
    }

    public static String getCertPem (String privateKeyPem, String csrPem) {

        PrivateKey privateKey;
        try {
          privateKey = loadPrivateKey(privateKeyPem);
          PKCS10CertificationRequest csr = loadCSR(csrPem);

          X509Certificate selfSignedCert = generateSelfSignedCert(privateKey, csr);

          // Print certificate in PEM format
          StringWriter certPem = new StringWriter();
          try (JcaPEMWriter pemWriter = new JcaPEMWriter(certPem)) {
              pemWriter.writeObject(selfSignedCert);
          }

          return certPem.toString();
        } catch (Exception e) {
          // TODO Auto-generated catch block
          e.printStackTrace();
        }
        
        
        return csrPem;
    }
}
