package org.etsi.osl.controllers.capif.invoker;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.nio.charset.StandardCharsets;
import java.security.KeyFactory;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateFactory;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Base64;
import java.util.Map.Entry;
import javax.net.ssl.TrustManagerFactory;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.etsi.osl.controllers.capif.invoker.ApiResponse.AefProfile;
import org.etsi.osl.controllers.capif.invoker.ApiResponse.ServiceAPIDescription;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.client.reactive.ClientHttpConnector;
import org.springframework.http.client.reactive.ReactorClientHttpConnector;
import org.springframework.stereotype.Service;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.reactive.function.client.ExchangeStrategies;
import org.springframework.web.reactive.function.client.WebClient;
import io.netty.channel.ChannelOption;
import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslContextBuilder;
import io.netty.handler.ssl.util.InsecureTrustManagerFactory;
import io.netty.handler.timeout.ReadTimeoutHandler;
import io.netty.handler.timeout.WriteTimeoutHandler;
import lombok.Data;
import reactor.netty.http.client.HttpClient;
import reactor.netty.tcp.TcpClient;

@Data
@Service
public class CapifService {

  private static final Logger logger = LoggerFactory.getLogger("org.etsi.osl.controllers.capif.invoker");

  @Value("${capif.REGISTER_HOSTNAME}")
  String REGISTER_HOSTNAME;

  @Value("${capif.REGISTER_PORT}")
  String REGISTER_PORT;

  @Value("${capif.USER_NAME}")
  String USER_NAME;

  @Value("${capif.USER_PASSWORD}")
  String USER_PASSWORD;

  @Value("${capif.CAPIF_HOSTNAME}")
  String CAPIF_HOSTNAME;

  @Value("${capif.CAPIF_PORT}")
  String CAPIF_PORT;

  @Value("${capif.ONBOARDING_URL}")
  String ONBOARDING_URL;

  @Value("${capif.ONBOARDING_URL_INVOKER}")
  String ONBOARDING_URL_INVOKER;

  @Value("${capif.DISCOVER_URL}")
  String DISCOVER_URL;

  @Value("${capif.ALLOW_INSECURE}")
  boolean ALLOW_INSECURE;

  @Value("${capif.TRUSTSTORE_PATH}")
  String TRUSTSTORE_PATH;

  @Value("${capif.TRUSTSTORE_PASSWORD}")
  String TRUSTSTORE_PASSWORD;

  /**
   * this is given after getAuth
   */
  String accessToken = null;

  String ca_root = null;

  public void authorizeToCapifService() throws Exception {

    logger.info("=== authorizeToCapifService ===");
    
    WebClient client = WebClient.builder().exchangeStrategies(getExchangeStrategies())
        .baseUrl("https://" + REGISTER_HOSTNAME + ":" + REGISTER_PORT)
        .clientConnector(clientHttpConnector())
        .defaultHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE)
        .defaultHeader(HttpHeaders.AUTHORIZATION, basicAuthHeader( USER_NAME, USER_PASSWORD))
        .build();

    String jsonResponse = client.get()
        .uri("/getauth")
        .retrieve()
        .onStatus(
            status -> status.is4xxClientError() || status.is5xxServerError(),
            clientResponse -> clientResponse.bodyToMono(String.class)
                .map(errorBody -> new RuntimeException("API error: " + errorBody) )
        )
        .bodyToMono(String.class).block();

    logger.debug(jsonResponse);

    ObjectMapper objectMapper = new ObjectMapper();
    JsonNode rootNode = objectMapper.readTree(jsonResponse);

    accessToken = rootNode.path("access_token").asText();
    ca_root = rootNode.path("ca_root").asText();



  }

  public SslContext createSslContext(boolean useTrustStore, String trustStorePath,
      String trustStorePassword, boolean allowInsecure) throws Exception {

    if (useTrustStore && trustStorePath != null && !trustStorePath.isEmpty()) {
      KeyStore trustStore = KeyStore.getInstance("JKS");
      try (FileInputStream fis = new FileInputStream(new File(trustStorePath))) {
        trustStore.load(fis, trustStorePassword.toCharArray());
      }

      TrustManagerFactory trustManagerFactory =
          TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
      trustManagerFactory.init(trustStore);

      return SslContextBuilder.forClient().trustManager(trustManagerFactory).build();

    } else if (allowInsecure) {
      // Insecure approach, for development only
      return SslContextBuilder.forClient().trustManager(InsecureTrustManagerFactory.INSTANCE)
          .build();
    }

    // Default SSL context (standard validation)
    return SslContextBuilder.forClient().build();
  }


  private ClientHttpConnector clientHttpConnector() throws Exception {

    // default SSL
    // createSslContext(false, null, null, false);

    SslContext sslContext;
    if (ALLOW_INSECURE) {
      sslContext = createSslContext(false, null, null, true);
    } else {
      sslContext = createSslContext(true, TRUSTSTORE_PATH, TRUSTSTORE_PASSWORD, false);
    }


    TcpClient tcpClient = TcpClient.create()
        // .wiretap(true) //logging on reactor.netty.tcp.TcpClient level to DEBUG
        .option(ChannelOption.CONNECT_TIMEOUT_MILLIS, 30000)
        .doOnConnected(connection -> connection.addHandlerLast(new ReadTimeoutHandler(30))
            .addHandlerLast(new WriteTimeoutHandler(30)));

    return new ReactorClientHttpConnector(HttpClient.from(tcpClient).wiretap(true)// To enable it,
                                                                                  // you must set
                                                                                  // the logger
                                                                                  // reactor.netty.http.client.HttpClient
                                                                                  // level to DEBUG
        .secure(sslContextSpec -> sslContextSpec.sslContext(sslContext)));
  }

  private ExchangeStrategies getExchangeStrategies() {

    ExchangeStrategies exchangeStrategies = ExchangeStrategies.builder()
        .codecs(configurer -> configurer.defaultCodecs().maxInMemorySize(-1)).build(); // spring.codec.max-in-memory-size=-1
                                                                                       // ?? if use
                                                                                       // autoconfiguration
    return exchangeStrategies;
  }

  private static String basicAuthHeader(String username, String password) {
    String credentials = username + ":" + password;
    return "Basic "
        + Base64.getEncoder().encodeToString(credentials.getBytes(StandardCharsets.UTF_8));
  }

  public CapifInvoker onboardInvoker(final CapifInvoker cif) throws Exception {


    logger.info("================== onboardInvoker {} =========================", cif.getName());

    CapifInvoker resultInv = cif;

    WebClient webClient;

    webClient = WebClient.builder()
        .exchangeStrategies(getExchangeStrategies())
        .baseUrl("https://" + CAPIF_HOSTNAME + ":" + CAPIF_PORT)
        .clientConnector(clientHttpConnector())
        .defaultHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE)
        .defaultHeader(HttpHeaders.AUTHORIZATION, "Bearer " + accessToken).build();

    ObjectNode objnode = resultInv.createPayload();;
    

    logger.info("================== onboardInvoker objnode {} =========================", objnode  );

     String response = webClient.post()
        .uri(ONBOARDING_URL_INVOKER)
        .bodyValue( objnode  )
        .retrieve()
        .onStatus(
            status -> status.is4xxClientError() || status.is5xxServerError(),
            clientResponse -> clientResponse.bodyToMono(String.class)
                .map(errorBody -> new RuntimeException("API error: " + errorBody) )
        )
        .bodyToMono(String.class).block();

    logger.info("POST Response from server: " + response);

    ObjectMapper objectMapper = new ObjectMapper();
    JsonNode rootNode = objectMapper.readTree(response);
    resultInv.invokerid = rootNode.path("apiInvokerId").asText();
    resultInv.setApiInvokerPublicKeyCSR(
        rootNode.path("onboardingInformation").path("apiInvokerPublicKey").asText());
    resultInv.setCertificate(
        rootNode.path("onboardingInformation").path("apiInvokerCertificate").asText());

    return resultInv;
  }

  public CapifInvoker putSecurityContext(CapifInvoker cif) throws Exception {

    logger.info("================== putSecurityContext {} =========================",
        cif.getName());

    WebClient webClient = WebClient.builder()
        .exchangeStrategies(getExchangeStrategies())
        .baseUrl("https://" + CAPIF_HOSTNAME + ":" + CAPIF_PORT)
        .clientConnector(clientHttpConnector(ca_root, cif))
        .defaultHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE).build();


    for (Entry<String, ApiResponse> entry : cif.getServiceApis().entrySet()) {
      
      
      ServiceAPIDescription serviceApi = entry.getValue().getServiceAPIDescriptions().get(0) ;
      AefProfile defaultProfile = serviceApi.getAefProfiles().get(0);
      
       

        String jsonPayload = """
            {
              "securityInfo": [
                {
                  "prefSecurityMethods": [
                    "%s"
                  ],
                  "authenticationInfo": "string",
                  "authorizationInfo": "string",
                  "aefId": "%s",
                  "apiId": "%s"
                }
              ],
              "notificationDestination": "https://mynotificationdest.com",
              "requestTestNotification": true,
              "websockNotifConfig": {
                "websocketUri": "string",
                "requestWebsocketUri": true
              },
              "supportedFeatures": "fff"
            }
            """.formatted(defaultProfile.getSecurityMethods().get(0), defaultProfile.getAefId() , serviceApi.getApiId() ) ;


        ObjectMapper mapper = new ObjectMapper();
        ObjectNode payloadNode = (ObjectNode) mapper.readTree(jsonPayload);


        String response = webClient.put()
            .uri("capif-security/v1/trustedInvokers/" + cif.invokerid)
            .bodyValue(payloadNode)
            .retrieve()
            .onStatus(
                status -> status.is4xxClientError() || status.is5xxServerError(),
                clientResponse -> clientResponse.bodyToMono(String.class)
                    .map(errorBody -> new RuntimeException("API error: " + errorBody) )
            ).bodyToMono(String.class).block();

        logger.info("Response from server: " + response);
      }
    
    return cif;

    }

    public CapifInvoker getServiceBearer(CapifInvoker cif) throws Exception {
          

      for (Entry<String, ApiResponse> entry : cif.getServiceApis().entrySet()) {
        
        
        ServiceAPIDescription serviceApi = entry.getValue().getServiceAPIDescriptions().get(0) ;
        AefProfile defaultProfile = serviceApi.getAefProfiles().get(0);
        
        logger.info("========= GetOAuTh Bearer apiid: {}, aefid: {} ==========", defaultProfile.getAefId(), serviceApi.getApiId() );

        WebClient webClient = WebClient.builder()
            .exchangeStrategies(getExchangeStrategies())
            .baseUrl("https://" + CAPIF_HOSTNAME  + ":" + CAPIF_PORT)
            .clientConnector(clientHttpConnector(ca_root, cif))
            .defaultHeader("Content-Type", MediaType.APPLICATION_FORM_URLENCODED_VALUE).build();

        MultiValueMap<String, String> formData = new LinkedMultiValueMap<>();
        formData.add("client_id", cif.invokerid);
        formData.add("grant_type", "client_credentials");
        formData.add("client_secret", "string");
        formData.add("scope", "3gpp#" + defaultProfile.getAefId() + ":" + serviceApi.getApiName() );

        String response = webClient.post().uri("/capif-security/v1/securities/" + cif.invokerid + "/token")
            .bodyValue(formData)
            .retrieve()
            .onStatus(
                status -> status.is4xxClientError() || status.is5xxServerError(),
                clientResponse -> clientResponse.bodyToMono(String.class)
                    .map(errorBody -> new RuntimeException("API error: " + errorBody) )
            ).bodyToMono(String.class).block();

        logger.info("Response: " + response);

        ObjectMapper objectMapper = new ObjectMapper();
        JsonNode rootNode = objectMapper.readTree(response);

        String accessToken = rootNode.path("access_token").asText();
        int expire = rootNode.path("_in").asInt();

        
        cif.setBearerAccessToken( accessToken );
        cif.setBearerExpiresIn(expire);

        logger.info("OAUTH token: " + accessToken);
        return cif;
        
      }
      return cif;
      
    }
    
    
    public void fetchURL(CapifInvoker cif) {
      
        //we must refresh the token.
        
        logger.info("==================Good luck invoker {}=======================", cif.getName());
        
        
        ServiceAPIDescription serviceApi = cif.getServiceApis()
            .entrySet().stream().findFirst().get()
            .getValue().getServiceAPIDescriptions().get(0) ;
        AefProfile defaultProfile = serviceApi.getAefProfiles().get(0);

        String IPV4ADDR = defaultProfile.getInterfaceDescriptions().get(0).getIpv4Addr() ;
        String SERVICE_PORT = defaultProfile.getInterfaceDescriptions().get(0).getPort() + ""  ;
        String SERVICE_URI = defaultProfile.getVersions().get(0).getResources().get(0).getUri();

        logger.info("IPV4ADDR {}", IPV4ADDR);
        logger.info("SERVICE_PORT {}", SERVICE_PORT);
        logger.info("SERVICE_URI {}", SERVICE_URI);
        new Thread(() -> {

          while (true) {

            // pm.environment.set('IPV4ADDR', api.aefProfiles[0].interfaceDescriptions[0].ipv4Addr);
            // pm.environment.set('PORT', api.aefProfiles[0].interfaceDescriptions[0].port);
            // pm.environment.set('URI', api.aefProfiles[0].versions[0].resources[0].uri);

            WebClient webClientx;
            try {
              webClientx = WebClient.builder().exchangeStrategies(getExchangeStrategies())
                  .baseUrl("http://" +IPV4ADDR+ ":" + SERVICE_PORT)
                  .clientConnector(clientHttpConnector(ca_root, cif))
                  .defaultHeader("Authorization", "Bearer " + accessToken)
                  .defaultHeader("Content-Type", MediaType.APPLICATION_JSON_VALUE).build();
                MultiValueMap<String, String> formData2 = new LinkedMultiValueMap<>();
                formData2.add("name", "ExamplePPayload from " + cif.getName());
        
                String responsex = webClientx.post().uri( SERVICE_URI ).bodyValue( formData2 ).retrieve()
                    .bodyToMono(String.class).block();
        
                logger.info("Response: " + responsex);
                
              
            } catch (Exception e) {
              // TODO Auto-generated catch block
              e.printStackTrace();
            }

            try {
              Thread.sleep (1000);
            } catch (InterruptedException e) {
              // TODO Auto-generated catch block
              e.printStackTrace();
            }
           
          }
        }).start();

        
    }




  public CapifInvoker discoverallServiceAPIs(final CapifInvoker cif) throws Exception {

    logger.info("================== discoverallServiceAPIs {}=========================",
        cif.getName());
    // needs ca_root
    // needs client certificates and private key
    CapifInvoker resultCif = cif;

    WebClient webClient = WebClient.builder().exchangeStrategies(getExchangeStrategies())
        .baseUrl("https://" + CAPIF_HOSTNAME + ":" + CAPIF_PORT)
        .clientConnector(clientHttpConnector(ca_root, cif))
        .defaultHeader(HttpHeaders.CONTENT_TYPE, MediaType.APPLICATION_JSON_VALUE).build();

    String response = webClient.get().uri(DISCOVER_URL + "?api-invoker-id=" + cif.getInvokerid())
        .retrieve().bodyToMono(String.class).block();

    System.out.println("Response from server Discover: " + response);

    resultCif.setDiscoveredServiceAPIs(response);

    ObjectMapper objectMapper = new ObjectMapper();
    ApiResponse apiResponse = objectMapper.readValue( response, ApiResponse.class);

    //the response can be a characteristic
    
    for (ServiceAPIDescription serviceApiDesc : apiResponse.getServiceAPIDescriptions()) {
      
      
      String apiName = serviceApiDesc.getApiName(); 
      logger.info("API Name: " + apiName);
      
      for (String reqApiName : resultCif.reqApiNames) {

        if (apiName.equals(reqApiName)) {

          logger.info("API Name found and added: " + apiName);

          String apiId = serviceApiDesc.getApiId();          
          logger.info("  apiId: " + apiId);
          String aefId = serviceApiDesc.getAefProfiles().get(0).getAefId();
          logger.info("  AEF Profile ID: " + aefId);
          
          resultCif.getServiceApis().put(apiId, apiResponse);

        }

      } 
      
      
    }

    
    
    





    return resultCif;

  }

  public static ClientHttpConnector clientHttpConnector(String ca_root, CapifInvoker cif)
      throws Exception {

    // default SSL
    // createSslContext(false, null, null, false);



    SslContext sslContext = createSslContextFile(true, ca_root, true, cif);

    TcpClient tcpClient = TcpClient.create()
        // .wiretap(true) //logging on reactor.netty.tcp.TcpClient level to DEBUG
        .option(ChannelOption.CONNECT_TIMEOUT_MILLIS, 30000)
        .doOnConnected(connection -> connection.addHandlerLast(new ReadTimeoutHandler(30))
            .addHandlerLast(new WriteTimeoutHandler(30)));

    return new ReactorClientHttpConnector(HttpClient.from(tcpClient).wiretap(true)// To enable it,
                                                                                  // you must set
                                                                                  // the logger
                                                                                  // reactor.netty.http.client.HttpClient
                                                                                  // level to DEBUG
        .secure(sslContextSpec -> sslContextSpec.sslContext(sslContext)));
  }

  public static SslContext createSslContextFile(boolean useTrustStore, String caCertPem,
      boolean allowInsecure, CapifInvoker cif) throws Exception {

    // CSR
    // String ourCSRKey= CSRGenerator.getCSR( privateKeyPem, publicKeyPem, "CN=" + invokerId );

    // crt file
    // String clientCertPem= SelfSignedCertFromCSR.getCertPem(privateKeyPem, ourCSRKey) ;


    // Load CA Certificate
    CertificateFactory certFactory = CertificateFactory.getInstance("X.509");
    Certificate caCert = certFactory
        .generateCertificate(new ByteArrayInputStream(caCertPem.getBytes(StandardCharsets.UTF_8)));

    // Load Client Certificate
    Certificate clientCert = certFactory.generateCertificate(
        new ByteArrayInputStream(cif.getCertificate().getBytes(StandardCharsets.UTF_8)));

    // Load Client Private Key
    String privateKeyContent =
        cif.getPrivateKey().replaceAll("-----\\w+ PRIVATE KEY-----", "").replaceAll("\\s+", "");
    byte[] decodedKey = Base64.getDecoder().decode(privateKeyContent);
    PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(decodedKey);
    KeyFactory kf = KeyFactory.getInstance("RSA"); // or "EC" depending on your key type
    PrivateKey privateKey = kf.generatePrivate(keySpec);


    // Create KeyStore with client cert and private key
    KeyStore keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
    keyStore.load(null);
    keyStore.setKeyEntry("client-key", privateKey, new char[0], new Certificate[] {clientCert});

    // TrustStore with CA Certificate
    KeyStore trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
    trustStore.load(null);
    trustStore.setCertificateEntry("ca-cert", caCert);

    // TrustManager setup
    TrustManagerFactory trustManagerFactory =
        TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
    trustManagerFactory.init(trustStore);

    // Build SSL context
    return SslContextBuilder.forClient()
        .keyManager(privateKey, (java.security.cert.X509Certificate) clientCert)
        .trustManager(trustManagerFactory).build();
  }

}
