package org.etsi.osl.controllers.capif.invoker;

import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import lombok.Builder;

public class CapifInvoker {


  private static final Logger logger = LoggerFactory.getLogger("org.etsi.osl.controllers.capif.invoker");


  // this is used only on slef sign tests
  @Builder.Default
  private KeyPair akeyPair = generateRSAKeyPair();
  private String name;
  private String description;
  private String privateKey;
  private String publicKey;
  private String apiInvokerPublicKeyCSR;
  private String certificate;
  String bearerAccessToken;
  private int bearerExpiresIn;
  private String discoveredServiceAPIs;
  
  private CapifInvokerState capifInvokerState = CapifInvokerState.UNKNOWN;

  @Builder.Default
  List<String> reqApiNames = new ArrayList<>();

  @Builder.Default
  Map<String, ApiResponse> serviceApis = new ConcurrentHashMap<>();
  String invokerid;

  /**
   * @param name the name to set
   */
  public void setName(String name) {
    this.name = name;
  }

  /**
   * @param aprivateKey the privateKey to set
   */
  public void setPrivateKey(String aprivateKey) {
    if (aprivateKey.equalsIgnoreCase("selfsign")) {
      aprivateKey = "-----BEGIN PRIVATE KEY-----\n"
          + Base64.getMimeEncoder().encodeToString(akeyPair.getPrivate().getEncoded())
          + "\n-----END PRIVATE KEY-----";
    }
    this.privateKey = aprivateKey;


  }

  /**
   * @return the discoveredServiceAPIs
   */
  public String getDiscoveredServiceAPIs() {
    return discoveredServiceAPIs;
  }

  /**
   * @param discoveredServiceAPIs the discoveredServiceAPIs to set
   */
  public void setDiscoveredServiceAPIs(String discoveredServiceAPIs) {
    this.discoveredServiceAPIs = discoveredServiceAPIs;
  }

  public String getPrivateKey() {
    if (this.privateKey.equalsIgnoreCase("selfsign")) {
      setPrivateKey(this.privateKey);
    }

    return this.privateKey;
  }

  /**
   * @param apublicKey the publicKey to set
   */
  public void setPublicKey(String apublicKey) {
    if (apublicKey.equalsIgnoreCase("selfsign")) {
      apublicKey = "-----BEGIN PUBLIC KEY-----\n"
          + Base64.getMimeEncoder().encodeToString(akeyPair.getPublic().getEncoded())
          + "\n-----END PUBLIC KEY-----";
      this.publicKey = apublicKey;

    } else {
      try {

        PrivateKey privateKey = CSRGenerator.loadPrivateKeyFromPEM(this.privateKey);
        PublicKey pKey = CSRGenerator.derivePublicKey(privateKey);
        String pemString = CSRGenerator.convertToPem(pKey);

        this.publicKey = pemString;
      } catch (Exception e) {
        // TODO Auto-generated catch block
        e.printStackTrace();
      }
    }


    logger.info("INVOKER: {} PrivateKEY {}", this.name, this.privateKey);

    logger.info("INVOKER: {} PublicKEY {}", this.name, this.publicKey);
  }

  public String getPublicKey() {


    return this.publicKey;
  }

  /**
   * @param certificate the certificate to set
   */
  public void setCertificate(String certificate) {
    this.certificate = certificate;
  }


  private static KeyPair generateRSAKeyPair() {
    KeyPairGenerator keyGen;
    try {
      keyGen = KeyPairGenerator.getInstance("RSA");
      keyGen.initialize(2048); // key size 2048 bits
      KeyPair pair = keyGen.generateKeyPair();
      return pair;
    } catch (NoSuchAlgorithmException e) {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
    return null;
  }



  public String getCSR() {
    return CSRGenerator.getCSR(getPrivateKey(), getPublicKey(), "CN=InternalOSL");
  }


  public ObjectNode createPayload() {

    apiInvokerPublicKeyCSR = getCSR();

    ObjectMapper mapper = new ObjectMapper();

    ObjectNode payload = mapper.createObjectNode();
    payload.put("notificationDestination", "http://host.docker.internal:8086/netapp_callback");
    payload.put("supportedFeatures", "fffffff");
    payload.put("apiInvokerInformation", "dummy");

    ObjectNode websockNotifConfig = payload.putObject("websockNotifConfig");
    websockNotifConfig.put("requestWebsocketUri", true);
    websockNotifConfig.put("websocketUri", "websocketUri");

    ObjectNode onboardingInformation = payload.putObject("onboardingInformation");
    onboardingInformation.put("apiInvokerPublicKey", apiInvokerPublicKeyCSR); // dynamic value here

    payload.put("requestTestNotification", true);

    return payload;
  }

  /**
   * @return the name
   */
  public String getName() {
    return name;
  }

  /**
   * @return the certificate
   */
  public String getCertificate() {
    return certificate;
  }


  /**
   * @return the invokerid
   */
  public String getInvokerid() {
    return invokerid;
  }

  /**
   * @return the serviceApis
   */
  public Map<String, ApiResponse> getServiceApis() {
    return serviceApis;
  }

  /**
   * @return the reqApiNames
   */
  public List<String> getReqApiNames() {
    return reqApiNames;
  }

  /**
   * @return the bearerAccessToken
   */
  public String getBearerAccessToken() {
    return bearerAccessToken;
  }

  /**
   * @param bearerAccessToken the bearerAccessToken to set
   */
  public void setBearerAccessToken(String bearerAccessToken) {
    this.bearerAccessToken = bearerAccessToken;
  }

  /**
   * @return the bearerExpiresIn
   */
  public int getBearerExpiresIn() {
    return bearerExpiresIn;
  }

  /**
   * @param bearerExpiresIn the bearerExpiresIn to set
   */
  public void setBearerExpiresIn(int bearerExpiresIn) {
    this.bearerExpiresIn = bearerExpiresIn;
  }

  /**
   * @param apiInvokerPublicKeyCSR the apiInvokerPublicKeyCSR to set
   */
  public void setApiInvokerPublicKeyCSR(String apiInvokerPublicKeyCSR) {
    this.apiInvokerPublicKeyCSR = apiInvokerPublicKeyCSR;
  }


  private String genPulicKey(String privateKeyPem) throws Exception {


    byte[] privateKeyBytes = Base64.getDecoder().decode(privateKeyPem);

    // Generate PrivateKey object
    PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(privateKeyBytes);
    KeyFactory keyFactory = KeyFactory.getInstance("RSA");
    PrivateKey privateKey = keyFactory.generatePrivate(keySpec);

    // Derive public key from private key
    RSAPublicKeySpec publicKeySpec = keyFactory.getKeySpec(privateKey, RSAPublicKeySpec.class);
    PublicKey publicKey = keyFactory.generatePublic(publicKeySpec);

    // Encode public key in PEM format
    byte[] publicKeyEncoded = publicKey.getEncoded();
    String publicKeyPem = "-----BEGIN PUBLIC KEY-----\n"
        + Base64.getMimeEncoder(64, new byte[] {'\n'}).encodeToString(publicKeyEncoded)
        + "\n-----END PUBLIC KEY-----";

    // Print it
    // System.out.println(publicKeyPem);
    return publicKeyPem;
  }



  /**
   * @return the description
   */
  public String getDescription() {
    return description;
  }

  /**
   * @param description the description to set
   */
  public void setDescription(String description) {
    this.description = description;
  }

  /**
   * @return the capifInvokerState
   */
  public CapifInvokerState getCapifInvokerState() {
    return capifInvokerState;
  }

  /**
   * @param capifInvokerState the capifInvokerState to set
   */
  public void setCapifInvokerState(CapifInvokerState capifInvokerState) {
    this.capifInvokerState = capifInvokerState;
  }



}
