package org.etsi.osl.controllers.capif.invoker;

import java.io.StringWriter;
import java.security.*;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Base64;

import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.operator.ContentSigner;
import org.bouncycastle.operator.jcajce.JcaContentSignerBuilder;
import org.bouncycastle.pkcs.PKCS10CertificationRequest;
import org.bouncycastle.pkcs.PKCS10CertificationRequestBuilder;
import org.bouncycastle.pkcs.jcajce.JcaPKCS10CertificationRequestBuilder;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

public class CSRGenerator {

    static {
        Security.addProvider(new BouncyCastleProvider());
    }

    // Generate CSR using derived Public Key and provided Private Key
    public static PKCS10CertificationRequest generateCSR(PrivateKey privateKey, PublicKey publicKey, String subjectDN) throws Exception {
        X500Name subject = new X500Name(subjectDN);

        JcaPKCS10CertificationRequestBuilder csrBuilder =
                new JcaPKCS10CertificationRequestBuilder(subject, publicKey);

        ContentSigner signer = new JcaContentSignerBuilder("SHA256withRSA")
                .setProvider("BC")
                .build(privateKey);

        return csrBuilder.build(signer);
    }

    // Helper to load a private key from PEM-formatted string
    public static PrivateKey loadPrivateKeyFromPEM(String pemKey) throws Exception {
        String privKeyPEM = pemKey.replace("-----BEGIN PRIVATE KEY-----", "")
                .replace("-----END PRIVATE KEY-----", "")
                .replaceAll("\\s+", "");

        byte[] decoded = Base64.getDecoder().decode(privKeyPEM);
        PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(decoded);
        return KeyFactory.getInstance("RSA").generatePrivate(keySpec);
    }
    
    // Helper to load a public key from PEM-formatted string
    public static PublicKey loadPublicKeyFromPEM(String pemKey) throws Exception {
      String publicKeyPEM = pemKey
          .replace("-----BEGIN PUBLIC KEY-----", "")
          .replace("-----END PUBLIC KEY-----", "")
          .replaceAll("\\s", "");

      byte[] decoded = Base64.getDecoder().decode(publicKeyPEM);
    
      X509EncodedKeySpec keySpec = new X509EncodedKeySpec(decoded);
      KeyFactory keyFactory = KeyFactory.getInstance("RSA"); // or EC if using EC keys
      return keyFactory.generatePublic(keySpec);
    }
    
    // Derive Public Key from Private Key (RSA only)
    public static PublicKey derivePublicKey(PrivateKey privateKey) throws Exception {
        RSAPrivateCrtKey rsaPrivateCrtKey = (RSAPrivateCrtKey) privateKey;
        RSAPublicKeySpec publicKeySpec = new RSAPublicKeySpec(
                rsaPrivateCrtKey.getModulus(), rsaPrivateCrtKey.getPublicExponent());
        KeyFactory keyFactory = KeyFactory.getInstance("RSA");
        return keyFactory.generatePublic(publicKeySpec);
    }

      public static String convertToPem(PublicKey publicKey) {
          String encodedKey = Base64.getMimeEncoder(64, new byte[]{'\n'}).encodeToString(publicKey.getEncoded());
          return "-----BEGIN PUBLIC KEY-----\n" + encodedKey + "\n-----END PUBLIC KEY-----";
      }
      
      
    public static String getCSR( String privateKeyPem, String publicKeyPem, String subjectDN ) {

        
        try {        
          
          PrivateKey privateKey = loadPrivateKeyFromPEM(privateKeyPem);
          PublicKey publicKey = loadPublicKeyFromPEM(publicKeyPem);
          
          

          //String subjectDN = "CN=myapp.example.com,O=MyCompany,L=Athens,C=GR";

          PKCS10CertificationRequest csr = generateCSR( privateKey, publicKey , subjectDN);

          StringWriter csrPem = new StringWriter();
          try (JcaPEMWriter pemWriter = new JcaPEMWriter(csrPem)) {
              pemWriter.writeObject(csr);
          }

          return csrPem.toString();
          
          
        } catch (Exception e) {
          // TODO Auto-generated catch block
          e.printStackTrace();
        }
        
        
        return null;


    }
}
