from flask import jsonify, request
from pydantic import ValidationError
from edge_cloud_management_api.managers.log_manager import logger
from edge_cloud_management_api.services.pi_edge_services import PiEdgeAPIClientFactory



class NotFound404Exception(Exception):
    pass


def submit_app(body: dict):
    """
    Controller for submitting application metadata.
    """
    try:
        pi_edge_factory = PiEdgeAPIClientFactory()
        api_client = pi_edge_factory.create_pi_edge_api_client()
        response = api_client.submit_app(body)
        return response

    except ValidationError as e:
        return jsonify({"error": "Invalid input", "details": e.errors()}), 400

    except Exception as e:
        return (
            jsonify({"error": "An unexpected error occurred", "details": str(e)}),
            500,
        )


def get_apps(x_correlator=None):
    """Retrieve metadata information of all applications"""
    try:
        pi_edge_factory = PiEdgeAPIClientFactory()
        api_client = pi_edge_factory.create_pi_edge_api_client()
        registered_apps = api_client.get_service_functions_catalogue()
        return registered_apps
    except Exception as e:
        return (
            jsonify({"error": "An unexpected error occurred", "details": str(e)}),
            500,
        )


def get_app(appId, x_correlator=None):
    """Retrieve the information of an Application"""
    try:
        pi_edge_factory = PiEdgeAPIClientFactory()
        api_client = pi_edge_factory.create_pi_edge_api_client()
        response = api_client.get_app(appId)
        return response

    except NotFound404Exception:
        return (
            jsonify({"status": 404, "code": "NOT_FOUND", "message": "Resource does not exist"}),
            404,
        )

    except Exception as e:
        return (
            jsonify({"error": "An unexpected error occurred", "details": str(e)}),
            500,
        )


def delete_app(appId, x_correlator=None):
    """Delete Application metadata from an Edge Cloud Provider"""
    try:
        pi_edge_factory = PiEdgeAPIClientFactory()
        api_client = pi_edge_factory.create_pi_edge_api_client()
        response = api_client.delete_app(appId=appId)
        return response

    except NotFound404Exception:
        return (
            jsonify({"status": 404, "code": "NOT_FOUND", "message": "Resource does not exist"}),
            404,
        )

    except Exception as e:
        return (
            jsonify({"status": 500, "code": "INTERNAL", "message": f"Internal server error: {str(e)}"}),
            500,
        )


def create_app_instance():
    logger.info("Received request to create app instance")
    
    try: 
        
       body = request.get_json()
       logger.debug(f"Request body: {body}")
       
       app_id = body.get("appId")
       edge_zone_id = body.get("edgeCloudZoneId")
    #    k8s_ref = body.get("kubernetesClusterRef")
       
       if not app_id or not edge_zone_id :
           return jsonify({"error": "Missing required fields: appId, edgeCloudZoneId, or kubernetesCLusterRef"}), 400
       
       logger.info(f"Preparing to send deployment request to SRM for appId={app_id}")
       
       pi_edge_client_factory = PiEdgeAPIClientFactory()
       pi_edge_client = pi_edge_client_factory.create_pi_edge_api_client()
       
       print("\n === Preparing Deployment Request ===")
       print(f" Endpoint: {pi_edge_client.base_url}/deployedServiceFunction")
       print(f" Headers: {pi_edge_client._get_headers()}")
       print(f"Payload: {body}")
       print("=== End of Deployment Request ===\n")
       
       try:
          response = pi_edge_client.deploy_service_function(data=body)
          
          if isinstance(response, dict) and "error" in response:
              logger.warning(f"Failed to deploy service function: {response}")
              return jsonify({
                  "warning": "Deployment not completed (SRM service unreachable)",
                  "details": response
                  
              }), 202
              
          logger.info(f"Deployment response from SRM: {response}")
       except Exception as inner_error:
           logger.error(f"Exception while trying to deploy to SRM: {inner_error}")
           return jsonify({
               "warning": "SRM backend unavailable. Deployment request was built correctly.",
               "details": str(inner_error)
           }),202
       return response   
    #    return jsonify({"message": f"Application {app_id} instantiation accepted"}), 202
    except ValidationError as e:
        logger.error(f"Validation error: {str(e)}")
        return jsonify({"error": "Validation error", "details": str(e)}), 400
    except Exception as e:
        logger.error(f"Unexpected error in create_app_instance:{str(e)}")
        return jsonify({"error": "An unexpected error occurred", "details": str(e)}), 500   

def get_app_instance(app_id=None, x_correlator=None, app_instance_id=None, region=None):
    """
    Retrieve application instances from the database.
    Supports filtering by app_id, app_instance_id, and region.
    """
    try:
        instances = None
        pi_edge_client_factory = PiEdgeAPIClientFactory()
        pi_edge_client = pi_edge_client_factory.create_pi_edge_api_client()

        if app_id is None and app_instance_id is None:
            instances = pi_edge_client.get_app_instances()

        if not instances:
            return jsonify({
                "status": 404,
                "code": "NOT_FOUND",
                "message": "No application instances found for the given parameters."
            }), 404

        return jsonify({"appInstanceInfo": instances}), 200

    except Exception as e:
        logger.exception("Failed to retrieve app instances")
        return jsonify({
            "status": 500,
            "code": "INTERNAL",
            "message": f"Internal server error: {str(e)}"
        }), 500


def delete_app_instance(appInstanceId: str, x_correlator=None):
    
    try:
        pi_edge_client_factory = PiEdgeAPIClientFactory()
        pi_edge_client = pi_edge_client_factory.create_pi_edge_api_client()
        response = pi_edge_client.delete_app_instance(appInstanceId)
        return jsonify({'result': response.text, 'status': response.status_code})

    except Exception as e:
        return (
            jsonify({
                "status": 500,
                "code": "INTERNAL",
                "message": f"Internal server error: {str(e)}"
            }),
            500,
        )
