# OpenCAPIF SDK

This repository develops a Python Software Development Kit (SDK) which facilitates integration with OpenCAPIF (Common API Framework for 3GPP Northbound APIs), significantly reducing integration complexity and enabling developers to focus on building Network Applications (Network Apps).

The SDK supports both CAPIF roles: **Invoker** and **Provider**, and provides simplified methods for interacting with OpenCAPIF. It is compatible with the following public releases:

- [OpenCAPIF Release 1.0](https://ocf.etsi.org/documentation/v1.0.0-release/)
- [OpenCAPIF Release 2.0](https://ocf.etsi.org/documentation/v2.0.0-release/)

## Network App Developers

A Network App refers to an external application or service that interacts with the 3GPP network using standardized APIs. These apps can be created by network operators, third-party providers, or other entities to access advanced network services like QoS, network slicing, or location services via CAPIF.

Network Apps take on one of two roles:

- **Invoker**: Consumes APIs exposed by providers to access network capabilities.
- **Provider**: Exposes APIs/services to be consumed by invokers. A Provider includes:
  - **AMF** (API Management Function): Manages onboarding/offboarding, monitoring, and auditing.
  - **APF** (API Publishing Function): Publishes APIs to the CCF.
  - **AEF** (API Exposing Function): Exposes APIs, validates authorizations, and logs usage.

The SDK maps Python functions to CAPIF OpenAPI endpoints as defined in [3GPP TS 29.222 v18.5.0](https://www.etsi.org/deliver/etsi_ts/129200_129299/129222/18.05.00_60/ts_129222v180500p.pdf). A detailed mapping of supported endpoints is available in the full documentation.

## Requirements

To use the OpenCAPIF SDK, a registered account on the CAPIF instance is needed. Contact the system administrator to obtain a CAPIF username and password.

## Installation

Install the SDK using pip:

```bash
pip install opencapif_sdk
```

## Configuration Overview

Configuration is done through a JSON file (`capif_sdk_config.json`) with common and role-specific fields.

### Common Fields (Invoker & Provider)
- `capif_host`, `register_host`
- `capif_https_port`, `capif_register_port`
- `capif_username`, `capif_password`
- `debug_mode`

### Invoker-specific Fields
- `invoker_folder`, `capif_callback_url`
- `supported_features`, `cert_generation`
- Optional: `discover_filter`, `check_authentication_data`

### Provider-specific Fields
- `provider_folder`, `supported_features`, `cert_generation`
- `APFs`, `AEFs`, `publish_req`, `api_description_path`

## Using `discover_filter`

The `discover_filter` must comply with the [Discover Service API](https://github.com/jdegre/5GC_APIs/blob/Rel-18/TS29222_CAPIF_Discover_Service_API.yaml). It allows filtering by specific fields (e.g., `api-name`) during discovery.

## Using `publish_req`

This section enables API publishing using the [CAPIF Publish Service API](https://github.com/jdegre/5GC_APIs/blob/Rel-18/TS29222_CAPIF_Publish_Service_API.yaml). Required fields include:

- `service_api_id`
- `publisher_apf_id`
- `publisher_aefs_ids`
- `api_description_path`

A helper is available to translate OpenAPI to ServiceAPIDescription format:

```python
from opencapif_sdk import api_schema_translator
translator = api_schema_translator("./path/to/openapi.yaml")
translator.build("https://192.168.1.10:8080/exampleAPI/v1", "0", "0")
```

## Configuration Field Descriptions

Key parameters include:
- `capif_username`, `capif_password`: Credentials
- `supported_features`: Hex bitmask (e.g., "4")
- `debug_mode`: Enables debug logging
- `api_description_path`: Path to the ServiceAPIDescription JSON

## Provider Artifacts

In `provider_folder/<capif_username>/`, the SDK stores:
- `provider_capif_ids.json`
- `capif_<api_name>_<api_id>.json`
- `service_received.json`
- `provider_service_ids.json`

## Invoker Artifacts

In `invoker_folder/<capif_username>/`, the SDK stores:
- `capif_api_security_context_details.json`, which includes:
  - `api_invoker_id`
  - Available API metadata (after discovery)
  - JWT token (after `get_tokens()`)

## Known Issues

The following APIs are not yet supported:
- [CAPIF Access Control Policy](https://github.com/jdegre/5GC_APIs/blob/Rel-18/TS29222_CAPIF_Access_Control_Policy_API.yaml)
- [CAPIF Auditing API](https://github.com/jdegre/5GC_APIs/blob/Rel-18/TS29222_CAPIF_Auditing_API.yaml)
- [CAPIF Routing Info](https://github.com/jdegre/5GC_APIs/blob/Rel-18/TS29222_CAPIF_Routing_Info_API.yaml)
- Security API operations:
  - `/trustedInvokers/{apiInvokerId}` (GET, DELETE)
  - `/trustedInvokers/{apiInvokerId}/delete` (POST)

Some publishing operations fail if the number of AEFs/APFs is increased after initial onboarding.

