
# OpenCAPIF SDK full documentation

The OpenCAPIF SDK facilitates the integration of applications with the CAPIF NF. It offers various features for manual usage, automated scripting, and direct integration into application code.

This documentation provides a step-by-step guide for utilizing the SDK, detailing its functionalities. Before proceeding, ensure all prerequisites are met, and review the available testing modes for the SDK.

## Getting Started

Before using the SDK, the following steps should be completed:
- Meet the [requirements](../README.md),
- Follow the [installation instructions](./sdk_developers.md),
- Configure the SDK by completing the relevant sections in the [configuration guide](./sdk_configuration.md), depending on the CAPIF role the Network App will assume.

## Available SDK Usage Modes

![GENERAL CAPIF USAGE FLOW](./images/flows-updated_opencapif.jpg)

The repository provides two modes for utilizing the OpenCAPIF SDK:

1. **Development Mode**: The SDK can be imported directly into code for development purposes. Sample applications using the SDK are available in the [network_app_samples](../network_app_samples/) folder.

2. **Manual Mode**: A set of Python [scripts](../scripts/) can be used to manually test each integration step. For manual usage, it is necessary to complete the utilities file with absolute paths from the target environment to finalize SDK configuration.

**IMPORTANT**: All SDK configuration files must be filled out based on the intended role and features. Further details can be found in the [Configuration Section](./sdk_configuration.md).

**NOTE**: The register file is not required for SDK usage, only for SDK consumers that wish to create their `capif_username`.

## Table of Contents

As outlined in the [Network App developers section](../README.md), the OpenCAPIF SDK supports two primary roles:

- [Provider Network App](#provider-network-app)
    - [Important Information for Providers](#important-information-for-providers)
    - [Provider Onboarding](#provider-onboarding)
    - [Service Publishing](#service-publishing)
    - [Service Deletion](#service-deletion)
    - [Service Updates](#service-update)
    - [Get Published Services](#get-services)
    - [Get All Published Services](#get-all-services)
    - [Update and Offboard Provider](#update-and-offboard-provider)
- [Invoker Network App](#invoker-network-app)
    - [Important Information for Invokers](#important-information-for-invokers)
    - [Invoker Onboarding](#invoker-onboarding)
    - [Service Discovery](#service-discovery)
    - [Obtain JWT Tokens](#obtain-jwt-tokens)
    - [Check authentication](#check-authentication)
    - [Update and Offboard Invoker](#update-and-offboard-invoker)
- [Other Features](#other-features)
    - [Openapi translation](#openapi-translation)
    - [CAPIF Registration and Login](#capif-registration-and-login)
    - [CAPIF Deregistration and Logout](#capif-registration-and-login)

## Provider Network App

The OpenCAPIF SDK enables efficient implementation of invoker functionality for Network App. This section details the SDK features related to CAPIF providers.

### Important Information for Providers

Within the `provider_folder`, directories are created based on the registered `capif_username`. Each folder contains:

- `provider_service_ids.json`: Contains all APFs and AEFs IDs onboarded with the associated username.
- `capif_<api_name>_<api_id>.json`: Stores the last payload for any published or updated API.
- `service_received.json`: Stores responses for Get API or Get All APIs.
- `provider_capif_ids.json`: A list of currently published APIs along with their IDs.

All the configuration values are available within the object capif_provider_connector.

The provider_capif_ids variable stores the `provider_service_ids.json` content in a dictionary form.

### Provider Onboarding

OpenCAPIF SDK references:
- **Function**: `onboard_provider(supp_features)`
- **Script**: `provider_capif_connector.py`

The SDK simplifies the onboarding process, allowing providers to register multiple APFs and AEFs. All APFs, AEFs, and AMF certificates are created and stored in `provider_service_ids.json`.
`supp_features` parameter is optional and it stands for communicating to the CCF the supported features.It's default value its 0.


![Provider_onboard](./images/flows-provider_onboard.jpg)

### Service Publishing

OpenCAPIF SDK references:
- **Function**: `publish_services()`
- **Script**: `provider_publish_api.py`

The SDK streamlines API publishing with the option to select specific APFs and AEFs. A copy of the uploaded API is stored in `capif_<api_name>_<api_id>.json`, and the `provider_capif_ids.json` is updated with the API name and its ID.

It is also important to have previously prepared the **API schema description** file of the API to be published. **This file must follow the [CAPIF_Publish_Service_API](https://github.com/jdegre/5GC_APIs/blob/Rel-18/TS29222_CAPIF_Publish_Service_API.yaml) 3GPP specification.**

To obtain this schema, opencapif_sdk has a facility to translate Openapi structures to ServiceAPIDescription schemas.
More information:[Translator functionality](#openapi-translation)

**Important**: The SDK will automatically add in the `custom operations`, within the API description path, the capabilities for exposing the [AEF_security](https://github.com/jdegre/5GC_APIs/blob/Rel-18/TS29222_AEF_Security_API.yaml) methods. The developer must code this API endpoints. 

**Required SDK inputs**:
- publisher_apf_id
- publisher_aefs_ids
- api_description_path

### Service Deletion

OpenCAPIF SDK references:
- **Function**: `unpublish_service()`
- **Script**: `provider_unpublish_api.py`
  
The SDK simplifies API deletion. Service deletion requires prior onboarding and service publication.

**Required SDK inputs**:
- publisher_apf_id
- publisher_aefs_ids
- service_api_id

### Service Update

OpenCAPIF SDK references:
- **Function**: `update_service()`
- **Script**: `provider_update_api.py`

This function enables to update a previously registered API. Selecting APFs and AEFs is possible. Onboarding and service publishing are prerequisites.

**Required SDK inputs**:
- service_api_id
- publisher_apf_id
- publisher_aefs_ids

### Get Services

OpenCAPIF SDK references:
- **Function**: `get_service()`
- **Script**: `provider_get_published_api.py`

Retrieve information of a previously published service, stored in `service_received.json`. Prior onboarding and service publication are necessary.

**Required SDK inputs**:
- service_api_id
- publisher_apf_id

### Get All Services

OpenCAPIF SDK references:
- **Function**: `get_all_services()`
- **Script**: `provider_get_all_published_api.py`

Retrieve information about all previously published services in `service_received.json`. Ensure you are [onboarded as a provider](#provider-onboarding) and have [published services](#services-publishing).

**Required SDK input**:
- publisher_apf_id

![Provider_publish](./images/flows-provider_publish_functions.jpg)

### Update and Offboard Provider

OpenCAPIF SDK references:
- **Functions**: `update_provider(supp_features)` and `offboard_provider()`
- **Scripts**: `provider_capif_connector_update.py` and `provider_capif_connector_offboarding.py`

`update_provider(supp_features)`: The provider updates his features such as `APFs`, `AEFs`, etc...
`offboard_provider()`: The provider offboards from CAPIF, this will cause the erase of the published APIs that were currently exposed. 

`supp_features` parameter is optional and it stands for communicating to the CCF the supported features.It's default value its 0.

The provider must be onboarded before using these features.

![Provider_update-offboard](./images/flows-provider_update_offboard.jpg)

### Create logs

OpenCAPIF SDK references:
- **Function**: `create_logs(aefId, jwt, supp_features)`

The provider notifies to the CCF that the published API has been used by certain invoker.

`supp_features` parameter is optional and it stands for communicating to the CCF the supported features.It's default value its 0.

For leveraging this feature the Provider must have onboarded and published an API previously.

**Required SDK input**:
- aefId (Within the function)
- jwt (Within the function)
- log (Within [SDK configuration](./sdk_configuration.md) or object)

![Provider_logs](./images/flows-provider_logs.jpg)

### Create subscription

OpenCAPIF SDK references:
- **Function**: `create_subscription(name, id, supp_features)`

The provider ask to the CCF about notifications related to services such as SERVICE_API_AVAILABLE or API_INVOKER_UPDATED.

`supp_features` parameter is optional and it stands for communicating to the CCF the supported features.It's default value its 0.

This services are specificated in [CAPIF Events API management](https://github.com/jdegre/5GC_APIs/blob/Rel-18/TS29222_CAPIF_Events_API.yaml) explained in [SDK configuration](./sdk_configuration.md#descriptions-of-capif_sdk_config-fields)

For leveraging this feature the Provider must have onboarded previously.

**Required SDK input**:

- aefId//apfId//amfId (Within the function)
- name: An arbitrary name we want to set in order to store it.
- events (Within [SDK configuration](./sdk_configuration.md#descriptions-of-capif_sdk_config-fields) or object)

### Delete subscription

OpenCAPIF SDK references:
- **Function**: `delete_subscription(name, id)`

The provider ask to the CCF to withdraw the subscription to the notifications asked previously

For leveraging this feature the Provider must have onboarded and created a subscription previously.

**Required SDK input**:

- aefId//apfId//amfId (Within the function)
- name: The name of your subscription.

### Update subscription

OpenCAPIF SDK references:
- **Function**: `update_subscription(name, id, supp_features)`

The provider ask to the CCF about updating the subscription for receiving different services such as SERVICE_API_AVAILABLE or API_INVOKER_UPDATED, changing the URL for receiving the notifications...

This services are specificated in [CAPIF Events API management](https://github.com/jdegre/5GC_APIs/blob/Rel-18/TS29222_CAPIF_Events_API.yaml) explained in [SDK configuration](./sdk_configuration.md#descriptions-of-capif_sdk_config-fields)

`supp_features` parameter is optional and it stands for communicating to the CCF the supported features.It's default value its 0.

For leveraging this feature the Provider must have onboarded and created a subscription previously.

![Events_feature](./images/flows-event_subscription.jpg)

**ONLY AVAILABLE IN CAPIF RELEASE 2**

**Required SDK input**:

- aefId//apfId//amfId (Within the function)
- name: The name of your subscription.
- events (Within [SDK configuration](./sdk_configuration.md#events_configuration) or object)

### Patch subscription

OpenCAPIF SDK references:
- **Function**: `update_subscription(name, id)`

The provider ask to the CCF about updating the subscription for receiving different services such as SERVICE_API_AVAILABLE or API_INVOKER_UPDATED.

`supp_features` parameter is optional and it stands for communicating to the CCF the supported features.It's default value its 0.

This services are specificated in [CAPIF Events API management](https://github.com/jdegre/5GC_APIs/blob/Rel-18/TS29222_CAPIF_Events_API.yaml) explained in [SDK configuration](./sdk_configuration.md#events_configuration)

For leveraging this feature the Provider must have onboarded and created a subscription previously.

**ONLY AVAILABLE IN CAPIF RELEASE 2**

**Required SDK input**:

- aefId//apfId//amfId (Within the function)
- name: The name of your subscription.
- events (Within [SDK configuration](./sdk_configuration.md#events_configuration) or object)


## Invoker Network App

The OpenCAPIF SDK enables efficient implementation of invoker functionality for Network App. This section details the SDK features related to CAPIF invokers.

### Important Information for Invokers

Within the `invoker_folder`, directories are created based on the registered `capif_username`. These directories contain:

- `capif_api_security_context_details.json`: Stores details about the invoker, including:
    1. `api_invoker_id`
    2. Discovered APIs and their information (if Service Discovery has been used)
    3. JWT access tokens for discovered APIs (if Service Get Token functionality has been used)

The `token` variable is also available for retrieving the JWT token after the get_tokens() functionality.

The invoker_capif_details variable stores the `capif_api_security_context_details.json` content.

### Invoker onboarding

OpenCAPIF SDK references:
- **Function**: `onboard_invoker(supp_features)`
- **Script**: `invoker_capif_connector.py`

The SDK streamlines the invoker onboarding process, storing the `api_invoker_id` in the `capif_api_security_context_details.json`.

`supp_features` parameter is optional and it stands for communicating to the CCF the supported features.It's default value its 0.


![Invoker_onboard](./images/flows-invoker_onboard.jpg)

### Service Discovery

OpenCAPIF SDK references:
- **Function**: `discover()`
- **Script**: `invoker_service_discover.py`

The [discover_filter](./sdk_configuration.md) can be used to retrieve access to APIs. The invoker must be onboarded before using this function. Discovered APIs and their information are stored in `capif_api_security_context_details.json`.

**Note**: A 404 warning may be received during the first discovery run, prompting registration for the security service.

Use the [discover_filter](./sdk_configuration.md) to retrieve access to target APIs. Ensure you are [onboarded as an invoker](#invoker-onboarding) before using this feature.

![Invoker_discover](./images/flows-invoker_discover.jpg)

### Obtain JWT Tokens

OpenCAPIF SDK references:
- **Function**: `get_tokens(supp_features)`
- **Script**: `invoker_service_get_token.py`

The SDK facilitates JWT token creation for secure access to target APIs. This process stores JWT access token in `capif_api_security_context_details.json`.
`supp_features` parameter is optional and it stands for retrieve the token of the services that have certain supported features.It's default value its 0.

![Invoker_get_token](./images/flows-invoker_get_tokens.jpg)

### Check authentication

OpenCAPIF SDK references:
- **Function**: `check_authentication(supported_features)`

The SDK allows the Network App Invoker to check the `supported_features` from the target Provider's API exposing function (AEF).

It is mandatory to have obtained the [JWT token](#obtain-jwt-tokens) previously. 

**Required SDK inputs**:
- check_authentication_data

![Invoker_check_authentication](./images/flows-invoker_check_authentication.jpg)

### Update and Offboard Invoker

OpenCAPIF SDK references:
- **Functions**: `update_invoker(supp_features)` and `offboard_invoker()`
- **Scripts**: `invoker_capif_connector_update.py` and `invoker_capif_connector_offboarding.py`

Onboarding is required before utilizing these functions.

`supp_features` parameter is optional and it stands for communicating to the CCF the supported features.It's default value its 0.


![Invoker_update-offboard](./images/flows-invoker_update_offboard.jpg)

## Other Features

### Openapi translation

The `api_description_path` must point to the Publish API to be shared, and it should follow the [ServiceAPIDescription](https://github.com/jdegre/5GC_APIs/blob/Rel-18/TS29222_CAPIF_Publish_Service_API.yaml) schema.

This schema could be obtained by applying this code.
```python
    import opencapif_sdk
    
    translator = opencapif_sdk.api_schema_translator("./path/to/openapi.yaml")
    translator.build("https://192.168.1.10:8080/exampleAPI/v1", "0", "0")
```
This code will read `openapi.yaml`, ensure the structure of it and translate the content into ServiceAPIDescription schema, then will create a .json named `api_description_name`. Also it is necessary to fill the ip and port fields to create correctly the schema.
The supported_features and api_supp_features fields corresponds to the capabilities of the provider and the service that the user is sharing.

### CAPIF Registration and Login

OpenCAPIF SDK reference:
* **Script**: `register_and_login.py`

Simplifies the login process for admin users and creates a CAPIF user.

### CAPIF Deregistration and Logout 

OpenCAPIF SDK reference:
* **Script**: `deregister_and_login.py`

Simplifies the logout process for admin users and removes a CAPIF user.

![Register picture](./images/flows-sdk_with_register.jpg)
