#!/bin/bash

# Use custom kubeconfig. If you set here the path to a kubeconfig file it will be used in installation/uninstallation scripts
export KUBECONFIG=""
if [ -z "$KUBECONFIG" ]; then
    echo "The variable KUBECONFIG is empty. Using default k8s environment..."
else
    KUBECONFIG="--kubeconfig $KUBECONFIG"
    echo "The variable KUBECONFIG is not empty. Its value is: $KUBECONFIG"
fi

# timestap to use along scripts
export timestamp=$(date +"%Y%m%d_%H%M%S")

# k8s public ip. NONE will indicate no local register service DNS resolution to reach CCF, empty value will try to get ip of ingress-nginx-controller NodePort
# and any other vaule will set resolution to K8S_IP set for CAPIF_HOSTNAME.
export K8S_IP=""

# Directories variables setup (no modification needed)
export SCRIPTS_DIR=$(dirname "$(readlink -f "$0")")
export HELM_DIR=$(dirname "$SCRIPTS_DIR")
export CAPIF_BASE_DIR=$(dirname "$HELM_DIR")

# Print scripts directory
echo "The /helm/scripts directory is: $SCRIPTS_DIR"
echo "The /helm directory is: $HELM_DIR"
echo "The base directory is: $CAPIF_BASE_DIR"

# Configuration needed before use installation/uninstallation scripts

# Vault installation variables
## Vault configuration
export VAULT_HOSTNAME=vault.testbed.develop
export VAULT_NAMESPACE=ocf-vault
export VAULT_SERVICE_NAME='vault'
export LABEL_TO_CHECK="app.kubernetes.io/name"

## File to store key and token
export VAULT_FILE="$HELM_DIR/vault_keys.txt"

## Vault domains to be included
export DOMAIN1=*.testbed.pre-production
export DOMAIN2=*.testbed.validation
export DOMAIN3=*.testbed.develop

## Vault configuration job
VAULT_JOB_NAME=vault-pki

# Monitoring installation variables
## Prometheus Hostname to be used at ingress configuration
export PROMETHEUS_HOSTNAME=prometheus.testbed.develop
export MONITORING_NAMESPACE=monitoring
export MONITORING_SERVICE_NAME=monitoring

# OpenCAPIF deployment variables
## Register and Capif hostname to be deployed
export CAPIF_HOSTNAME="capif.testbed.develop"
export REGISTER_HOSTNAME="register.testbed.develop"
## namespace to use
export CAPIF_NAMESPACE=ocf-capif
## version to be used on deployment
export CAPIF_NAME_VERSION_CHART=ocf-release1
## Configuration of endpoints in ingress for grafana, mock-server and both mongo express instances.
### this configuration is used to add this script to ocf-mon-$CAPIF_CI_ENV_ENDPOINT.$CAPIF_DOMAIN mock-server-$CAPIF_CI_ENV_ENDPOINT.$CAPIF_DOMAIN mongo-express-register-$CAPIF_CI_ENV_ENDPOINT.$CAPIF_DOMAIN mongo-express-$CAPIF_CI_ENV_ENDPOINT.$CAPIF_DOMAIN
export CAPIF_CI_ENV_ENDPOINT=capif
### Domain to ve used in grafana, mock-server and both mongo express instances.
export CAPIF_DOMAIN=testbed.develop
## Configuration of images to be used on deplyment
### Docker Registry to download images (must be accesible by k8s cluster)
export CAPIF_DOCKER_REGISTRY="labs.etsi.org:5050/ocf/capif/prod"
### Tag to be used
export CAPIF_IMAGE_TAG="v1.0.0-release"
## Prometheus url, usually internal k8s hostname (if capif will be deployed on same k8s cluster) with port 9090
export PROMETHEUS_URL="http://$MONITORING_SERVICE_NAME-prometheus.$MONITORING_NAMESPACE.svc.cluster.local:9090"
## vault capif configuration
export VAULT_INTERNAL_HOSTNAME="$VAULT_SERVICE_NAME.$VAULT_NAMESPACE.svc.cluster.local"
export VAULT_PORT="8200"
export VAULT_ACCESS_TOKEN="dev-only-token"

### To deploy in other environment we need to setup urls according to it and also using specific kubeconfig:
if [ -f "$VAULT_FILE" ] && [ -s "$VAULT_FILE" ]; then
    VAULT_ACCESS_TOKEN=$(awk '/Initial Root Token/{ print $4 }' $VAULT_FILE)
    echo "$VAULT_FILE exists and has content."
else
    echo "$VAULT_FILE not exists or content is empty."
fi
echo "Using value on VAULT_ACCESS_TOKEN=$VAULT_ACCESS_TOKEN"

### If K8S_IP is empty, then script will try to get ingress-nginx-controller NodePort to grant DNS resolution for register to connect locally to CAPIF nginx
if [ "$K8S_IP" == "NONE" ]; then
    echo "K8S_IP value is NONE. Register service will not have local DNS resolution"
elif [ -z "$K8S_IP" ]; then
    K8S_IP=$(kubectl $KUBECONFIG get svc -A | grep ingress-nginx-controller | awk '/NodePort/{ print $4 }')
    echo "K8S_IP value will be $K8S_IP"
fi
