#!/bin/bash
source $(dirname "$(readlink -f "$0")")/variables.sh

help() {
  echo "Usage: $1 <options>"
  echo "       -c : Clean capif services"
  echo "       -v : Clean vault service"
  echo "       -r : Clean register service"
  echo "       -m : Clean monitoring service"
  echo "       -s : Clean Robot Mock service"
  echo "       -a : Clean all services"
  echo "       -z : Clean images generated by docker-compose. Boolean. Default false"
  echo "       -h : show this help"
  exit 1
}

if [[ $# -lt 1 ]]
then
  echo "You must specify an option before run script."
  help
fi

FILES=()
echo "${FILES[@]}"

# Read params
while getopts "cvrahmsz:" opt; do
  case $opt in
    c)
      echo "Remove Capif services"
      FILES+=("$SERVICES_DIR/docker-compose-capif.yml")
      ;;
    v)
      echo "Remove vault service"
      FILES+=("$SERVICES_DIR/docker-compose-vault.yml")
      ;;
    r)
      echo "Remove register service"
      FILES+=("$SERVICES_DIR/docker-compose-register.yml")
      ;;
    m)
      echo "Remove monitoring service"
      FILES+=("$SERVICES_DIR/monitoring/docker-compose.yml")
      ;;
    s)
      echo "Robot Mock Server"
      FILES+=("$SERVICES_DIR/docker-compose-mock-server.yml")
      ;;
    a)
      echo "Remove all services"
      FILES=("$SERVICES_DIR/docker-compose-capif.yml" "$SERVICES_DIR/docker-compose-vault.yml" "$SERVICES_DIR/docker-compose-register.yml" "$SERVICES_DIR/docker-compose-mock-server.yml" "$SERVICES_DIR//monitoring/docker-compose.yml")
      ;;
    z)
      echo "Remove images generated by docker-compose"
      REMOVE_IMAGES=$OPTARG
      ;;
    h)
      help
      ;;
    \?)
      echo "Not valid option: -$OPTARG" >&2
      help
      exit 1
      ;;
    :)
      echo "The -$OPTARG option requires an argument." >&2
      help
      exit 1
      ;;
  esac
done
echo "after check"
echo "${FILES[@]}"

echo "Remove images set to $REMOVE_IMAGES"
if [ $REMOVE_IMAGES == "true" ] ; then
  echo "Removing images generated by docker-compose"
  REMOVE_IMAGES="--rmi all"
else
  echo "Not removing images generated by docker-compose"
  REMOVE_IMAGES=""
fi


for FILE in "${FILES[@]}"; do
  echo "Executing 'docker compose down' for file $FILE"
  REGISTRY_BASE_URL=$REGISTRY_BASE_URL SERVICES_DIR=$SERVICES_DIR OCF_VERSION=$OCF_VERSION CAPIF_PRIV_KEY=$CAPIF_PRIV_KEY_BASE_64 DUID=$DUID DGID=$DGID MONITORING=$MONITORING_STATE LOG_LEVEL=$LOG_LEVEL docker compose -f "$FILE" down $REMOVE_IMAGES
  status=$?
    if [ $status -eq 0 ]; then
        echo "*** Removed Service from $FILE ***"
    else
        echo "*** Some services of $FILE failed on clean ***"
    fi
done

# Path to the register config.yaml file
REGISTER_CONFIG_FILE="$SERVICES_DIR/register/config.yaml"
# Check if the backup config.yaml file exists before restoring
if [ -f "$REGISTER_CONFIG_FILE.bak" ]; then
  git update-index --no-assume-unchanged "$REGISTER_CONFIG_FILE.bak"
  mv "$REGISTER_CONFIG_FILE.bak" "$REGISTER_CONFIG_FILE"
  git update-index --no-assume-unchanged "$REGISTER_CONFIG_FILE"
else
  echo "Backup config file not found, skipping restore."
fi

docker network rm capif-network

docker volume prune --all --force

echo "Clean complete."
