*** Settings ***
Documentation       This resource file contains the basic requests used by Capif. NGINX_HOSTNAME and CAPIF_AUTH can be set as global variables, depends on environment used

Library             RequestsLibrary
Library             Collections
Library             OperatingSystem
Library             XML
Library             Telnet


*** Variables ***
${CAPIF_AUTH}                           ${EMPTY}
${CAPIF_BEARER}                         ${EMPTY}

${LOCATION_INVOKER_RESOURCE_REGEX}
...                                     ^/api-invoker-management/v1/onboardedInvokers/[0-9a-zA-Z]+
${LOCATION_EVENT_RESOURCE_REGEX}
...                                     ^/capif-events/v1/[0-9a-zA-Z]+/subscriptions/[0-9a-zA-Z]+
${LOCATION_INVOKER_RESOURCE_REGEX}
...                                     ^/api-invoker-management/v1/onboardedInvokers/[0-9a-zA-Z]+
${LOCATION_PUBLISH_RESOURCE_REGEX}
...                                     ^/published-apis/v1/[0-9a-zA-Z]+/service-apis/[0-9a-zA-Z]+
${LOCATION_SECURITY_RESOURCE_REGEX}
...                                     ^/capif-security/v1/trustedInvokers/[0-9a-zA-Z]+
${LOCATION_PROVIDER_RESOURCE_REGEX}
...                                     ^/api-provider-management/v1/registrations/[0-9a-zA-Z]+
${LOCATION_LOGGING_RESOURCE_REGEX}
...                                     ^/api-invocation-logs/v1/[0-9a-zA-Z]+/logs/[0-9a-zA-Z]+

${INVOKER_ROLE}                         invoker
${AMF_ROLE}                             amf
${APF_ROLE}                             apf
${AEF_ROLE}                             aef


*** Keywords ***
Create CAPIF Session
    [Documentation]    Create needed session and headers.
    ...    If server input data is set to NONE, it will try to use NGINX_HOSTNAME variable.
    [Arguments]    ${server}=${NONE}    ${access_token}=${NONE}    ${verify}=${NONE}    ${vault_token}=${NONE}

    IF    "${server}" != "${NONE}"
        Create Session    apisession    ${server}    verify=${verify}
    END

    ${headers}=    Create Dictionary
    IF    "${access_token}" != "${NONE}"
        ${headers}=    Create Dictionary    Authorization=Bearer ${access_token}
    END

    IF    "${vault_token}" != "${NONE}"
        ${headers}=    Create Dictionary    X-Vault-Token    ${vault_token}
    END

    RETURN    ${headers}

Create Register Admin Session
    [Documentation]    Create needed session to reach Register as Administrator.
    [Arguments]    ${server}=${NONE}    ${access_token}=${NONE}    ${verify}=${NONE}    ${vault_token}=${NONE}
    IF    "${server}" != "${NONE}"
        IF    "${access_token}" != "${NONE}"
            ## Return Header with bearer
            ${headers}=    Create Dictionary    Authorization=Bearer ${access_token}

            RETURN    ${headers}
        END

        # Request Admin Login to retrieve access token
        Create Session    register_session    ${server}    verify=${verify}    disable_warnings=1

        ${auth}=    Set variable    ${{ ('${REGISTER_ADMIN_USER}','${REGISTER_ADMIN_PASSWORD}') }}
        ${resp}=    POST On Session    register_session    /login    auth=${auth}

        Log Dictionary    ${resp.json()}

        ## Crear sesión con token
        ${headers}=    Create Dictionary    Authorization=Bearer ${resp.json()['access_token']}

        RETURN    ${headers}
    END

    RETURN    ${NONE}

## NEW REQUESTS TO REGISTER

Post Request Admin Register
    [Timeout]    60s
    [Arguments]
    ...    ${endpoint}
    ...    ${json}=${NONE}
    ...    ${server}=${NONE}
    ...    ${access_token}=${NONE}
    ...    ${auth}=${NONE}
    ...    ${verify}=${FALSE}
    ...    ${cert}=${NONE}
    ...    ${username}=${NONE}
    ...    ${data}=${NONE}

    ${headers}=    Create Register Admin Session    ${server}    ${access_token}    ${verify}

    IF    '${username}' != '${NONE}'
        ${cert}=    Set variable    ${{ ('${username}.crt','${username}.key') }}
    END

    ${resp}=    POST On Session
    ...    register_session
    ...    ${endpoint}
    ...    headers=${headers}
    ...    json=${json}
    ...    expected_status=any
    ...    verify=${verify}
    ...    cert=${cert}
    ...    data=${data}
    RETURN    ${resp}

Get Request Admin Register
    [Timeout]    60s
    [Arguments]
    ...    ${endpoint}
    ...    ${server}=${NONE}
    ...    ${access_token}=${NONE}
    ...    ${auth}=${NONE}
    ...    ${verify}=${FALSE}
    ...    ${cert}=${NONE}
    ...    ${username}=${NONE}

    ${headers}=    Create Register Admin Session    ${server}    ${access_token}

    IF    '${username}' != '${NONE}'
        ${cert}=    Set variable    ${{ ('${username}.crt','${username}.key') }}
    END

    ${resp}=    GET On Session
    ...    register_session
    ...    ${endpoint}
    ...    headers=${headers}
    ...    expected_status=any
    ...    verify=${verify}
    ...    cert=${cert}
    RETURN    ${resp}

# NEW REQUESTS END

Post Request Capif
    [Timeout]    60s
    [Arguments]
    ...    ${endpoint}
    ...    ${json}=${NONE}
    ...    ${server}=${NONE}
    ...    ${access_token}=${NONE}
    ...    ${auth}=${NONE}
    ...    ${verify}=${FALSE}
    ...    ${cert}=${NONE}
    ...    ${username}=${NONE}
    ...    ${data}=${NONE}

    ${headers}=    Create CAPIF Session    ${server}    ${access_token}    ${verify}

    IF    '${username}' != '${NONE}'
        ${cert}=    Set variable    ${{ ('${username}.crt','${username}.key') }}
    END

    ${resp}=    POST On Session
    ...    apisession
    ...    ${endpoint}
    ...    headers=${headers}
    ...    json=${json}
    ...    expected_status=any
    ...    verify=${verify}
    ...    cert=${cert}
    ...    data=${data}
    RETURN    ${resp}

Get Request Capif
    [Timeout]    60s
    [Arguments]
    ...    ${endpoint}
    ...    ${server}=${NONE}
    ...    ${access_token}=${NONE}
    ...    ${auth}=${NONE}
    ...    ${verify}=${FALSE}
    ...    ${cert}=${NONE}
    ...    ${username}=${NONE}

    ${headers}=    Create CAPIF Session    ${server}    ${access_token}

    IF    '${username}' != '${NONE}'
        ${cert}=    Set variable    ${{ ('${username}.crt','${username}.key') }}
    END

    ${resp}=    GET On Session
    ...    apisession
    ...    ${endpoint}
    ...    headers=${headers}
    ...    expected_status=any
    ...    verify=${verify}
    ...    cert=${cert}
    RETURN    ${resp}

Get CA Vault
    [Timeout]    60s
    [Arguments]
    ...    ${endpoint}
    ...    ${server}=${NONE}
    ...    ${access_token}=${NONE}
    ...    ${auth}=${NONE}
    ...    ${verify}=${FALSE}
    ...    ${cert}=${NONE}
    ...    ${username}=${NONE}

    ${headers}=    Create CAPIF Session    ${server}    ${access_token}    vault_token=${CAPIF_VAULT_TOKEN}

    IF    '${username}' != '${NONE}'
        ${cert}=    Set variable    ${{ ('${username}.crt','${username}.key') }}
    END

    ${resp}=    GET On Session
    ...    apisession
    ...    ${endpoint}
    ...    headers=${headers}
    ...    expected_status=any
    ...    verify=${verify}
    ...    cert=${cert}
    RETURN    ${resp}
    RETURN    ${response}

Put Request Capif
    [Timeout]    60s
    [Arguments]
    ...    ${endpoint}
    ...    ${json}=${NONE}
    ...    ${server}=${NONE}
    ...    ${access_token}=${NONE}
    ...    ${auth}=${NONE}
    ...    ${verify}=${FALSE}
    ...    ${cert}=${NONE}
    ...    ${username}=${NONE}

    ${headers}=    Create CAPIF Session    ${server}    ${access_token}

    IF    '${username}' != '${NONE}'
        ${cert}=    Set variable    ${{ ('${username}.crt','${username}.key') }}
    END

    ${resp}=    PUT On Session
    ...    apisession
    ...    ${endpoint}
    ...    headers=${headers}
    ...    json=${json}
    ...    expected_status=any
    ...    verify=${verify}
    ...    cert=${cert}

    RETURN    ${resp}

Patch Request Capif
    [Timeout]    60s
    [Arguments]
    ...    ${endpoint}
    ...    ${json}=${NONE}
    ...    ${server}=${NONE}
    ...    ${access_token}=${NONE}
    ...    ${auth}=${NONE}
    ...    ${verify}=${FALSE}
    ...    ${cert}=${NONE}
    ...    ${username}=${NONE}

    ${headers}=    Create CAPIF Session    ${server}    ${access_token}

    IF    '${username}' != '${NONE}'
        ${cert}=    Set variable    ${{ ('${username}.crt','${username}.key') }}
    END

    ${resp}=    PATCH On Session
    ...    apisession
    ...    ${endpoint}
    ...    headers=${headers}
    ...    json=${json}
    ...    expected_status=any
    ...    verify=${verify}
    ...    cert=${cert}

    RETURN    ${resp}

Delete Request Capif
    [Timeout]    60s
    [Arguments]
    ...    ${endpoint}
    ...    ${server}=${NONE}
    ...    ${access_token}=${NONE}
    ...    ${auth}=${NONE}
    ...    ${verify}=${FALSE}
    ...    ${cert}=${NONE}
    ...    ${username}=${NONE}

    ${headers}=    Create CAPIF Session    ${server}    ${access_token}

    IF    '${username}' != '${NONE}'
        ${cert}=    Set variable    ${{ ('${username}.crt','${username}.key') }}
    END

    ${resp}=    DELETE On Session
    ...    apisession
    ...    ${endpoint}
    ...    headers=${headers}
    ...    expected_status=any
    ...    verify=${verify}
    ...    cert=${cert}

    RETURN    ${resp}

Register User At Jwt Auth
    [Arguments]    ${username}    ${role}    ${password}=password    ${description}=Testing

    ${cn}=    Set Variable    ${username}
    # Create certificate and private_key for this machine.
    IF    "${role}" == "${INVOKER_ROLE}"
        ${cn}=    Set Variable    invoker
        ${csr_request}=    Create User Csr    ${username}    ${cn}
        Log    inside if cn=${cn}
    ELSE
        ${csr_request}=    Set Variable    ${None}
    END

    Log    cn=${cn}

    ${resp}=    Create User At Register
    ...    ${username}
    ...    ${password}
    ...    ${description}
    ...    email="${username}@nobody.com"

    ${get_auth_response}=    Get Auth For User    ${username}    ${password}

    Log Dictionary    ${get_auth_response}

    ${register_user_info}=    Create Dictionary
    ...    netappID=${resp.json()['uuid']}
    ...    csr_request=${csr_request}
    ...    &{resp.json()}
    ...    &{get_auth_response}

    Log Dictionary    ${register_user_info}

    IF    "ca_root" in @{register_user_info.keys()}
        Store In File    ca.crt    ${register_user_info['ca_root']}
    END

    IF    "cert" in @{register_user_info.keys()}
        Store In File    ${username}.crt    ${register_user_info['cert']}
    END
    IF    "private_key" in @{register_user_info.keys()}
        Store In File    ${username}.key    ${register_user_info['private_key']}
    END

    Call Method    ${CAPIF_USERS}    update_register_users    ${register_user_info['uuid']}    ${username}

    RETURN    ${register_user_info}

Register User At Jwt Auth Provider
    [Arguments]    ${username}    ${role}    ${password}=password    ${description}=Testing

    ${apf_username}=    Set Variable    APF_${username}
    ${aef_username}=    Set Variable    AEF_${username}
    ${amf_username}=    Set Variable    AMF_${username}

    # Create a certificate for each kind of role under provider
    ${csr_request}=    Create User Csr    ${username}    provider

    ${apf_csr_request}=    Create User Csr    ${apf_username}    apf
    ${aef_csr_request}=    Create User Csr    ${aef_username}    aef
    ${amf_csr_request}=    Create User Csr    ${amf_username}    amf

    # Register provider
    ${resp}=    Create User At Register
    ...    ${username}
    ...    ${password}
    ...    ${description}
    ...    email="${username}@nobody.com"

    ${get_auth_response}=    Get Auth For User    ${username}    ${password}

    Log Dictionary    ${get_auth_response}

    ${register_user_info}=    Create Dictionary
    ...    netappID=${resp.json()['uuid']}
    ...    csr_request=${csr_request}
    ...    apf_csr_request=${apf_csr_request}
    ...    aef_csr_request=${aef_csr_request}
    ...    amf_csr_request=${amf_csr_request}
    ...    apf_username=${apf_username}
    ...    aef_username=${aef_username}
    ...    amf_username=${amf_username}
    ...    &{resp.json()}
    ...    &{get_auth_response}

    Log Dictionary    ${register_user_info}

    Call Method    ${CAPIF_USERS}    update_register_users    ${register_user_info['uuid']}    ${username}

    RETURN    ${register_user_info}

Login Register Admin
    ${headers}=    Create Register Admin Session    ${CAPIF_HTTPS_REGISTER_URL}
    RETURN    ${headers}

Create User At Register
    [Documentation]    (Administrator) This Keyword create a user at register component.
    [Arguments]    ${username}    ${password}    ${description}    ${email}

    # Obtain Admin Token to request creation of User
    ${headers}=    Login Register Admin

    &{body}=    Create Dictionary
    ...    username=${username}
    ...    password=${password}
    ...    description=${description}
    ...    email=${email}
    ${resp}=    Post On Session    register_session    /createUser    headers=${headers}    json=${body}
    Should Be Equal As Strings    ${resp.status_code}    201

    RETURN    ${resp}

Delete User At Register
    [Documentation]    (Administrator) This Keyword delete a user from register.
    [Arguments]    ${username}=${NONE}    ${uuid}=${NONE}
    ${user_uuid}=    Set Variable    ${uuid}

    IF    "${username}" != "${NONE}"
        ${user_uuid}=    Call Method    ${CAPIF_USERS}    get_user_uuid    ${username}
    END

    ${headers}=    Create Register Admin Session    ${CAPIF_HTTPS_REGISTER_URL}    verify=False

    ${resp}=    DELETE On Session    register_session    /deleteUser/${user_uuid}    headers=${headers}

    Should Be Equal As Strings    ${resp.status_code}    204

    Call Method    ${CAPIF_USERS}    remove_register_users_entry    ${user_uuid}

    RETURN    ${resp}

Get List of User At Register
    [Documentation]    (Administrator) This Keyword retrieve a list of users from register.
    ${headers}=    Create Register Admin Session    ${CAPIF_HTTPS_REGISTER_URL}    verify=False

    ${resp}=    DELETE On Session    register_session    /getUsers    headers=${headers}

    Should Be Equal As Strings    ${resp.status_code}    200

    RETURN    ${resp.json()['users']}

Get Auth For User
    [Documentation]    (User) This Keyword retrieve token to be used by user towards first interaction with CCF.
    [Arguments]    ${username}    ${password}
    ${auth}=    Set variable    ${{ ('${username}','${password}') }}
    ${resp}=    GET On Session    register_session    /getauth    auth=${auth}

    Should Be Equal As Strings    ${resp.status_code}    200
    Log Dictionary    ${resp.json()}

    RETURN    ${resp.json()}

Clean Test Information
    ${capif_users_dict}=    Call Method    ${CAPIF_USERS}    get_capif_users_dict

    ${register_users_dict}=    Call Method    ${CAPIF_USERS}    get_register_users_dict

    ${keys}=    Get Dictionary Keys    ${capif_users_dict}

    FOR    ${key}    IN    @{keys}
        ${value}=    Get From Dictionary    ${capif_users_dict}    ${key}
        ${resp}=    Delete Request Capif
        ...    ${key}
        ...    server=${CAPIF_HTTPS_URL}
        ...    verify=ca.crt
        ...    username=${value}

        Status Should Be    204    ${resp}

        Call Method    ${CAPIF_USERS}    remove_capif_users_entry    ${key}
    END

    ${uuids}=    Get Dictionary Keys    ${register_users_dict}
    FOR    ${uuid}    IN    @{uuids}
        Delete User At Register    uuid=${uuid}
    END

Remove entity
    [Arguments]    ${entity_user}    ${certificate_name}=${entity_user}

    ${capif_users_dict}=    Call Method    ${CAPIF_USERS}    get_capif_users_dict

    ${register_users_dict}=    Call Method    ${CAPIF_USERS}    get_register_users_dict

    Log Dictionary    ${capif_users_dict}
    Log Dictionary    ${register_users_dict}
    ${keys}=    Get Dictionary Keys    ${capif_users_dict}

    FOR    ${key}    IN    @{keys}
        ${value}=    Get From Dictionary    ${capif_users_dict}    ${key}
        IF    "${value}" == "${certificate_name}"
            ${resp}=    Delete Request Capif
            ...    ${key}
            ...    server=${CAPIF_HTTPS_URL}
            ...    verify=ca.crt
            ...    username=${value}

            Status Should Be    204    ${resp}

            Call Method    ${CAPIF_USERS}    remove_capif_users_entry    ${key}
        END
    END

    Delete User At Register    username=${entity_user}

    Log Dictionary    ${capif_users_dict}
    Log Dictionary    ${register_users_dict}

Remove Resource
    [Arguments]    ${resource_url}    ${management_cert}    ${username}

    ${resp}=    Delete Request Capif
    ...    ${resource_url}
    ...    server=${CAPIF_HTTPS_URL}
    ...    verify=ca.crt
    ...    username=${management_cert}

    Status Should Be    204    ${resp}

    &{body}=    Create Dictionary
    ...    password=password
    ...    username=${username}

    Create Session    jwtsession    ${CAPIF_HTTPS_REGISTER_URL}    verify=False    disable_warnings=1

    ${resp}=    DELETE On Session    jwtsession    /remove    json=${body}

    Should Be Equal As Strings    ${resp.status_code}    204

Invoker Default Onboarding
    [Arguments]    ${invoker_username}=${INVOKER_USERNAME}
    ${register_user_info}=    Register User At Jwt Auth
    ...    username=${invoker_username}    role=${INVOKER_ROLE}

    # Send Onboarding Request
    ${request_body}=    Create Onboarding Notification Body
    ...    http://${CAPIF_CALLBACK_IP}:${CAPIF_CALLBACK_PORT}/netapp_callback
    ...    ${register_user_info['csr_request']}
    ...    ${invoker_username}
    ${resp}=    Post Request Capif
    ...    ${register_user_info['ccf_onboarding_url']}
    ...    json=${request_body}
    ...    server=${CAPIF_HTTPS_URL}
    ...    verify=ca.crt
    ...    access_token=${register_user_info['access_token']}

    Set To Dictionary    ${register_user_info}    api_invoker_id=${resp.json()['apiInvokerId']}
    Log Dictionary    ${register_user_info}

    # Assertions
    Status Should Be    201    ${resp}
    Check Variable    ${resp.json()}    APIInvokerEnrolmentDetails
    ${resource_url}=    Check Location Header    ${resp}    ${LOCATION_INVOKER_RESOURCE_REGEX}
    # Store dummy signede certificate
    Store In File    ${invoker_username}.crt    ${resp.json()['onboardingInformation']['apiInvokerCertificate']}

    ${url}=    Parse Url    ${resp.headers['Location']}
    Call Method    ${CAPIF_USERS}    update_capif_users_dicts    ${url.path}    ${invoker_username}

    Set To Dictionary    ${register_user_info}    resource_url=${resource_url}
    Set To Dictionary    ${register_user_info}    management_cert=${invoker_username}

    RETURN    ${register_user_info}    ${url}    ${request_body}

Provider Registration
    [Arguments]    ${register_user_info}

    # Create provider Registration Body
    ${apf_func_details}=    Create Api Provider Function Details
    ...    ${register_user_info['apf_username']}
    ...    ${register_user_info['apf_csr_request']}
    ...    APF
    ${aef_func_details}=    Create Api Provider Function Details
    ...    ${register_user_info['aef_username']}
    ...    ${register_user_info['aef_csr_request']}
    ...    AEF
    ${amf_func_details}=    Create Api Provider Function Details
    ...    ${register_user_info['amf_username']}
    ...    ${register_user_info['amf_csr_request']}
    ...    AMF
    ${api_prov_funcs}=    Create List    ${apf_func_details}    ${aef_func_details}    ${amf_func_details}

    ${request_body}=    Create Api Provider Enrolment Details Body
    ...    ${register_user_info['access_token']}
    ...    ${api_prov_funcs}

    # Register Provider
    ${resp}=    Post Request Capif
    ...    /api-provider-management/v1/registrations
    ...    json=${request_body}
    ...    server=${CAPIF_HTTPS_URL}
    ...    verify=ca.crt
    ...    access_token=${register_user_info['access_token']}

    # Check Results
    Check Response Variable Type And Values    ${resp}    201    APIProviderEnrolmentDetails
    ${resource_url}=    Check Location Header    ${resp}    ${LOCATION_PROVIDER_RESOURCE_REGEX}

    Log Dictionary    ${resp.json()}

    FOR    ${prov}    IN    @{resp.json()['apiProvFuncs']}
        Log Dictionary    ${prov}
        Store In File    ${prov['apiProvFuncInfo']}.crt    ${prov['regInfo']['apiProvCert']}
        IF    "${prov['apiProvFuncRole']}" == "APF"
            Set To Dictionary    ${register_user_info}    apf_id=${prov['apiProvFuncId']}
        ELSE IF    "${prov['apiProvFuncRole']}" == "AEF"
            Set To Dictionary    ${register_user_info}    aef_id=${prov['apiProvFuncId']}
        ELSE IF    "${prov['apiProvFuncRole']}" == "AMF"
            Set To Dictionary    ${register_user_info}    amf_id=${prov['apiProvFuncId']}
        ELSE
            Fail    "${prov['apiProvFuncRole']} is not valid role"
        END
    END

    Set To Dictionary
    ...    ${register_user_info}
    ...    provider_enrollment_details=${request_body}
    ...    resource_url=${resource_url}
    ...    provider_register_response=${resp}
    ...    management_cert=${register_user_info['amf_username']}

    Call Method
    ...    ${CAPIF_USERS}
    ...    update_capif_users_dicts
    ...    ${register_user_info['resource_url'].path}
    ...    ${register_user_info['amf_username']}

    RETURN    ${register_user_info}

Provider Default Registration
    [Arguments]    ${provider_username}=${PROVIDER_USERNAME}
    # Register Provider
    ${register_user_info}=    Register User At Jwt Auth Provider
    ...    username=${provider_username}    role=${PROVIDER_ROLE}

    ${register_user_info}=    Provider Registration    ${register_user_info}

    Log Dictionary    ${register_user_info}

    RETURN    ${register_user_info}

Publish Service Api
    [Arguments]    ${register_user_info_provider}    ${service_name}=service_1

    ${request_body}=    Create Service Api Description    ${service_name}    ${register_user_info_provider['aef_id']}
    ${resp}=    Post Request Capif
    ...    /published-apis/v1/${register_user_info_provider['apf_id']}/service-apis
    ...    json=${request_body}
    ...    server=${CAPIF_HTTPS_URL}
    ...    verify=ca.crt
    ...    username=${register_user_info_provider['apf_username']}

    Check Response Variable Type And Values    ${resp}    201    ServiceAPIDescription
    Dictionary Should Contain Key    ${resp.json()}    apiId
    ${resource_url}=    Check Location Header    ${resp}    ${LOCATION_PUBLISH_RESOURCE_REGEX}

    RETURN    ${resp.json()}    ${resource_url}    ${request_body}

Basic ACL registration
    [Arguments]    ${create_security_context}=${True}
    # Register APF
    ${register_user_info_provider}=    Provider Default Registration

    ${service_api_description_published}    ${resource_url}    ${request_body}=    Publish Service Api
    ...    ${register_user_info_provider}
    ...    service_1

    # Store apiId1
    ${serviceApiId}=    Set Variable    ${service_api_description_published['apiId']}

    # Retrieve Services 1
    ${resp}=    Get Request Capif
    ...    /published-apis/v1/${register_user_info_provider['apf_id']}/service-apis/${serviceApiId}
    ...    server=${CAPIF_HTTPS_URL}
    ...    verify=ca.crt
    ...    username=${APF_PROVIDER_USERNAME}

    Check Response Variable Type And Values    ${resp}    200    ServiceAPIDescription
    Dictionaries Should Be Equal    ${resp.json()}    ${service_api_description_published}

    # Default Invoker Registration and Onboarding
    ${register_user_info_invoker}    ${url}    ${request_body}=    Invoker Default Onboarding

    Call Method    ${CAPIF_USERS}    update_capif_users_dicts    ${url.path}    ${INVOKER_USERNAME}

    # Test
    ${discover_response}=    Get Request Capif
    ...    ${DISCOVER_URL}${register_user_info_invoker['api_invoker_id']}&aef-id=${register_user_info_provider['aef_id']}
    ...    server=${CAPIF_HTTPS_URL}
    ...    verify=ca.crt
    ...    username=${INVOKER_USERNAME}

    Check Response Variable Type And Values    ${discover_response}    200    DiscoveredAPIs

    IF    ${create_security_context} == ${True}
        # create Security Context
        ${request_body}=    Create Service Security From Discover Response
        ...    http://${CAPIF_HOSTNAME}:${CAPIF_HTTP_PORT}/test
        ...    ${discover_response}
        ${resp}=    Put Request Capif
        ...    /capif-security/v1/trustedInvokers/${register_user_info_invoker['api_invoker_id']}
        ...    json=${request_body}
        ...    server=${CAPIF_HTTPS_URL}
        ...    verify=ca.crt
        ...    username=${INVOKER_USERNAME}
        # Check Service Security
        Check Response Variable Type And Values    ${resp}    201    ServiceSecurity
        ${resource_url}=    Check Location Header    ${resp}    ${LOCATION_SECURITY_RESOURCE_REGEX}
    END

    RETURN    ${register_user_info_invoker}    ${register_user_info_provider}    ${service_api_description_published}

Create Security Context Between invoker and provider
    [Arguments]    ${register_user_info_invoker}    ${register_user_info_provider}

    ${discover_response}=    Get Request Capif
    ...    ${DISCOVER_URL}${register_user_info_invoker['api_invoker_id']}&aef-id=${register_user_info_provider['aef_id']}
    ...    server=${CAPIF_HTTPS_URL}
    ...    verify=ca.crt
    ...    username=${register_user_info_invoker['management_cert']}

    Check Response Variable Type And Values    ${discover_response}    200    DiscoveredAPIs

    # create Security Context
    ${request_body}=    Create Service Security From Discover Response
    ...    http://${CAPIF_HOSTNAME}:${CAPIF_HTTP_PORT}/test
    ...    ${discover_response}

    ${resp}=    Put Request Capif
    ...    /capif-security/v1/trustedInvokers/${register_user_info_invoker['api_invoker_id']}
    ...    json=${request_body}
    ...    server=${CAPIF_HTTPS_URL}
    ...    verify=ca.crt
    ...    username=${register_user_info_invoker['management_cert']}

    Check Response Variable Type And Values    ${resp}    201    ServiceSecurity


Create Events From invocationLogs
    [Arguments]    ${subscription_id}    ${invocation_log}

    ${events}=    Create List

    # Now we create the expected events received at notification server according to logs sent to loggin service in order to check if all are present.
    ${invocation_log_base}=    Copy Dictionary    ${invocation_log}    deepcopy=True
    # Store log array because each log will be notified in one Event Notification
    ${invocation_log_logs}=    Copy List    ${invocation_log_base['logs']}
    # Remove logs array from invocationLog data
    Remove From Dictionary    ${invocation_log_base}    logs

    FOR    ${log}    IN    @{invocation_log_logs}
        Log Dictionary    ${log}
        ${invocation_logs}=    Copy Dictionary    ${invocation_log_base}    deepcopy=True

        # Get Event Enum for this result
        ${event_enum}=    Set Variable
        IF    ${log['result']} >= 200 and ${log['result']} < 300
            ${event_enum}=    Set Variable    SERVICE_API_INVOCATION_SUCCESS
        ELSE
            ${event_enum}=    Set Variable    SERVICE_API_INVOCATION_FAILURE
        END
        # Create a log array with only one component
        ${log_list}=    Create List    ${log}
        # Setup logs array with previously created list
        Set To Dictionary    ${invocation_logs}    logs=${log_list}
        # Create event details for each log
        ${event_details}=    Create dictionary    invocationLogs=${invocation_logs}
        # Create Event with Event Details from invocationLog and also is appended to events array
        ${event}=    Create Dictionary
        ...    subscriptionId=${subscription_id}
        ...    events=${event_enum}
        ...    eventDetail=${event_details}
        Check Variable    ${event}    EventNotification
        Append To List    ${events}    ${event}
    END

    Log List    ${events}
    ${events_length}=    Get Length    ${events}
    RETURN    ${events}    ${events_length}
