# Terms and References

TTCN-3 entities
: Distinct language elements that make up the test suite.
: Such as modules, variables, types, fields, behaviouris, attributes, etc.

Type
: The set of possible values that an expression can have.

Base type
: The base type is the type from which the current type directly inherits.

Subtype
: A subtype is a type that is derived from another type.

Union type
: A union type is a type that can hold values of different types, called
: variants, members or alternatives.  

Sum type
: See [Union type]

Variant type
: See [Union type]

# Lexical structure

## Source code representation

TTCN-3 source code is represented as a sequence of Unicode code points. The
Unicode code points are encoded using the UTF-8 encoding scheme.

> *Note*: The handling of the Unicode NULL character (U+0000) is
> implementation-specific. It is strongly recommended to avoid the use of this
> character in TTCN-3 source code.

The term _character_ will be used interchangeably with _Unicode code point_ in
this document.

## Syntax notation

The syntax is specified using a
[variant](https://en.wikipedia.org/wiki/Wirth_syntax_notation) of Extended
Backus-Naur Form (EBNF):

```ebnf
Syntax      = { Production } .
Production  = production_name "=" [ Expression ] "." .
Expression  = Term { "|" Term } .
Term        = Factor { Factor } .
Factor      = production_name | token [ "…" token ] | Group | Option | Repetition .
Group       = "(" Expression ")" .
Option      = "[" Expression "]" .
Repetition  = "{" Expression "}" .
```

A production is an expression constructed from terms and the following
operators, in increasing precedence:

```
|   alternation
()  grouping
[]  option (0 or 1 times)
{}  repetition (0 to n times)
```

A CamelCase production name identifies non-terminals. A lowercase production
name identifies terminals. A tokens is enclosed in double quotes (e.g.
<code>"testcase"</code>) or back quotes (e.g. <code>`testcase`</code>).

The form a … b represents the set of characters from a through b as
alternatives. The horizontal ellipsis … is also used elsewhere in this document
to informally denote various enumerations or code snippets that are not further
specified. The character … (as opposed to the three characters `...`) is not a
token of the TTCN-3 language.

The primary purpose of this notation is readability and unambiguity to the
reader. It is not a goal to be parsed by a specific tool or parser
algorithm.


## Lexical elements

### Tokens

Tokens form the vocabulary of the TTCN-3 language. There are several kinds of tokens:
***identifiers***, ***keywords***, ***literals***, ***operators*** and ***punctuation***.

**White space** is ignored, except it may act as separator for tokens that
would otherwise combine into a single token.

When breaking input into tokens, the longest sequence of characters that form a
valid token is the next token. For example, the input `<<<` is tokenized as
`<<` and `<`.


### White space

Valid white space characters are horizontal tab (U+0009), line feed (U+000A), vertical
tab (U+000B), form feed (U+000C), carriage return (U+000D), and space (U+0020).

Following characters denote a single end of line:
- line feed
- vertical tab
- form feed
- carriage return
- carriage return followed by a line feed


### Comments

A ***line comment*** begins with the characters `//` and extends to the end of the line.

```ttcn3 title="line comments"
// This is a comment.
var integer i := 1; // This is another comment.
```

A ***block comment*** begins with the characters `/\*` and ends with the characters `\*/`.

```ttcn3 title="block comments"
/* This is a comment. */
```

A comment cannot start inside of a another token. A block comment containing no line
break acts like a space. Any other comment acts like a line break.

```ttcn3 title="comments and tokens"
"/* this is not a comment */ but a string"
123/* this line contains three tokens: an integer, a comment and a second integer */456
```

Comments may be used to add [documentation or metadata](#documentation-comments) to the source code.


### Operators and punctuation

Following tokens represent operators or punctuation symbols:

```
+   -   *   /   =   <   <=
<<  <@  @>  >>  ..  &   ->  =>
?   ??? "   '   ,   ;
:=  ==  !=  .   ...
[   ]   {   }   (   )
```


**Semicolon**

The formal syntax use semicolon (`;`) as terminator in a number of productions.
A semicolon may be omitted before and after a closing brace (`}`).


**Trailing comma**

A comma-separated list may be terminated by a trailing comma.

```ttcn3
type record R {
    integer a,
    integer b,
    integer c, // <-- trailing comma
}
```

### Identifiers

Identifiers name TTCN-3 entities such as variables, types, modules, etc.

An identifier is a sequence of one or more letters, digits and underscore (`_`). The first
character in an identifier shall be a letter. TTCN-3 implementations are
permitted to use identifiers beginning with underscore for their own purposes.

```ebnf
id = ( "A" … "Z" | "a" … "z" | "_" ) { "A" … "Z" | "a" … "z" | "_" | "0" … "9" }.
```

<!-- TODO: Add reference to predeclared identifiers -->


### Keywords

The following keywords are reserved and may not be used as user-defined identifiers:

<!-- TODO: Add list of keywords -->
<!-- TODO: Some can be used as identifiers: testcase, class, ... -->


### Constant literals

```ebnf
booleanLiteral = "true" | "false".

```

* numerical: `infinity` and `not_a_number`
* boolean: `true` and `false`
* verdict: `none`, `pass`, `inconc`, `fail`, `error`
* `null`
* `__MODULE__`, `__FILE__`, `__BFILE__`, `__LINE__`, `__SCOPE__`


### Modifiers

A modifier is a pre-defined, identifier-like sequence that starts with the at sign (`@`).

```ebnf
modifier = "@" id
```

Following modifiers are available:

<!-- TODO: Add list of modifiers -->


### Numeric literals

A numeric literal represents an integer or float value. It consists of an
integer part followed by an optional fractional part and an optional exponent
part.

The integer part is a sequence of decimal digits. The first digit must be
non-zero unless the integer part is `0`.

The fractional part is a decimal point (`.`) followed by a sequence of decimal digits.

The exponent part is an `e` or `E` followed by an optional sign (`+` or `-`) and another integer.

```ebnf
number = int [ fraction ] [ ( "e" | "E" ) [ "+" | "-" ] int ].
int = "0" | "1" … "9" { "0" … "9" }.
fraction = "." "0" … "9" { "0" … "9" }.
```


### String literals

A string literal is a sequence of characters enclosed in double quotes (`"`).
The characters may be any Unicode code point except newline.

The backslash (`\`) is used as an escape character.
Quotes inside the string literal must be escaped by a backslash (`\"`) or a
pair of quotes (`""`).


```ebnf
string = `"` { unicode_char | backslash_escape | quote_escape } `"`.
backslash_escape = "\" ( "a" | "b" | "f" | "n" | "r" | "t" | "v" | "\\" | "'" | `"` ).
quote_escape     = "\"\"".
```


### Binary string literals

A binary string represents binary data. It is a sequence of digits enclosed in single
quotes (`'`) followed by a single letter denoting the base of the digits.

* `B` for binary digits (0 and 1)
* `H` for hexadecimal digits (0-9, A-F)
* `O` for hexadecimal octets (00-FF)

Denotation letters and digits are case-insensitive.

A binary string may also contain whitespace characters and the matching symbols
`?` and `*`.

```ebnf
bitstring   = "'" { bit     | whitespace } "'" ( "B" | "b" ).
hexstring   = "'" { hex     | whitespace } "'" ( "H" | "h" ).
octetstring = "'" { hex hex | whitespace } "'" ( "O" | "o" ).
bit = "0" | "1".
hex = "0" … "9" | "A" … "F" | "a" … "f".
```


### Matching symbols

Matching symbol __any__ (`?` ) is a placeholder for a single element. Matching
symbol __any or none__ (`*`) is a placeholder for zero or more elements.

Any or none matching symbol can be followed by a length restriction.
* ?
* *
* #(1,2)

```ebnf
matching_symbol = "?" | "*".
```

# Types and Values

## Predeclared types

TTCN-3 provides the following predeclared types:

* **`integer`**: This type represents the set of positive and negative whole
  numbers, including zero.
* **`float`**: This type represents the set of floating-point numbers and the
  special float values `not_a_number` and `infinity`.
* **`boolean`**: This type represents the set of boolean values `true` and
  `false`.
* **`verdicttype`**: The type `verdicttype` represents the ordered set of
  verdict values `none`, `pass`, `inconc`, `fail`, and `error`.


## List types

### Character strings


### Binary strings

- Bitstring
- Hexstring
- Octetstring

### Record of

```ebnf
ListType = "type" ("record"|"set") [LengthConstraint] of NestedType Name [ValueConstraint] [LengthConstraint].
```

- set of

## Enumerated


```ebnf
EnumType = "type" "enumerated" Name "{" EnumValue { "," EnumValue } [","] "}".
EnumValue = Name [ "(" ")" ].
```


## Map

```ebnf
MapType = "type" "map" "from" NestedType "to" NestedType Name.
```

## Record

```ebnf
StructType = "type" ("record"|"set") "{" [ StructMember { "," StructMember } [","] ] "}".
StructMember = NestedType Name [ValueConstraint] [LengthConstraint] ["optional"].
```

## Union

```ebnf
UnionType = "type" "union" "{" UnionMember { "," UnionMember } [","] "}".
UnionMember = ["@default"] NestedType [Name]
```

<!-- TODO: Anytype -->


## Any

The open type `any` is a union type that can hold any value.

<!-- TODO: isselected and select type works, too -->

## Template types

A template type is the union of its base type and matching symbols.

```ebnf
TemplateType = "template" ["(" TemplateRestriction ")"] | TemplateRestriction] NestedType.
TemplateRestriction = "omit" | "value" | "present".
```

## Reference types
- Objects/Classes
- Predefined classes:
 - Timer
 - Port
   - address
 - Component
 - Default type
- For timer, component, ... refer to the corresponding sections.

## Behaviour types

```ebnf
BehaviourType = "type" ("function"|"testcase"|"altstep") Name FormalParameters ReturnType.
```


## Subtypes

A subtype definition creates a new, distinct type from a given base type. The
subtype inherits all operations, constraints and attributes of its base type.

```ttcn3 test
// @purpose: subtype inherits operations of base type (addition)
type B A;
type C B;
type integer C;

testcase inherit_addition() {
    var A a := 1;
    if (a + 1 == 2) {
        setverdict(pass);
    }
}
```

<!-- TODO: clarify if methods as inherited as well -->

```ebnf
SubType          = "type" BaseType name [ValueConstraint] [LengthConstraint] [WithAttributes].
BaseType         = PrimaryExpr.
ValueConstraint  = "(" Expr { "," Expr } [","] ")".
LengthConstraint = "length" "(" [ ["!"] lowerBound ".."] ["!"] upperBound ")".
lowerBound       = integer.
upperBound       = integer | "infinity".
```

A subtype definition may also specifiy a set of templates and type references.

<!-- TODO: base type constraints are intersections (AND) -->
<!-- TODO: subtype constraints are unions (OR) -->
<!-- TODO: "a".."z" is a shorthand notation for charstring template -->
<!-- TODO: length-constraint is a shorthand notation for ? length() -->
<!-- TODO: provide test with reference type base types (and constraints) -->
<!-- TODO: add note about recommendation that empty types provoke a warning/error -->
<!-- TODO: add reference how attributes behave with subtypes -->

## Identity

A named type is only equal to itself.

Two types are equal if they have the same full qualified name.

- behaviours?
- renamed imports?
- anytype & co.

```
type record R {
    m2.A a,
}

type record S {
    R.a a,
}

typeof(S.a) == typeof(R.a) // true or false?
```



```ttcn3 title="Type equality"
module m1 {
    import from m1 all;
    import from m1 -> m3 all;
    import from m3 all;
}

module m2 {
    type integer A;
    type boolean B;
    type integer C;
}

module m3 {
    type integer A;
    type boolean B;
}

```

- types
- values



## Type conversion
- no implicit
- conversion to incompatible types only via predefined functions (int32 -> int8)
- conversion to compatible types automatic via assignment or via `integer:1` (for strong typing)



## Value assignment

- structural compatibility (type compatibility)

- value types vs. reference types

- implicit assignments (parameters, foreach, ...)
- Addressability

## Method sets

# Uniqueness of identifiers

# Variable Declarations and constants

A _variable_ binds a name to a value.

```ebnf
VarDecl = ("var" | "const" | "modulepar" ) TemplateType Declarator { "," Declarator }.
Declarator = Name [ ":=" Expr ].
```

<!-- TODO: partially defined -->
<!-- TODO: permissiable values -->

# Behaviours

```ebnf
BehaviourDef = ExternalFunctionDef|FunctionDef|TestcaseDef|AltstepDef.
ExternalFunctionDef = "external" "function" Signature.
FunctionDef = "function" Signature Body.
TestcaseDef = "testcase" Signature Body.
AltstepDef = "altstep" Signature Body.
Signature = Name FormalParameters [ConfigurationClause] [ReturnType].
ConfigurationClause = "runs" "on" Reference ["system" Reference] ["mtc" Reference].

```

# Expressions

## General

An expression represents the computation of a value.

<!-- TODO: infix, operand, operator, precedence, associativity, primary expression, ... -->

## Primary Expressions

- literals
- references (retrieving the value specified by the reference)

```ebnf
PrimaryExpr = id.
SelectExpr  = PrimaryExpr "." Name.
IndexExpr   = PrimaryExpr "[" Expr "]".
CallExpr = PrimaryExpr "(" [ Expr { "," Expr } [","] ")".

```

## Composite Literal

```ebnf
CompositeLiteral = "{" Element { "," Element } [","] "}".
Element = Expr
        | Name ":=" Expr
        | "[" Expr "]" ":=" Expr
        .
```

## Function Literal

```ebnf
FunctionLiteral = "function" FormalParameters Block.
```

## Numeric operators

+ - * / mod rem

## Concatenation

&

## Logical operators

not and or xor

## Comparison operators

== !=

## Relational operators

< <= > >=

## Bitwise operators

not4b and4b or4b xor4b

## Shift operators

<< <@ @> >>

## Decode

=>

## Value selection

-> value


# Statements

## General

```ebnf
Statement = .
SimpleStmt = IncrementStmt | DecrementStmt | Assignment | VarDecl | ExprStmt.
```


## Expression statements

```ebnf
ExprStmt = Reference.
```

## Assignments

```ebnf
Assignment = Reference ":=" Expr.
IncrementStmt = Reference "++".
DecrementStmt = Reference "--".
```

## Initialisation statements

`if`, `select`, `while` and `for` statements may specify an initialisation
statement, which is executed before the condition is evaluated. This
initialisation statement shall be a simple statement. The scope of variables
declared within this initialisation statement shall be limited to the
statement.

```ebnf
InitStmt = SimpleStmt.
```

The following two examples are semantically equivalent:

```ttcn3 example
if (var integer i := rand(); i < 0.5) {
    // ...
} else {
    // ...
}
```

```ttcn3 example
{
    var integer i := rand();
    if (i < 0.5) {
        // ...
    } else {
        // ...
    }
}
```


## Statement block

## Exceptions
## Conditional statements

### `if`

The `if` statement executes a block based on the evaluation of a condition.


```ebnf
IfStmt = "if" ([ InitStmt ";" ] Epxr) Block [ "else" ( IfStmt | Block ) ].
```

The condition shall be an boolean expression.

If the condition yields `true`, the first block is executed. When and if execution reaches an end point, the program continues with the statement following the `if` statement.

If the condition yields `false`, the else part is executed if present.

The following two examples are semantically equivalent:

```ttcn3 example
if (cond1) {
    // consequence 1
} else if (cond2) {
    // consequence 2
} else {
    // alternative
}
```

```ttcn3 example
if (cond1) {
    // consequence 1
} else {
    if (cond2) {
        // consequence 2
    } else {
        // alternative
    }
}
```

### `select` statements

The `select` statement evaluates a condition once and compares the result to a
set of cases. The first case that matches the condition is executed.

The condition may be preceded by a simple statement, which executes before the
condition is evaluated. The scope of variables declared within that simple
statement shall be limited to the select-statement.

```ebnf
SelectStmt = "select" ["union"|"type"|"match"] "(" [InitStmt ";" ] Expr ")" "{" { SelectCase [CaseElse]} "}".
SelectCase = "case" "(" Expr ")" Block.
CaseElse  =  ["case" "else" Block] .
```

The condition is an expression that evaluates to a boolean value. It is
evaluated only once.

<!-- TODO: `select union` -->
<!-- TODO: `select type` -->
<!-- TODO: `select match` -->


### For statement


```ebnf
ForStmt = "for" "(" ForClause | RangeClause ")" Block.
ForClause = [InitStmt] ";" [Expr] ";" [SimpleStmt].
RangeClause = [InitStmt] "in" Expr.
```

### While statement

The `while` statement executes a block based on the evaluation of its condition zero or more times.

```ebnf
WhileStmt = "while" "(" [ InitStmt ";" ] Expr ")" Block.
```

The condition shall be an boolean expression. If the condition yields `true`,
the block is executed. When and if execution reaches an end point, the program
continues with the statement following the `while` statement.

### Do-while statement

The `do`-`while` statement executes a block based on the evaluation of its condition one or more times.

```ebnf
DoWhileStmt = "do" Block "while" "(" Expr ")".
```

The condition shall be an boolean expression.

The code block is executed. When and if execution reaches an end point, the condition is evaluated. If the condition yields `true`, the block is executed again. If the condition yields `false`, the program continues with the statement following the `do`-`while` statement.

### Break statement

The `break` statement terminates the execution of the innermost loop, `alt` or `interleave` statement.

```ebnf
BreakStmt = "break".
```

A `break` statement that is not within a loop shall cause an error.

### Continue statement

The `continue` statement terminates the current iteration of the innermost loop and continues with the next iteration.

```ebnf
ContinueStmt = "continue".
```

A `continue` statement that is not within a loop shall cause an error.

## `repeat`

The `repeat` statement re-evaluates the innermost `alt` statement.

```ebnf
RepeatStmt = "repeat".
```

## `goto` statement

The `goto` statement transfers control to a statement with a matching label
inside the same function.

```ebnf
GotoStmt = "goto" Label.
Label = id.
LabeledStmt = Label ":" (Block|ForStmt|WhileStmt|DoWhileStmt|SelectStmt|IfStmt|AltStmt|InterleaveStmt).
```

The `goto` statement shall not jump into a block.

## Return statement

A `return` statement terminates the execution of the current behaviour and
returns control to the point from which the behaviour was called.

An optional expression may be provided to return a value to the caller. The
expression shall be compatible with the return type of the behaviour.

```ebnf
ReturnStmt = "return" [ Expr ].
```

## `alt` and `interleave` statements

See [Matching a state].


# Pattern matching

## Templates

A _template_ is pattern that describes a set of values.

<!-- TODO: Restrictions -->

## Struct-type templates

## List-type templates

{1,2,*,3}
{1,2,?,3}
"12" & * & "3" == pattern "12*3"
'12' & * & '3' == '12*3'H




### Charstring templates



<!-- TODO: Charstring pattern -->
<!-- TODO: Constraints -->

## Matching a state
`alt`/`interleave`

# Timers

```ebnf
```

```ttcn3
type class timer {
    function start(in float duration := -);
    function stop();
    function running();
    function timeout();
    function read() return float;
}
```

<!-- NOTE: start, stop need mandatory parentheses to distinguish from behaviour
type -->

# Components

```ttcn3
type class component {
    function start(function());
    function stop();
    function kill();
    function done();
    function running();
}
```

# Ports

```ebnf
PortType = "type" "port" Name "{" "}".
```

```ttcn3
p.receive() ["from" Expr] -> ...
p.send(Expr) ["to" Expr]
```

# Attributes

The with statement is used to associate attributes to TTCN-3 language elements (and sets thereof).

```ebnf
Attribute         = AttributeKind [AttributeModifier] [AttributeTarget] AttributeValue.
AttributeKind     = "encode" | "variant" | "display" | "extension" | "optional".
AttributeModifier = "override" | "@local".
AttributeTarget   = "(" Reference ")".
AttributeValue    = ( string | "{" string "," { string } [","] "}" ) [ "." string ].
```

## Predefined Values

Following encoding variants are predefined:

`8 bit`
`8 bit`
`unsigned 8 bit`
`16 bit`
`unsigned 16 bit`
`32 bit`
`unsigned 32 bit`
`64 bit`
`unsigned 64 bit`
`IEEE754 float`
`IEEE754 double`
`IEEE754 extended float`
`IEEE754 extended double`
`UTF-8`
`UTF-16`
`UTF-16LE`
`UTF-16BE`
`UTF-32`
`UTF-32LE`
`UTF-32BE`
`IDL:fixed FORMAL/01-12-01 v.2.6`

# Annex: Documentation comments (informative)

A documentation comment is a consecutive block of comment tokens that associates
documentation or metadata with the TTCN-3 entity that follows it.



```ttcn3
// regular comment

// documentation line 1
// documentation line 2
// documentation line 3
module m {}
```



```ttcn3
module M1 {} // regular comment
// documentation line 1
module M3 {}
```

- content
- formatting
- tags
- proposed tags
