#include "LibSecurity_Hash.hh"
#include "LibSecurity_Signature.hh"
#include "LibSecurity_Certificates.hh"

#include "loggers.hh"

#include "sha1.hh"
#include "sha256.hh"
#include "sha384.hh"
#include "certs_db.hh"
#include "security_services.hh"

static std::unique_ptr<security_services> _security_services;

INTEGER LibSecurity__Certificates::fx__init__certs__db(const CHARSTRING& p_certs_db_path) {
  loggers::get_instance().log_msg(">>> fx__init__certs__db: ", p_certs_db_path);

  int ret = _security_services->initialize(std::string(static_cast<const char*>(p_certs_db_path)));

  loggers::get_instance().log("<<< fx__init__certs__db.");
  return ret;
}

INTEGER LibSecurity__Certificates::fx__load__certificate(const CHARSTRING& p_certificate_name, const CHARSTRING& p_private_key_name, const CHARSTRING& p_private_key_passwd) {
  loggers::get_instance().log_msg(">>> fx__load__certificate: ", p_certificate_name);
  loggers::get_instance().log_msg(">>> fx__load__certificate: ", p_private_key_name);

  const X509* certificate;
  int ret = _security_services->load_certificate(std::string(static_cast<const char*>(p_certificate_name)), std::string(static_cast<const char*>(p_private_key_name)), std::string(static_cast<const char*>(p_private_key_passwd)), &certificate);
  loggers::get_instance().log("fx__load__certificate: certificate: '%p'", certificate);
  if (ret == 0) {
    ASN1_INTEGER* asn1_serial = ::X509_get_serialNumber((X509*)certificate);
    if (asn1_serial == nullptr) {
      loggers::get_instance().log("fx__load__certificate: Failed to retrieve X509 serial number");
      return -1;
    }
    uint64_t v;
    ASN1_INTEGER_get_uint64(&v, asn1_serial);
    loggers::get_instance().log("fx__load__certificate: Loaded certificate: serial number: %ld", v);
  }

  loggers::get_instance().log("<<< fx__load__certificate");
  return ret;
}

OCTETSTRING LibSecurity__Hash::fx__hash(const OCTETSTRING& p_to_be_hashed, const LibSecurity__Hash::HashAlgorithm& p_hash_algorithm) {
  loggers::get_instance().log_msg(">>> fx__hash: ", p_to_be_hashed);

  OCTETSTRING hash;
  switch (p_hash_algorithm) {
    case LibSecurity__Hash::HashAlgorithm::e__sha1: {
        sha1 s;
        s.generate(p_to_be_hashed, hash);
      }
      break;
    case LibSecurity__Hash::HashAlgorithm::e__sha256: {
        sha256 s;
        s.generate(p_to_be_hashed, hash);
      }
      break;
    case LibSecurity__Hash::HashAlgorithm::e__sha384: {
        sha384 s;
        s.generate(p_to_be_hashed, hash);
      }
      break;
  } // End of 'switch' statement

  loggers::get_instance().log_msg("<<< fx__hash: ", hash);
  return hash;
}

INTEGER LibSecurity__Signature::fx__sign(const OCTETSTRING& p_encoded_message, const CHARSTRING& p_certificate_name, const CHARSTRING& p_private_key_name, const CHARSTRING& p_private_key_passwd, OCTETSTRING& p_signature, OCTETSTRING& p_digest, CHARSTRING& p_x509_certificate_subject, CHARSTRING& p_x509_certificate_pem, CHARSTRING& p_pull_request_signed_canonicalized) {
  loggers::get_instance().log_msg(">>> fx__sign: ", p_encoded_message);

  if (_security_services->do_sign(p_encoded_message, p_certificate_name, p_private_key_name, p_private_key_passwd, p_signature, p_digest, p_x509_certificate_subject, p_x509_certificate_pem, p_pull_request_signed_canonicalized) == -1) {
    loggers::get_instance().log("fx__sign: Failed to signed message");
    return -1;
  }
  
  return 0;
}

BOOLEAN LibSecurity__Signature::fx__do__sign__verify(const CHARSTRING& p_message, const UNIVERSAL_CHARSTRING& p_canonicalization_method, const UNIVERSAL_CHARSTRING& p_signature_method, const UNIVERSAL_CHARSTRING& p_digest_method, const UNIVERSAL_CHARSTRING& p_digest_value, const UNIVERSAL_CHARSTRING& p_signature_value, const UNIVERSAL_CHARSTRING& p_subject_name, const UNIVERSAL_CHARSTRING& p_certificate, const CHARSTRING& p_debug_message) {
  loggers::get_instance().log(">>> fx__do__sign__verify");

  if (!_security_services->do_sign_verify(p_message, p_canonicalization_method, p_signature_method, p_digest_method, p_digest_value, p_signature_value, p_subject_name, p_certificate, p_debug_message)) {
    loggers::get_instance().log("fx__do__sign__verify: Failed to verify message signature");
    return false;
  }
  
  return true;
}
  