/*!
 * \file      certs_cache.cc
 * \brief     Source file for a certificates caching storage mechanism.
 *            It is used to store certificates received from neighbors and not present in the data base
 * \author    ETSI STF637
 * \copyright ETSI Copyright Notification
 *            No part may be reproduced except as authorized by written permission.
 *            The copyright and the foregoing restriction extend to reproduction in all media.
 *            All rights reserved.
 * \version   0.1
 */
#include "certs_cache.hh"
#include "certs_loader.hh"

#include "TTCN3.hh"
#include "sha1.hh"

#include "converter.hh"

#include "loggers.hh"

certs_cache::certs_cache() : _certificates_idx(), _certificates_subject(), _certificates() { loggers::get_instance().log(">>> certs_cache::certs_cache"); } // End of ctor

certs_cache::~certs_cache() {
  loggers::get_instance().log(">>> certs_cache::~certs_cache");
  clear();
} // End of dtor

int certs_cache::clear() {
  loggers::get_instance().log(">>> certs_cache::clear");
  _certificates.clear(); // Smart pointers will do the job
  _certificates_idx.clear();
  _certificates_subject.clear();

  return 0;
} // End of clear method

int certs_cache::load_certificate(const std::string& p_certificate_id, const certs_db_record** p_record) {
  loggers::get_instance().log(">>> certs_cache::load_certificate (1): '%s'", p_certificate_id.c_str());

  // Sanity check
  std::map<std::string, std::unique_ptr<const certs_db_record>>::const_iterator it = _certificates.find(p_certificate_id);
  if (it == _certificates.cend()) {
    loggers::get_instance().warning("certs_cache::load_certificate: Failed to load certificate");
    return -1;
  }

  *p_record = it->second.get();

  return 0;
}

int certs_cache::load_certificate(const std::string& p_certificate_name, const std::string& p_private_key_name, const std::string& p_private_key_passwd, const certs_db_record** p_record) {
  loggers::get_instance().log(">>> certs_cache::load_certificate (2): '%s'", p_certificate_name.c_str());

  std::string certificate_id;
  if (certs_loader::get_instance().get_certificate_id(p_certificate_name, certificate_id) != 0) {
    loggers::get_instance().warning("certs_cache::load_certificate: Failed to retrieve certificate identifier");
    return -1;  
  }
  loggers::get_instance().log("certs_cache::load_certificate (2): certificate_id: '%s'", certificate_id.c_str());
  
  std::map<std::string, const std::string>::const_iterator s = _certificates_idx.find(certificate_id);
  if (s == _certificates_idx.cend()) { // Certificate is not in the DB
    loggers::get_instance().log("certs_cache::load_certificate (2): Record not found");
    if (certs_loader::get_instance().load_certificate(p_certificate_name, p_private_key_name, p_private_key_passwd, certificate_id, _certificates) == -1) {
      loggers::get_instance().warning("certs_cache::load_certificate: Failed to load certificate");
      return -1;
    }
    // Certificate is on the DB, load it
    std::map<std::string, std::unique_ptr<const certs_db_record>>::const_iterator it = _certificates.find(certificate_id);
    *p_record = it->second.get();
    // Mapping certificate id/certificate name
    _certificates_idx.insert(std::pair<std::string, const std::string>(certificate_id, p_certificate_name));
    std::string sn(256, (char)0x00);
    const X509_NAME* subject = (*p_record)->subject_name();
    ::X509_NAME_oneline(subject, sn.data(), sn.length());
    loggers::get_instance().log("certs_cache::load_certificate: sn: '%s'", sn.c_str());
    _certificates_subject.insert(std::pair<std::string, const std::string>(sn, certificate_id));
  } else {
    // Certificate is on the DB, load it
    std::map<std::string, std::unique_ptr<const certs_db_record>>::const_iterator it = _certificates.find(certificate_id);
    *p_record = it->second.get();
  }

  return 0;
}

int certs_cache::get_certificate(const std::string& p_certificate_name, const std::string& p_private_key_name, const std::string& p_private_key_passwd, const X509** p_certificate) {
  loggers::get_instance().log(">>> certs_cache::get_certificate (1): '%s'", p_certificate_name.c_str());

  const certs_db_record* record;
  if (load_certificate(p_certificate_name, p_private_key_name, p_private_key_passwd, &record) == -1) {
    loggers::get_instance().warning("certs_cache::get_certificate (1): Failed to load certificate");
    return -1;
  }

  *p_certificate = record->certificate();
  loggers::get_instance().log("certs_cache::get_certificate (1): p_certificate: '%p'", *p_certificate);

  return 0;
}

int certs_cache::get_certificate(const std::string& p_certificate_id, const X509** p_certificate) {
  loggers::get_instance().log(">>> certs_cache::get_certificate (2): '%s'", p_certificate_id.c_str());

  const certs_db_record* record;
  if (load_certificate(p_certificate_id, &record) == -1) {
    loggers::get_instance().warning("certs_cache::get_certificate (2): Failed to load certificate");
    return -1;
  }

  *p_certificate = record->certificate();
  loggers::get_instance().log("certs_cache::get_certificate (2): p_certificate: '%p'", *p_certificate);

  return 0;
}

int certs_cache::get_certificate_by_subject_name(const std::string& p_certificate_subject_name, const X509** p_certificate) {
  loggers::get_instance().log(">>> certs_cache::get_certificate_by_subject_name: '%s'", p_certificate_subject_name.c_str());

  std::map<std::string, const std::string>::const_iterator s = _certificates_subject.find(p_certificate_subject_name);
  if (s == _certificates_subject.cend()) { // Certificate is not in the DB
    loggers::get_instance().warning("certs_cache::get_certificate_by_subject_name: Record not found");
    return -1;
  }

  if (get_certificate(s->second, p_certificate) == -1) {
    loggers::get_instance().error("certs_cache::get_certificate_by_subject_name: DB corrupted: '%s'", s->second.c_str());
    return -1;
  }
  loggers::get_instance().log("certs_cache::get_certificate_by_subject_name: p_certificate: '%p'", *p_certificate);

  return 0;
}

int certs_cache::get_private_key(const std::string& p_certificate_id, const EVP_PKEY** p_private_key) {
  loggers::get_instance().log(">>> certs_cache::get_private_key: '%s'", p_certificate_id.c_str());

  const certs_db_record *record;
  if (load_certificate(p_certificate_id, &record) == -1) {
    loggers::get_instance().warning("certs_cache::get_private_key: Failed to load certificate");
    return -1;
  }

  *p_private_key = record->private_key();
  loggers::get_instance().log("certs_cache::get_private_key: p_private_key: '%p'", *p_private_key);

  return 0;
}

int certs_cache::get_public_keys(const std::string& p_certificate_id, const EVP_PKEY** p_public_key) {
  loggers::get_instance().log(">>> certs_cache::get_public_keys: '%s'", p_certificate_id.c_str());

  const certs_db_record* record;
  if (load_certificate(p_certificate_id, &record) == -1) {
    loggers::get_instance().warning("certs_cache::get_public_keys: Failed to load certificate");
    return -1;
  }

  *p_public_key = record->public_key();
  loggers::get_instance().log("certs_cache::p_public_key: p_public_key: '%p'", *p_public_key);

  return 0;
}

int certs_cache::get_certificate_pem(const std::string& p_certificate_id, std::string& p_certificate_pem) {
  loggers::get_instance().log(">>> certs_cache::get_certificate_pem: '%s'", p_certificate_id.c_str());

  const certs_db_record* record;
  if (load_certificate(p_certificate_id, &record) == -1) {
    loggers::get_instance().warning("certs_cache::get_certificate_pem: Failed to load certificate");
    return -1;
  }

  p_certificate_pem = record->pem();
  loggers::get_instance().log("certs_cache::get_certificate_pem: p_certificate_pem: '%s'", p_certificate_pem.c_str());

  return 0;
}

int certs_cache::store_certificate(const std::string& p_certificate_name, const std::string& p_certificate_pem, std::string& p_certificate_id, const certs_db_record** p_record) {
  loggers::get_instance().log(">>> certs_cache::store_certificate: '%s'", p_certificate_name.c_str());

  if (certs_loader::get_instance().store_certificate(p_certificate_name, p_certificate_pem, p_certificate_id, _certificates) == -1) {
      loggers::get_instance().warning("certs_cache::store_certificate: Failed to load certificate");
      return -1;
    }
  // Certificate is on the DB, load it
  std::map<std::string, std::unique_ptr<const certs_db_record>>::const_iterator it = _certificates.find(p_certificate_id);
  *p_record = it->second.get();
  // Mapping certificate id/certificate name
  _certificates_idx.insert(std::pair<std::string, const std::string>(p_certificate_id, p_certificate_name));
  std::string sn(256, (char)0x00);
  const X509_NAME* subject = (*p_record)->subject_name();
  ::X509_NAME_oneline(subject, sn.data(), sn.length());
    loggers::get_instance().log("certs_cache::store_certificate: sn: '%s'", sn.c_str());
  _certificates_subject.insert(std::pair<std::string, const std::string>(sn, p_certificate_id));

  return 0; 
}

void certs_cache::dump() const {
  loggers::get_instance().log("certs_cache::dump_certificates: # items = %d", _certificates.size());

  for (std::map<std::string, std::unique_ptr<const certs_db_record>>::const_iterator it = _certificates.cbegin(); it != _certificates.cend(); ++it) {
    const certs_db_record *p = it->second.get();
  } // End of 'for' statement
} // End of method dump

const std::string certs_cache::cert_to_string(const std::string& p_certificate_id) {
  loggers::get_instance().log(">>> certs_cache::cert_to_string: '%s'", p_certificate_id.c_str());

  const certs_db_record* record;
  if (load_certificate(p_certificate_id, &record) == -1) {
    loggers::get_instance().warning("certs_cache::cert_to_string: Failed to load certificate");
    return std::string(::ERR_error_string(::ERR_get_error(), nullptr));;
  }

  BIO* bio = ::BIO_new(::BIO_s_mem());
  if (bio == nullptr) {
    loggers::get_instance().warning("certs_cache::cert_to_string: Failed to dunp certificate");
    return std::string(::ERR_error_string(::ERR_get_error(), nullptr));
  }

  if (::PEM_write_bio_X509(bio, (X509*)record->certificate()) == 0) {
    ::BIO_free(bio);
    loggers::get_instance().warning("certs_cache::cert_to_string: Failed to dunp certificate");
    return std::string(::ERR_error_string(::ERR_get_error(), nullptr));
  }

  size_t len = ::BIO_get_mem_data(bio, nullptr);
  std::string s(len, 0x00);
  ::BIO_read(bio, s.data(), len);
  ::BIO_free(bio);
  loggers::get_instance().log("certs_cache::cert_to_string: dump: '%s'", s.c_str());

  return s;
}

int certs_cache::publickey_to_string(const EVP_PKEY* p_public_kep, std::vector<unsigned char>& p_buffer) {
  loggers::get_instance().log(">>> certs_cache::publickey_to_string: '%p'", p_public_kep);

  unsigned char buffer[512];
  unsigned char* ptr = &buffer[0];
  int ret = ::i2d_PublicKey((EVP_PKEY*)p_public_kep, &ptr);
  loggers::get_instance().log("certs_cache::publickey_to_string: ret: '%d'", ret);
  if (ret < 0) {
    loggers::get_instance().warning("certs_cache::publickey_to_string: Failed to dunp certificate");
    p_buffer.clear();
    return -1;
  }

  p_buffer.assign((unsigned char*)buffer, (unsigned char*)(buffer + ret));
  loggers::get_instance().log("certs_cache::publickey_to_string: p_buffer len: '%d'", p_buffer.size());
  loggers::get_instance().log("certs_cache::publickey_to_string: dump: '%s'", converter::get_instance().bytes_to_hexa(p_buffer).c_str());

  return 0;
}