/*!
 * \file      certs_loader.hh
 * \brief     Header file for X506 PEM certificates loader definition.
 * \author    ETSI STF625
 * \copyright ETSI Copyright Notification
 *            No part may be reproduced except as authorized by written permission.
 *            The copyright and the foregoing restriction extend to reproduction in all media.
 *            All rights reserved.
 * \version   0.1
 * \remark Use `openssl x509 -in <certificate.pem> -noout -text` to view the certficate content
 *         Use `openssl [rsa|dsa|...] -inform PEM -in <private-key.pem> -text -noout` to view the private key content
 *         Use `openssl x509 -in <certificate.der> -out <certificate.pem>` to convert a DER certificate into a PEM certificate
 */
#pragma once

#include <map>
#include <set>
#include <string>
#include <vector>

#include <experimental/filesystem>

#include <openssl/bio.h>
#include <openssl/err.h>
#include <openssl/pem.h>
#include <openssl/x509.h>

#include "certs_db_record.hh"

/*!
 * \class certs_loader
 * \brief This class provides mechanism  to load the certificates from the filesystem according the struecture defined in ETSI TS 103 099
 * \remark Singleton pattern
 */
class certs_loader {
  std::experimental::filesystem::path _db_path; //! Path of the certificates storage
  bool                                _is_cache_initialized; //! Set to true when certificates are successfully loaded from file system
  static certs_loader*                instance; //! Unique static object reference of this class

  /*!
   * \brief Default private ctor
   */
  certs_loader();
  /*!
   * \brief Default private dtor
   */
  ~certs_loader() {
    if (instance != NULL) {
      delete instance;
      instance = NULL;
    }
  };

public: /*! \publicsection */
  /*!
   * \brief Public accessor to the single object reference
   */
  inline static certs_loader &get_instance() {
    if (instance == NULL)
      instance = new certs_loader();
    return *instance;
  };

  /*!
   * \fn int build_path(const std::string& p_root_directory);
   * \brief Set the pass of the certificate storage
   * \param[in] p_db_path The pass of the certificate storage
   */
  int build_path(const std::string& p_root_directory);
  /*!
   * \fn int get_certificate_id(const std::string& p_certificate_name, std::string& p_certificate_id);
   * \brief Compute the SHA-1 hash of the certificate name to provide the certificate identifier
   * \param[in] p_certificate_name The certificate name
   * \param[out] p_certificate_id The certificate identifier
   * \return 0 on success, -1 otherwise
   */
  int get_certificate_id(const std::string& p_certificate_name, std::string& p_certificate_id);
  /*!
   * \fn int load_certificate(const std::string& p_certificate_name, const std::string& p_private_key_name, const std::string& p_private_key_passwd, std::map<std::string, std::unique_ptr<const certs_db_record>> & p_certificates);
   * \brief Store in memory the specified certificate
   * \param[in] p_certificate_name The certificate full path bame
   * \param[in] p_private_key_name The certificate's private key full path name, in PKCS#8 format
   * \param[in] p_private_key_passwd The certificate's private key password for PKCS#8 format encryption. Empty if no encryption
   * \param[out] p_certificate_id The certificate identifier
   * \param[inout] p_certificates The map of the loaded in memory certificates
   * \return 0 on success, -1 otherwise
   */
  int load_certificate(const std::string& p_certificate_name, const std::string& p_private_key_name, const std::string& p_private_key_passwd, std::string& p_certificate_id, std::map<std::string, std::unique_ptr<const certs_db_record>> & p_certificates);
  /*!
   * \fn int store_certificate(const std::string& p_certificate_name, const std::string& p_certificate_pem, std::string& p_certificate_id, std::map<std::string, std::map<std::string, std::unique_ptr<const certs_db_record>> & p_certificates);
   * \brief Save the specified certificate and update the internal maps
   * \param[in] p_certificate_name The certificate neame
   * \param[in] p_certificate_pem The certificate to save in PEM format
   * \param[out] p_certificate_id The certificate identifier
   * \return 0 on success, -1 otherwise
   */
  int store_certificate(const std::string& p_certificate_name, const std::string& p_certificate_pem, std::string& p_certificate_id, std::map<std::string, std::unique_ptr<const certs_db_record>> & p_certificates);
}; // End of class certs_loader

