/*!
 * \file      xml_converters.cc
 * \brief     Source file for the xml_converters functionality.
 * \author    ETSI STF637
 * \copyright ETSI Copyright Notification
 *            No part may be reproduced except as authorized by written permission.
 *            The copyright and the foregoing restriction extend to reproduction in all media.
 *            All rights reserved.
 * \version   0.1
 */
#include "xml_converters.hh"

#include "loggers.hh"

#include <libxml/parser.h>
#include <libxml/c14n.h>

xml_converters* xml_converters::_instance = nullptr;

int xml_converters::xml_canonicalization(const std::string& p_to_canonical, std::string& p_canonicalized) const {
  loggers::get_instance().log(">>> xml_converters::xml_canonicalization: '%s'", p_to_canonical.c_str());

  xmlDocPtr doc = xmlParseMemory(p_to_canonical.c_str(), p_to_canonical.length());
  if (doc == nullptr) {
    loggers::get_instance().log("xml_converters::xml_canonicalization: Failed to read XML input");
    return -1;
  }

  xmlOutputBufferPtr buffer = xmlAllocOutputBuffer(nullptr);
  if (buffer == nullptr) {
    loggers::get_instance().log("xml_converters::xml_canonicalization: Failed to read XML input");
    xmlFreeDoc(doc);
    return -1;
  }

  // Do canonicalization
  loggers::get_instance().log("xml_converters::xml_canonicalization: Before");
  if (xmlC14NExecute(doc, (xmlC14NIsVisibleCallback)&xml_converters::xml_node_set_contains_callback, nullptr, XML_C14N_1_1, nullptr, 1, buffer) == -1) {
    loggers::get_instance().log("xml_converters::xml_canonicalization: Failed to read XML input");
    xmlFreeDoc(doc);
    xmlOutputBufferClose(buffer);
    return -1;
  }
  loggers::get_instance().log("xml_converters::xml_canonicalization: After");

  // Retrieve the canonicalization XML document
  const char* data = (const char*)xmlOutputBufferGetContent(buffer);
  size_t size = xmlOutputBufferGetSize(buffer);
  p_canonicalized.assign(data, data + size);

  // Free resources
  xmlOutputBufferClose(buffer);
  xmlFreeDoc(doc);

  loggers::get_instance().log("<<< xml_converters::xml_canonicalization: '%s'", p_canonicalized.c_str());
  return 0;
}

int xml_converters::xml_node_set_contains_callback(void* p_user_data, xmlNodePtr p_node, xmlNodePtr p_parent) {
  //loggers::get_instance().log("xml_converters::xml_node_set_contains_callback");
  return 1;
}
