/*!
 * \file      certs_db_record.hh
 * \brief     Header file for a storage of certificates mechanism.
 * \author    ETSI STF637
 * \copyright ETSI Copyright Notification
 *            No part may be reproduced except as authorized by written permission.
 *            The copyright and the foregoing restriction extend to reproduction in all media.
 *            All rights reserved.
 * \version   0.1
 */
#pragma once

#include <string>

#include <openssl/x509.h>

/*!
 * \class certs_db_record
 * \brief This class provides security record description for security database
 */
class certs_db_record {

  std::string _certificate_id;   /*!< Certificate name */
  X509*       _certificate;      /*!< X509 Certificate */
  EVP_PKEY*   _private_key;      /*!< Private key */
  std::string _pem;              /*!< Certificate PEM format */

public: /*! \publicsection */
  /*!
   * \brief Default ctor
   */
  explicit certs_db_record()
    : _certificate_id(nullptr), _certificate(nullptr), _private_key(nullptr), _pem{} {};
  /*!
   * \brief Specialised ctor
   * \param[in] p_certificate_id The certificate identifier
   * \param[in] p_certificate The X509 certificate in openssl format
   * \param[in] p_private_key The certificate's private key in openssl format
   * \param[in] p_certificate_pem The certificate's PEM format
   */
  certs_db_record(const std::string& p_certificate_id, X509* p_certificate, EVP_PKEY* p_private_key, const std::string& p_certificate_pem);

  /*!
   * \brief Default dtor
   */
  ~certs_db_record();

  /*!
   * \inline
   * \fn const std::string& certificate_id() const;
   * \brief Retrieve the certificate identifier value
   * \return The certificate identifier
   */
  inline const std::string& certificate_id() const { return _certificate_id; };
  /*!
   * \inline
   * \fn const X509* certificate() const;
   * \brief Retrieve the COER encoded certificate
   * \return The COER encoded certificate
   */
  inline const X509* certificate() const { return _certificate; };
  inline const EVP_PKEY* private_key() const { return _private_key; };
  inline const EVP_PKEY* public_key() const { return ::X509_get_pubkey(_certificate); };
  inline const std::string& pem() const { return _pem; };
}; // End of class certs_db_record
