/*!
 * \file      certs_cache.hh
 * \brief     Header file for a certificates caching storage mechanism.
 *            It is used to store certificates received from neighbors and not present in the data base
 * \author    ETSI STF637
 * \copyright ETSI Copyright Notification
 *            No part may be reproduced except as authorized by written permission.
 *            The copyright and the foregoing restriction extend to reproduction in all media.
 *            All rights reserved.
 * \version   0.1
 */
#pragma once

#include <map>
#include <memory>
#include <string>
#include <vector>

#include <openssl/x509.h>

#include "certs_db_record.hh"

/*!
 * \class certs_cache
 * \brief This class provides security record description for in memory security database. In memory mens there is no disk load/save mechanism
 */
class certs_cache {
  static bool fill_vector(std::string& p_vector, const std::string& p_org);

protected: /*! \protectedsection */
  // TODO Enforce with const security_cache_record, and const std::string
  std::map<std::string, const std::string> _certificates_idx; //! List of the certificate names indexed by the certificate identifier, i.e. the SH1 of the certificate name
  std::map<std::string, std::unique_ptr<const certs_db_record>> _certificates; //! List of the certificates indexed by the certificate identifier

public: /*! \publicsection */
  /*!
   * \brief Default ctor
   */
  certs_cache();
  /*!
   * \brief Default private dtor
   */
  virtual ~certs_cache();

  int get_certificate(const std::string& p_certificate_name, const std::string& p_private_key_name, const X509** p_certificate);
  /*!
   * \fn int get_certificate(const std::string& p_hashed_id8, const X509** p_certificate);
   * \brief Retrive the specified certificate either from memory
   * \param[in] p_certificate_id The certificate name
   * \param[out] p_certificate The X509 certificate
   * \return 0 on success, -1 otherwise
   */
  int get_certificate(const std::string& p_certificate_id, const X509** p_certificate);
  /*!
   * \fn int get_private_key(const std::string& p_certificate_id, const EVP_PKEY** p_private_key);
   * \brief Retrive the private key of the specified certificate
   * \param[in] p_certificate_id The certificate name
   * \param[out] p_private_key The private key
   * \return 0 on success, -1 otherwise
   */
  int get_private_key(const std::string& p_certificate_id, const EVP_PKEY** p_private_key);
  /*!
   * \fn int get_public_keys(const std::string& p_certificate_id, const EVP_PKEY** p_public_key);
   * \brief Retrive the public keys (in uncompressed format) of the specified certificate
   * \param[in] p_certificate_id The certificate name
   * \param[out] p_public_key The public key
   * \return 0 on success, -1 otherwise
   */
  int get_public_keys(const std::string& p_certificate_id, const EVP_PKEY** p_public_key);

  int get_certificate_pem(const std::string& p_certificate_id, std::string& p_certificate_pem);

  virtual int store_certificate(const std::string& p_certificate_name, const std::string& p_certificate, const EVP_PKEY* p_private_key);

  /*!
   * \fn int int clear();
   * \brief Cleanup the cache
   * \return 0 on success, -1 otherwise
   * \virtual
   */
  virtual int clear();

  void dump() const;

private: /*! \privatesection */
  /*!
   * \fn int load_certificate(const std::string& p_certificate_name, const std::string& p_private_key_name, const certs_db_record** p_record);
   * \brief Retrive the specified certificate either from memory of from storage location
   * \param[in] p_certificate_name The certificate name
   * \param[in] p_private_key_name The certificate's private key name
   * \param[out] p_record A reference to the certificate record
   * \return 0 on success, -1 otherwise
   */
  int load_certificate(const std::string& p_certificate_name, const std::string& p_private_key_name, const certs_db_record** p_record);
  /*!
   * \fn int load_certificate(const std::string& p_certificate_id, const certs_db_record** p_record);
   * \brief Retrive the specified certificate either from memory of from storage location
   * \param[in] p_certificate_id The certificate identifier
   * \param[out] p_record A reference to the certificate record
   * \return 0 on success, -1 otherwise
   */
  int load_certificate(const std::string& p_certificate_id, const certs_db_record** p_record);

}; // End of class certs_cache
