#include <fstream>
#include <iostream>

#include <openssl/bio.h>
#include <openssl/err.h>
#include <openssl/pem.h>
#include <openssl/x509.h>

#include <TTCN3.hh>

#include "sha1.hh"
#include "certs_loader.hh"

#include "loggers.hh"

certs_loader *certs_loader::instance = nullptr;

certs_loader::certs_loader() : _db_path(), _is_cache_initialized{false} {
  loggers::get_instance().log(">>> certs_loader::certs_loader");
} // End of ctor

int certs_loader::build_path(const std::string& p_root_directory) { 
  loggers::get_instance().log(">>> certs_loader::build_path: '%s'", p_root_directory.c_str());

  // Sanity check
  if (_is_cache_initialized) {
    return 0;
  }

  // Build full path
  if (!p_root_directory.empty()) {
    _db_path = p_root_directory;
    if (!std::experimental::filesystem::exists(_db_path) ||
        !std::experimental::filesystem::is_directory(_db_path)) { // FIXME Coredump when app hasn't the rights to create the directory!!!!
      // Create directory
      if (!std::experimental::filesystem::create_directory(_db_path)) {
        _db_path = "./";
      } else { // Set rights for all users
        std::experimental::filesystem::permissions(_db_path,
                                                   std::experimental::filesystem::perms::add_perms | std::experimental::filesystem::perms::owner_all |
                                                   std::experimental::filesystem::perms::group_all | std::experimental::filesystem::perms::others_all);
      }
    }
  } else {
    _db_path = "./";
  }
  std::experimental::filesystem::canonical(_db_path);
  loggers::get_instance().log("certs_loader::build_path: full path: %s", _db_path.string().c_str());
  if (!std::experimental::filesystem::exists(_db_path)) {
    loggers::get_instance().warning("certificates_loader::build_path: Invalid path");
    _db_path.clear();
    return -1;
  }


  loggers::get_instance().log("<<< certs_loader::build_path");
  return 0;
}
  
int certs_loader::get_certificate_id(const std::string& p_certificate_name, std::string& p_certificate_id) {
  loggers::get_instance().log(">>> certs_loader::get_certificate_id: '%s'", p_certificate_name.c_str());

  // Build the certificate identifier
  sha1 s;
  const OCTETSTRING buffer = OCTETSTRING(p_certificate_name.length(), (unsigned char*)p_certificate_name.c_str());
  OCTETSTRING hash;
  if (s.generate(buffer, hash) != 0) {
    loggers::get_instance().warning("certs_cache::load_certificate: Failed to build the certificate identifier");
    return -1;   
  }
  loggers::get_instance().log_msg("certs_loader::get_certificate_id: hash: ", hash);
  CHARSTRING hash_str = oct2str(hash);
  p_certificate_id.assign(static_cast<const char*>(hash_str), static_cast<const char*>(hash_str) + hash_str.lengthof());

  loggers::get_instance().log("<<< certs_loader::get_certificate_id: '%s'", p_certificate_id.c_str());
  return 0;
}

int certs_loader::load_certificate(const std::string& p_certificate_name, const std::string& p_private_key_name, const std::string& p_private_key_passwd, std::string& p_certificate_id, std::map<std::string, std::unique_ptr<const certs_db_record>> & p_certificates) {
  loggers::get_instance().log(">>> certs_loader::load_certificate: '%s'", p_certificate_name.c_str());
  loggers::get_instance().log(">>> certs_loader::load_certificate: '%s'", p_private_key_name.c_str());

  // Load certificate file
  BIO* certbio = ::BIO_new(::BIO_s_file());
  if (certbio == nullptr) {
    loggers::get_instance().warning("certs_loader::load_certificate: Error creating BIO");
    return -1;
  }
  std::experimental::filesystem::path p = _db_path.string() + "/" + p_certificate_name;
  loggers::get_instance().log("certs_loader::load_certificate: p='%s'", p.string().c_str());
  int ret = BIO_read_filename(certbio, p.string().c_str());

  X509* cert = ::PEM_read_bio_X509(certbio, NULL, 0, NULL);
  if (cert == nullptr) {
    loggers::get_instance().warning("certs_loader::load_certificate: Error loading cert into memory");
    ::BIO_free_all(certbio);
    return -1;
  }

  std::unique_ptr<BIO, decltype(&::BIO_free)> bio(BIO_new(BIO_s_mem()), ::BIO_free);
  ret = ::PEM_write_bio_X509(bio.get(), cert);
  if (ret != 1) {
    loggers::get_instance().warning("certs_loader::load_certificate: PEM_write_bio_X509 failed, error %s", ::ERR_get_error());
    ::X509_free(cert);
    ::BIO_free_all(certbio);
    return -1;
  }
  BUF_MEM *mem = NULL;
  ::BIO_get_mem_ptr(bio.get(), &mem);
  if (!mem || !mem->data || !mem->length) {
    loggers::get_instance().warning("certs_loader::load_certificate: BIO_get_mem_ptr failed, error %s", ::ERR_get_error());
    ::X509_free(cert);
    ::BIO_free_all(certbio);
    return -1;
  }
  std::string pem(mem->data, mem->length);
  // Remove labels
  loggers::get_instance().log("certs_loader::load_certificate: certificate pem (1): '%s'", pem.c_str());
  std::string s("-----BEGIN CERTIFICATE-----\n");
  std::string::size_type idx = pem.find_first_of(s);
  pem = pem.substr(s.length());
  s = "\n-----END CERTIFICATE-----\n";
  idx = pem.rfind(s);
  pem = pem.substr(0, pem.length() - s.length());
  //loggers::get_instance().log("certs_loader::load_certificate: certificate pem (2): '%s'", pem.c_str());

  ::BIO_free_all(certbio);

  // Load private key file
  certbio = ::BIO_new(::BIO_s_file());
  if (certbio == nullptr) {
    loggers::get_instance().warning("certs_loader::load_certificate: Error creating BIO");
    ::X509_free(cert);
    return -1;
  }
  p = _db_path.string() + "/" + p_private_key_name;
  ret = BIO_read_filename(certbio, p.string().c_str());
  pem_password_cb* cb = NULL;
  if (!p_private_key_passwd.empty() > 0) {
    // TODO
  }
  EVP_PKEY* private_key = ::PEM_read_bio_PrivateKey(certbio, NULL, cb, NULL);
  if (private_key == nullptr) {
    loggers::get_instance().warning("certs_loader::load_certificate: Error loading cert into memory");
    ::X509_free(cert);
    ::BIO_free_all(certbio);
    return -1;
  }
  ::BIO_free_all(certbio);

  // Build the certificate identifier
  if (get_certificate_id(p_certificate_name, p_certificate_id) != 0) {
    loggers::get_instance().warning("certs_cache::load_certificate: Failed to build the certificate identifier");
    return -1;   
  }
  loggers::get_instance().log("certs_loader::load_certificate: p_certificate_name: '%s'", p_certificate_name.c_str());
  loggers::get_instance().log("certs_loader::load_certificate: p_certificate_id: '%s'", p_certificate_id.c_str());
  loggers::get_instance().log("certs_loader::load_certificate: cert: '%p'", cert);
  loggers::get_instance().log("certs_loader::load_certificate: private_key: '%p'", private_key);
  //loggers::get_instance().log("certs_loader::load_certificate: certificate pem: '%s'", pem.c_str());

  // Create new record
  certs_db_record *r = new certs_db_record(p_certificate_name, cert, private_key, pem);
  std::pair<std::map<std::string, std::unique_ptr<const certs_db_record>>::iterator, bool> result = p_certificates.insert(std::pair<std::string, std::unique_ptr<const certs_db_record>>(p_certificate_id, std::unique_ptr<const certs_db_record>(r)));
  if (result.second == false) {
    loggers::get_instance().warning("certs_loader::build_certificates_cache: Failed to insert new record '%s'", p_certificate_name.c_str());
    delete r;
    return -1;
  }

  return 0;
}

int certs_loader::store_certificate(const std::string& p_certificate_name, const std::string& p_certificate_pem, std::string& p_certificate_id, std::map<std::string, std::unique_ptr<const certs_db_record>> & p_certificates) {
  loggers::get_instance().log(">>> certs_loader::store_certificate '%s'", p_certificate_name.c_str());
  loggers::get_instance().log(">>> certs_loader::store_certificate: '%s'", p_certificate_pem.c_str());

  // Build the certificate identifier
  if (get_certificate_id(p_certificate_name, p_certificate_id) != 0) {
    loggers::get_instance().warning("certs_cache::store_certificate: Failed to build the certificate identifier");
    return -1;   
  }

  // Load certificate file
  std::string s("-----BEGIN CERTIFICATE-----\n");
  if (p_certificate_pem.find(s) == std::string::npos) { // Add delimeters
    loggers::get_instance().log("certs_loader::store_certificate: Add delimeters");
    s = s.append(p_certificate_pem);
    s = s.append("\n-----END CERTIFICATE-----\n");
  } else {
    loggers::get_instance().log("certs_loader::store_certificate: Do not add delimeters");
    s = p_certificate_pem;
  }
  loggers::get_instance().log("certs_loader::store_certificate: pem: '%s'", s.c_str());
  BIO* certbio = ::BIO_new_mem_buf(s.c_str(), -1);
  if (certbio == nullptr) {
    loggers::get_instance().warning("certs_loader::store_certificate: Error creating in memory BIO");
    return -1;
  }

  X509* cert = ::PEM_read_bio_X509(certbio, NULL, 0, NULL);
  if (cert == nullptr) {
    loggers::get_instance().warning("certs_loader::store_certificate: Error loading cert into memory");
    ::BIO_free_all(certbio);
    return -1;
  }
  ::BIO_free(certbio);

  loggers::get_instance().log("certs_loader::store_certificate: p_certificate_name: '%s'", p_certificate_name.c_str());
  loggers::get_instance().log("certs_loader::store_certificate: p_certificate_id: '%s'", p_certificate_id.c_str());
  loggers::get_instance().log("certs_loader::store_certificate: cert: '%p'", cert);

  // Create new record
  certs_db_record *r = new certs_db_record(p_certificate_name, cert, nullptr, p_certificate_pem);
  std::pair<std::map<std::string, std::unique_ptr<const certs_db_record>>::iterator, bool> result = p_certificates.insert(std::pair<std::string, std::unique_ptr<const certs_db_record>>(p_certificate_id, std::unique_ptr<const certs_db_record>(r)));
  if (result.second == false) {
    loggers::get_instance().warning("certs_loader::store_certificate: Failed to insert new record '%s'", p_certificate_name.c_str());
    delete r;
    return -1;
  }

  return 0;
} // End of method store_certificate

