#include "LibSecurity_Hash.hh"
#include "LibSecurity_Signature.hh"
#include "LibSecurity_Certificates.hh"

#include "loggers.hh"

#include "sha1.hh"
#include "sha256.hh"
#include "sha384.hh"
#include "certs_db.hh"
#include "security_services.hh"

static std::unique_ptr<security_services> _security_services;

static int transform_signature_workaround(std::string& str) {
  loggers::get_instance().log(">>> transform_signature_workaround: '%s'", str.c_str());
  int first = str.find("&lt;");
  while ((first != -1) && (first < str.length())) {
    str = str.substr(0, first) + "<" + str.substr(first + 4,  str.length() - 4);
    //loggers::get_instance().log("transform_signature_workaround: New str (1): '%s'", str.c_str());
    first = str.find("&lt;");
  } // End of 'while'statement
  first = str.find("&gt;");
  while ((first != -1) && (first < str.length())) {
    str = str.substr(0, first) + ">" + str.substr(first + 4,  str.length() - 4);
    //loggers::get_instance().log("transform_signature_workaround: New str (2): '%s'", str.c_str());
    first = str.find("&gt;");
  } // End of 'while'statement
  first = str.find("&quot;");
  while ((first != -1) && (first < str.length())) {
    str = str.substr(0, first) + "'" + str.substr(first + 6,  str.length() - 6);
    //loggers::get_instance().log("transform_signature_workaround: New str (3): '%s'", str.c_str());
    first = str.find("&quot;");
  } // End of 'while'statement
  first = str.find("&#13;");
  while ((first != -1) && (first < str.length())) {
    str = str.substr(0, first) + "\r" + str.substr(first + 5,  str.length() - 5);
    //loggers::get_instance().log("transform_signature_workaround: New str (4): '%s'", str.c_str());
    first = str.find("&#13;");
  } // End of 'while'statement

  std::replace(str.begin(), str.end(), '\'', '\"');

  loggers::get_instance().log("<<< transform_signature_workaround: '%s'", str.c_str());
  return 0;
}

static int transform_xslt_workaround(std::string& str) {
  loggers::get_instance().log(">>> transform_xslt_workaround: '%s'", str.c_str());

  int start = str.find("<xsl:stylesheet");
  //loggers::get_instance().log("transform_xslt_workaround: start='%d' ", start);
  int stop = str.find("</Transform>");
  //loggers::get_instance().log("transform_xslt_workaround: stop='%d' ", stop);
  int first = str.find("<", start);
  while ((first != -1) && (first < stop)) {
    //loggers::get_instance().log("transform_xslt_workaround: first='%d' ", first);
    str = str.substr(0, first) + "&lt;" + str.substr(first + 1,  str.length() - 1);
    //loggers::get_instance().log("transform_xslt_workaround: New str (1): '%s'", str.c_str());
    first = str.find("<", first);
    stop = str.find("</Transform>");
  } // End of 'while'statement
  first = str.find(">", start);
  while ((first != -1) && (first < stop)) {
    //loggers::get_instance().log("transform_xslt_workaround: first='%d' ", first);
    str = str.substr(0, first) + "&gt;" + str.substr(first + 1,  str.length() - 1);
    //loggers::get_instance().log("transform_xslt_workaround: New str (1): '%s'", str.c_str());
    first = str.find(">", first);
    stop = str.find("</Transform>");
  } // End of 'while'statement
  first = str.find("\"", start);
  while ((first != -1) && (first < stop)) {
    //loggers::get_instance().log("transform_xslt_workaround: first='%d' ", first);
    str = str.substr(0, first) + "&quot;" + str.substr(first + 1,  str.length() - 1);
    //loggers::get_instance().log("transform_xslt_workaround: New str (1): '%s'", str.c_str());
    first = str.find("\"", first);
    stop = str.find("</Transform>");
  } // End of 'while'statement

  loggers::get_instance().log("<<< transform_xslt_workaround: '%s'", str.c_str());
  return 0;
}

INTEGER LibSecurity__Certificates::fx__init__certs__db(const CHARSTRING& p_certs_db_path) {
  loggers::get_instance().log_msg(">>> fx__init__certs__db: ", p_certs_db_path);

  int ret = _security_services->initialize(std::string(static_cast<const char*>(p_certs_db_path)));

  loggers::get_instance().log("<<< fx__init__certs__db.");
  return ret;
}

INTEGER LibSecurity__Certificates::fx__load__certificate(const CHARSTRING& p_certificate_name, const CHARSTRING& p_private_key_name, const CHARSTRING& p_private_key_passwd) {
  loggers::get_instance().log_msg(">>> fx__load__certificate: ", p_certificate_name);
  loggers::get_instance().log_msg(">>> fx__load__certificate: ", p_private_key_name);

  const X509* certificate;
  int ret = _security_services->load_certificate(std::string(static_cast<const char*>(p_certificate_name)), std::string(static_cast<const char*>(p_private_key_name)), std::string(static_cast<const char*>(p_private_key_passwd)), &certificate);
  loggers::get_instance().log("fx__load__certificate: certificate: '%p'", certificate);
  if (ret == 0) {
    ASN1_INTEGER* asn1_serial = ::X509_get_serialNumber((X509*)certificate);
    if (asn1_serial == nullptr) {
      loggers::get_instance().log("fx__load__certificate: Failed to retrieve X509 serial number");
      return -1;
    }
    uint64_t v;
    ASN1_INTEGER_get_uint64(&v, asn1_serial);
    loggers::get_instance().log("fx__load__certificate: Loaded certificate: serial number: %ld", v);
  }

  loggers::get_instance().log("<<< fx__load__certificate");
  return ret;
}

OCTETSTRING LibSecurity__Hash::fx__hash(const OCTETSTRING& p_to_be_hashed, const LibSecurity__Hash::HashAlgorithm& p_hash_algorithm) {
  loggers::get_instance().log_msg(">>> fx__hash: ", p_to_be_hashed);

  OCTETSTRING hash;
  switch (p_hash_algorithm) {
    case LibSecurity__Hash::HashAlgorithm::e__sha1: {
        sha1 s;
        s.generate(p_to_be_hashed, hash);
      }
      break;
    case LibSecurity__Hash::HashAlgorithm::e__sha256: {
        sha256 s;
        s.generate(p_to_be_hashed, hash);
      }
      break;
    case LibSecurity__Hash::HashAlgorithm::e__sha384: {
        sha384 s;
        s.generate(p_to_be_hashed, hash);
      }
      break;
  } // End of 'switch' statement

  loggers::get_instance().log_msg("<<< fx__hash: ", hash);
  return hash;
}

BITSTRING LibSecurity__Signature::fx__enc__xmldsig__signed__info(const http__www__w3__org__2000__09__xmldsig::Signature_signedInfo& s) { // FIXME Use enc/dec TITAN function external function f_enc_value(in Value x) return bitstring with { extension "prototype(convert) encode(abc)" }
  loggers::get_instance().log(">>> fx__enc__xmldsig__signed__info");

  TTCN_EncDec::clear_error();
  TTCN_EncDec::set_error_behavior(TTCN_EncDec::ET_ALL, TTCN_EncDec::EB_DEFAULT);
  TTCN_Buffer encoding_buffer;

  s.encode(http__www__w3__org__2000__09__xmldsig::Signature_signedInfo_descr_, encoding_buffer, TTCN_EncDec::CT_XER, XER_EXTENDED);

  // FIXME Update <transform>
  std::string str(static_cast<const char*>((const char*)encoding_buffer.get_data()), encoding_buffer.get_len() + static_cast<const char*>((const char*)encoding_buffer.get_data()));
  loggers::get_instance().log("fx__enc__xmldsig__signed__info: Before str: '%s'", str.c_str());
  transform_signature_workaround(str);
  loggers::get_instance().log("fx__enc__xmldsig__signed__info: Afer str: '%s'", str.c_str());

  OCTETSTRING os = char2oct(CHARSTRING(str.c_str()));
  loggers::get_instance().log_msg("fx__enc__xmldsig__signed__info: os: ", os);

  return oct2bit(os);
}

BITSTRING LibSecurity__Signature::fx__enc__xmldsig(const http__www__w3__org__2000__09__xmldsig::Signature& s) { // FIXME Use enc/dec TITAN function external function f_enc_value(in Value x) return bitstring with { extension "prototype(convert) encode(abc)" }
  loggers::get_instance().log(">>> fx__enc__xmldsig");

  TTCN_EncDec::clear_error();
  TTCN_EncDec::set_error_behavior(TTCN_EncDec::ET_ALL, TTCN_EncDec::EB_DEFAULT);
  TTCN_Buffer encoding_buffer;

  s.encode(http__www__w3__org__2000__09__xmldsig::Signature_descr_, encoding_buffer, TTCN_EncDec::CT_XER, XER_EXTENDED);

  // FIXME Update <transform>
  std::string str(static_cast<const char*>((const char*)encoding_buffer.get_data()), encoding_buffer.get_len() + static_cast<const char*>((const char*)encoding_buffer.get_data()));
  loggers::get_instance().log("fx__enc__xmldsig: Before str: '%s'", str.c_str());
  transform_signature_workaround(str);
  loggers::get_instance().log("fx__enc__xmldsig: Afer str: '%s'", str.c_str());

  OCTETSTRING os = char2oct(CHARSTRING(str.c_str()));
  loggers::get_instance().log_msg("fx__enc__xmldsig: os: ", os);

  return oct2bit(os);
}

INTEGER LibSecurity__Signature::fx__dec__xmldsig(BITSTRING& bs, http__www__w3__org__2000__09__xmldsig::Signature& s) { // FIXME Use enc/dec TITAN function external function f_enc_value(in Value x) return bitstring with { extension "prototype(convert) encode(abc)" }
  loggers::get_instance().log(">>> fx__dec__xmldsig");

  std::string str(static_cast<const char*>(oct2char(bit2oct(bs))));
  loggers::get_instance().log("fx__dec__xmldsig: Before str: '%s'", str.c_str());
  transform_signature_workaround(str);
  transform_xslt_workaround(str);
  loggers::get_instance().log("fx__dec__xmldsig: Afer str: '%s'", str.c_str());

  TTCN_EncDec::clear_error();
  TTCN_EncDec::set_error_behavior(TTCN_EncDec::ET_ALL, TTCN_EncDec::EB_DEFAULT);
  TTCN_Buffer decoding_buffer(OCTETSTRING(str.length(), (const unsigned char*)str.c_str()));

  s.decode(http__www__w3__org__2000__09__xmldsig::Signature_descr_, decoding_buffer, TTCN_EncDec::CT_XER, XER_EXTENDED);

  loggers::get_instance().log_msg("<<< fx__dec__xmldsig: ", s);
  return 0;
}

INTEGER LibSecurity__Signature::fx__sign(const OCTETSTRING& p_encoded_message, const OCTETSTRING& p_empty_signature, const CHARSTRING& p_certificate_name, const CHARSTRING& p_private_key_name, const CHARSTRING& p_private_key_passwd, OCTETSTRING& p_signature, OCTETSTRING& p_digest, CHARSTRING& p_x509_certificate_subject, CHARSTRING& p_x509_certificate_pem, CHARSTRING& p_pull_request_canonicalized) {
  loggers::get_instance().log_msg(">>> fx__sign: ", p_encoded_message);

  if (_security_services->do_sign(p_encoded_message, p_empty_signature, p_certificate_name, p_private_key_name, p_private_key_passwd, p_signature, p_digest, p_x509_certificate_subject, p_x509_certificate_pem, p_pull_request_canonicalized) == -1) {
    loggers::get_instance().log("fx__sign: Failed to signed message");
    return -1;
  }
  
  return 0;
}

BOOLEAN LibSecurity__Signature::fx__do__sign__verify(const CHARSTRING& p_message, const OCTETSTRING& p_empty_signature, const UNIVERSAL_CHARSTRING& p_canonicalization_method, const UNIVERSAL_CHARSTRING& p_signature_method, const UNIVERSAL_CHARSTRING& p_digest_method, const UNIVERSAL_CHARSTRING& p_digest_value, const UNIVERSAL_CHARSTRING& p_signature_value, const UNIVERSAL_CHARSTRING& p_subject_name, const UNIVERSAL_CHARSTRING& p_certificate, CHARSTRING& p_debug_message) {
  loggers::get_instance().log(">>> fx__do__sign__verify");

  if (!_security_services->do_sign_verify(p_message, p_empty_signature, p_canonicalization_method, p_signature_method, p_digest_method, p_digest_value, p_signature_value, p_subject_name, p_certificate, p_debug_message)) {
    loggers::get_instance().log("fx__do__sign__verify: Failed to verify message signature");
    return false;
  }
  
  return true;
}
