/*!
 * \file      sha1.hh
 * \brief     Header file for SHA-1 helper methods.
 * \author    ETSI STF637
 * \copyright ETSI Copyright Notification
 *            No part may be reproduced except as authorized by written permission.
 *            The copyright and the foregoing restriction extend to reproduction in all media.
 *            All rights reserved.
 * \version   0.1
 */
#pragma once

#include <openssl/objects.h>
#include <openssl/sha.h>

class OCTETSTRING; //! TITAN forward declaration

/*!
 * \class sha1
 * \brief  This class provides description of SHA-1 helper methods
 */
class sha1 {
  SHA_CTX _ctx; //! SHA context
public:
  /*!
   * \brief Default constructor
   *        Create a new instance of the sha1 class
   */
  explicit sha1() : _ctx{} {};
  /*!
   * \brief Default destructor
   */
  virtual ~sha1(){};

  /*!
   * \fn int generate(const OCTETSTRING& p_buffer, OCTETSTRING& p_hash);
   * \brief Receive bytes formated data from the lower layers
   * \param[in] p_buffer The data used to generate the SHA-1 hash
   * \param[out] p_hash The SHA-1 hash value based of the provided data
   * \return 0 on success, -1 otherwise
   */
  int generate(const OCTETSTRING &p_buffer, OCTETSTRING &p_hash);

  /*!
   * \fn int generate(const uint8_t* p_buffer, OCTETSTRING& p_hash);
   * \brief Receive bytes formated data from the lower layers
   * \param[in] p_buffer The data used to generate the SHA-1 hash
   * \param[in] The length of the data buffer
   * \param[out] p_hash The SHA-1 hash value based of the provided data
   * \return 0 on success, -1 otherwise
   */
  int generate(const uint8_t *p_buffer, const size_t p_length, OCTETSTRING &p_hash);

  /*!
   * \fn const OCTETSTRING get_sha1_empty_string() const;
   * \brief Return the SHA-1 of an empty string
   * \return The SHA-1 of an empty string
   */
  inline const OCTETSTRING get_sha1_empty_string() const {
    static uint8_t sha1_empty_string[] = {
      0xda, 0x39, 0xa3, 0xee, 0x5e, 0x6b, 0x4b, 0x0d, 0x32, 0x55, 0xbf, 0xef, 0x95, 0x60, 0x18, 0x90, 
      0xaf, 0xd8, 0x07, 0x09}; //! SHA-1 of an empty string
    return OCTETSTRING(20, sha1_empty_string);
  };

}; // End of class sha1
