/*!
 * \file      params.cc
 * \brief     Source file for the parameter dictionary.
 * \author    ETSI STF525
 * \copyright ETSI Copyright Notification
 *            No part may be reproduced except as authorized by written permission.
 *            The copyright and the foregoing restriction extend to reproduction in all media.
 *            All rights reserved.
 * \version   0.1
 */
#include <regex>
#include <stdexcept>

#include "loggers.hh"
#include "params.hh"

const std::string& params::debug        = std::string("debug");

const std::string& params::loopback     = std::string("loopback");

const std::string& params::timestamp    = std::string("timestamp");

const std::string& params::mac_src      = std::string("mac_src");
const std::string& params::mac_dst      = std::string("mac_dst");
const std::string& params::mac_bc       = std::string("mac_bc");
const std::string& params::eth_type     = std::string("eth_type");
const std::string& params::filter       = std::string("filter");

const std::string& params::nic          = std::string("nic");

const std::string& params::ip_src      = std::string("ip_src");
const std::string& params::ip_dst      = std::string("ip_dst");
const std::string& params::ip_proto    = std::string("ip_proto");

const std::string& params::server       = std::string("server");
const std::string& params::port         = std::string("port");
const std::string& params::use_ssl      = std::string("use_ssl");
const std::string& params::mutual_auth  = std::string("mutual_auth");
const std::string& params::trusted_ca_list = std::string("trusted_ca_list");
const std::string& params::certificate  = std::string("certificate");
const std::string& params::privkey      = std::string("privkey");

const std::string& params::server_mode  = std::string("server_mode");
const std::string& params::local_server = std::string("local_server");
const std::string& params::local_port   = std::string("local_port");

const std::string& params::method       = std::string("method");
const std::string& params::uri          = std::string("uri");
const std::string& params::host         = std::string("host");
const std::string& params::content_type = std::string("content_type");

const std::string& params::codecs = std::string("codecs");

void params::convert(params &p_param, const std::string p_parameters) {
  // Sanity checks
  if (p_parameters.length() == 0) {
    return;
  }
  loggers::get_instance().log(">>> params::convert: '%s'", p_parameters.c_str());
  // Extract parameters
  try {
    std::regex           rgx("(\\w+)=(.*?)(,|$)");
    std::sregex_iterator begin(p_parameters.cbegin(), p_parameters.cend(), rgx);
    std::sregex_iterator end = std::sregex_iterator();
    for (std::sregex_iterator it = begin; it != end; ++it) {
      std::smatch m = *it;
      loggers::get_instance().log("params::convert: %d - %s - %s - %s - %s", m.size(), m[0].str().c_str(), m[1].str().c_str(), m[2].str().c_str(),
                                  m[3].str().c_str());
      p_param.insert(std::pair<std::string, std::string>(m[1].str(), m[2].str()));
    } // End of 'for' statement
  } catch (const std::logic_error &e) {
    p_param.clear();
  }
  loggers::get_instance().log("<<< params::convert");
}

void params::log() const {
  loggers::get_instance().log("params::log");
  if (size() == 0) {
    loggers::get_instance().log("\tEmpty");
  } else {
    for (const_iterator it = cbegin(); it != cend(); ++it) {
      loggers::get_instance().log("\t(%s, %s)", it->first.c_str(), it->second.c_str());
    } // End of 'for' statement
  }
}

void params::log() {
  loggers::get_instance().log("params::log");
  if (size() == 0) {
    loggers::get_instance().log("\tEmpty");
  } else {
    for (const_iterator it = cbegin(); it != cend(); ++it) {
      loggers::get_instance().log("\t(%s, %s)", it->first.c_str(), it->second.c_str());
    } // End of 'for' statement
  }
}

void params::reset() {
  loggers::get_instance().log("params::reset");
  for (iterator it = begin(); it != end(); ++it) {
    it->second.clear();
  } // End of 'for' statement
}
