/*!
 * \file      sctp_layer.cc
 * \brief     CC file for SCTP socket based protocol port layer.
 * \author    ETSI TTF041
 * \copyright ETSI Copyright Notification
 *            No part may be reproduced except as authorized by written permission.
 *            The copyright and the foregoing restriction extend to reproduction in all media.
 *            All rights reserved.
 * \version   0.1
 */

#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>
#include <chrono>

#include "sctp_layer_factory.hh"
#include "sctp.hh"
#include "loggers.hh"
#include "scoped_thread.hh"

sctp::SctpClient *client;
ScopedThread *receiverThread;

sctp_layer::sctp_layer(const std::string&  p_type, const std::string&  param) : layer(p_type), SSL_Socket(), PORT(p_type.c_str()), _params(), _client_id{-1}, _time_key("sctp_layer::Handle_Fd_Event_Readable"), _reconnect_on_send{false} {
  loggers::get_instance().log(">>> sctp_layer::sctp_layer (1): '%s', %s", to_string().c_str(), param.c_str());
  // Setup parameters
  params::convert(_params, param);
  _params.log();

  init();                                                                                                                                                   }

sctp_layer::sctp_layer(const std::string&  p_type, const params & param) : layer(p_type), SSL_Socket(), PORT(p_type.c_str()), _params(), _client_id{-1}, _time_key("sctp_layer::Handle_Fd_Event_Readable"), _reconnect_on_send{false} {
  loggers::get_instance().log(">>> sctp_layer::sctp_layer (2): '%s'", to_string().c_str());
  // Setup parameters
  _params = param;

  init();
}

void sctp_layer::init() {
  loggers::get_instance().log(">>> sctp_layer::init");

  set_socket_debugging(false);
  params::const_iterator it = _params.find(params::debug);
  if (it == _params.cend()) {
    _params.insert(std::pair<std::string, std::string>(std::string("debug"), "0"));
  } else if (it->second.compare("1") == 0) {
    set_socket_debugging(true);
  }
  /*it = _params.find(std::string("sctp_fragmented"));
  if (it == _params.cend()) {
    _params.insert(std::pair<std::string, std::string>(std::string("sctp_fragmented"), "0"));
  } else {
    _params.insert(std::pair<std::string, std::string>(std::string("sctp_fragmented"), "1"));
  }*/

  it = _params.find("src_ip");
  if (it == _params.cend()) {
    _params.insert(std::pair<std::string, std::string>(std::string("src_ip"), "127.0.0.1"));
  }
  it = _params.find("src_port");
  if (it == _params.cend()) {
    _params.insert(std::pair<std::string, std::string>(std::string("src_port"), "0")); // Dynamic binding requested
  }

  bool server_mode = false;
  it = _params.find(params::server_mode);
  if (it != _params.cend()) {
    server_mode = (1 == std::stoi(it->second));
  } else {
    _params.insert(std::pair<std::string, std::string>(std::string("server_mode"), "0"));
  }
  it = _params.find(params::server);
  if (it == _params.cend()) {
    _params.insert(std::pair<std::string, std::string>(std::string("server"), "127.0.0.1")); // TODO Try using params::server instead of std::string("server")
  }
  if (!parameter_set(params::server.c_str(), _params[params::server].c_str())) {
    loggers::get_instance().warning("sctp_layer::set_parameter: Unprocessed parameter: '%s'", params::server.c_str());
  }
  bool ssl_mode = false;
  /* = _params.find(params::use_ssl);
  if (it == _params.cend()) {
    _params.insert(std::pair<std::string, std::string>(std::string("use_ssl"), "0"));
  } else if (it->second.compare("1") == 0) {
    _params.insert(std::pair<std::string, std::string>(std::string("use_ssl"), "1"));
    ssl_mode = true;
  }
  set_ssl_use_ssl(ssl_mode);*/
  it = _params.find(params::port);
  if (it == _params.cend()) {
    if (_params[params::use_ssl].compare("0") == 0) { // Use standard HTTP port
      _params.insert(std::pair<std::string, std::string>(std::string("port"), "80"));
    } else { // Use standard HTTPS port
      _params.insert(std::pair<std::string, std::string>(std::string("port"), "443"));
    }
  }
  if (!parameter_set(remote_port_name(), _params[params::port].c_str())) {
    loggers::get_instance().warning("sctp_layer::set_parameter: Unprocessed parameter: '%s'", params::port.c_str());
  }
  it = _params.find(params::local_port);
  if (it == _params.cend()) {
    if (_params[params::use_ssl].compare("0") == 0) { // Use standard HTTP local_port
      _params.insert(std::pair<std::string, std::string>(std::string("local_port"), "80"));
    } else { // Use standard HTTPS local_port
      _params.insert(std::pair<std::string, std::string>(std::string("local_port"), "443"));
    }
  }
  if (!parameter_set(local_port_name(), _params[params::local_port].c_str())) {
    loggers::get_instance().warning("sctp_layer::set_parameter: Unprocessed parameter: '%s'", params::local_port.c_str());
  }
  
  if (!parameter_set(local_port_name(), _params[params::local_port].c_str())) {
    loggers::get_instance().warning("sctp_layer::set_parameter: Unprocessed parameter: '%s'", params::local_port.c_str());
  }

 it = _params.find("ppid");
  if (it == _params.cend()) {
    _params.insert(std::pair<std::string, std::string>(std::string("ppid"), "0")); // Dynamic binding requested
  }


  parameter_set(use_connection_ASPs_name(), (!server_mode) ? "yes" : "no");
  loggers::get_instance().warning("sctp_layer::set_parameter: Limit to one simultanneous accepted connection (server_backlog == 1)");
  parameter_set(server_backlog_name(), "1"); // Limit to one simultanneous accepted connection 
  loggers::get_instance().log("sctp_layer::init: server_mode=%x", server_mode);
  set_server_mode(server_mode);
  if (server_mode) {
    parameter_set("serverPort", _params[params::local_port].c_str());
  }
 /* if (ssl_mode) { // Add certificate bundle
    // Check mutual authentication param
    _params.insert(std::pair<std::string, std::string>(std::string("mutual_tls"), "0"));
    parameter_set(ssl_verifycertificate_name(), "no");
    it = _params.find(params::mutual_auth);
    if (it != _params.cend()) {
      if (_params[params::mutual_auth].compare("1") == 0) { // Use mutual authentication
        parameter_set(ssl_verifycertificate_name(), "yes");
        _params.insert(std::pair<std::string, std::string>(std::string("mutual_tls"), "1"));
      }
    }
    // Set trusted CA file
    it = _params.find(params::trusted_ca_list);
    if (it != _params.cend()) {
      parameter_set(ssl_trustedCAlist_file_name(), it->second.c_str());
    } else {
      // Use Let's Encrypt to generate your certificates
      // https://manpages.ubuntu.com/manpages/impish/en/man1/certbot.1.html
      loggers::get_instance().error("sctp_layer::set_parameter: Trusted CA list is required for TLS");
    }
    // Set additional certificates
    it = _params.find(params::privkey);
    if (it != _params.cend()) {
      parameter_set(ssl_private_key_file_name(), it->second.c_str());
    } else {
      // Use Let's Encrypt to generate your certificates
      // https://manpages.ubuntu.com/manpages/impish/en/man1/certbot.1.html
      loggers::get_instance().error("sctp_layer::set_parameter: Certificate private key is required for TLS");
    }
    it = _params.find(params::certificate);
    if (it != _params.cend()) {
      parameter_set(ssl_certificate_file_name(), it->second.c_str());
    } else {
      // Use Let's Encrypt to generate your certificates
      // https://manpages.ubuntu.com/manpages/impish/en/man1/certbot.1.html
      loggers::get_instance().error("sctp_layer::set_parameter: Certificate is required for TLS");
    }
  }*/
  set_ttcn_buffer_usercontrol(false);
  set_handle_half_close(true);

  map_user();

  //parameter_set(client_SCTP_reconnect_name(), "yes");

  if (server_mode == 0) {
    loggers::get_instance().log("sctp_layer::init: Establish connection: '%s'/%s", _params[params::server].c_str(), _params[params::port].c_str());
	  //open_client_connection(_params[params::server].c_str(), _params[params::port].c_str(), NULL, NULL);
	  //auto *client = new sctp::SctpClient(sctp::convert(_params["ppid"].c_str)) ;
	  /*auto*/ client = new sctp::SctpClient(sctp::PayloadProtocolId::NGAP,this) ;
	   client->bind(_params["src_ip"].c_str(), std::stoul(_params["src_port"].c_str()));
	   client->connect( _params[params::server].c_str(), std::stoul(_params[params::port].c_str(),nullptr,0));
	   
	   receiverThread = new ScopedThread( [](void *arg) { sctp::SctpClient::ReceiverThread(reinterpret_cast<sctp::SctpClient *>(arg)); }, (client));
	   //receiverThread = new ScopedThread( [](void *arg) { sctp::SctpClient::ReceiverThread(reinterpret_cast<std::pair<sctp::SctpClient *, sctp_layer *> *>(arg)); }, new std::pair<sctp::SctpClient *, sctp_layer *>(client,this));
	   
  }
}

sctp_layer::~sctp_layer() {
  loggers::get_instance().log(">>> sctp_layer::~sctp_layer: %d", _client_id);
  if (_client_id != -1) {
    remove_client(_client_id);
  }

  unmap_user();
}

void sctp_layer::Handle_Fd_Event(int fd, boolean is_readable, boolean is_writable, boolean is_error)
{
  loggers::get_instance().log(">>> sctp_layer::Handle_Fd_Event: %d - _client_id: %d", fd, _client_id);
  Handle_Socket_Event(fd, is_readable, is_writable, is_error);
  log_debug("<<< sctp_layer::Handle_Fd_Event");
}

void sctp_layer::Handle_Timeout(double time_since_last_call)
{
  loggers::get_instance().log(">>> sctp_layer::Handle_Timeout: %f", time_since_last_call);
  Handle_Timeout_Event(time_since_last_call);
  loggers::get_instance().log("<<< sctp_layer::Handle_Timeout");
}


////////////void sctp_layer::sendMsg(const NGAP__PDU__Descriptions::NGAP__PDU& p, params &p_params) {
  //loggers::get_instance().log(">>> sctp_layer::sendMsg");
  // params.log();

  // Encode PDU
/////////////  OCTETSTRING data = int2oct(0,1);
  //_codec.encode(p.msgOut(), data);
////////////  send_data(data, p_params);
//////////////}


void sctp_layer::send_data(OCTETSTRING& data, params& params) {
  //loggers::get_instance().log_msg(">>> sctp_layer::send_data: ", data);

  //loggers::get_instance().log("sctp_layer::send_data: SSL mode: %x", get_ssl_use_ssl());
  //loggers::get_instance().log("sctp_layer::send_data: server_mode: '%s'", _params[params::server_mode].c_str());
  //loggers::get_instance().log("sctp_layer::send_data: peer_list_get_nr_of_peers: %d", peer_list_get_nr_of_peers());

  /*if ((_params[params::server_mode].compare("0") == 0) && (peer_list_get_nr_of_peers() == 0)) {
    // Reconnect (e.g. HTTP connection lost
    loggers::get_instance().log("sctp_layer::send_data: Re-establish connection: '%s'/%s", _params[params::server].c_str(), _params[params::port].c_str());
	  open_client_connection(_params[params::server].c_str(), _params[params::port].c_str(), NULL, NULL);
  }*/
  //send_outgoing(static_cast<const unsigned char*>(data), data.lengthof(), _client_id);
  
  //client->send(  static_cast<const unsigned char*>(data),data.lengthof());
  client->send(  static_cast<const uint8_t*>(data),data.lengthof());

}

void sctp_layer::receive_data(OCTETSTRING& data, params& params) {
  //loggers::get_instance().log_msg(">>> sctp_layer::receive_data: ", data);

  //NGAP__PDU__Descriptions::NGAP__PDU ngap_pdu;
  //incomming_message(ngap_pdu);
  receive_to_all_layers(data, params);
  //loggers::get_instance().log(">>> sctp_layer::receive_data:to_all_layers... ");
}

void sctp_layer::message_incoming(const unsigned char* message_buffer, int length, int client_id) {
  //loggers::get_instance().log(">>> sctp_layer::message_incoming");
  //loggers::get_instance().log_to_hexa("sctp_layer::message_incoming: ", message_buffer, length);

  //float duration;
  //loggers::get_instance().set_start_time(_time_key);
  OCTETSTRING data(length, message_buffer);
  params params;
  params.insert(std::pair<std::string, std::string>(
    std::string("timestamp"),
    std::to_string(std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now().time_since_epoch()).count())));
  this->receive_data(data, params); // TODO Check execution time for decoding operation
  //loggers::get_instance().set_stop_time(_time_key, duration);
}

void sctp_layer::client_connection_opened(int p_client_id)
{
  loggers::get_instance().log(">>> sctp_layer::client_connection_opened - _client_id: %d: %d", p_client_id, _client_id);
}

bool sctp_layer::add_user_data(int p_client_id)
{
  loggers::get_instance().log(">>> sctp_layer::add_user_data: %d - _client_id: %d", p_client_id, _client_id);
  _client_id = p_client_id;
  if (_params[params::use_ssl].compare("0") == 0) {
    loggers::get_instance().log("sctp_layer::add_user_data: Non secured mode");
    return Abstract_Socket::add_user_data(p_client_id);
  }
  loggers::get_instance().log("sctp_layer::add_user_data: SSL mode");
  return SSL_Socket::add_user_data(p_client_id);
}

int sctp_layer::send_message_on_fd(int p_client_id, const unsigned char * message_buffer, int length_of_message)
{
  loggers::get_instance().log(">>> sctp_layer::send_message_on_fd: %d", p_client_id);

  if(get_user_data(p_client_id)) {
    loggers::get_instance().log("sctp_layer::send_message_on_fd: SSL mode");
    return SSL_Socket::send_message_on_fd(p_client_id, message_buffer, length_of_message);
  }

  loggers::get_instance().log("sctp_layer::send_message_on_fd: Non secured mode");
  return Abstract_Socket::send_message_on_fd(p_client_id, message_buffer, length_of_message);
}

int sctp_layer::send_message_on_nonblocking_fd(int p_client_id, const unsigned char * message_buffer, int length_of_message)
{
	loggers::get_instance().log(">>> sctp_layer::send_message_on_nonblocking_fd: %d", p_client_id);

  if(get_user_data(p_client_id)) {
    loggers::get_instance().log("sctp_layer::send_message_on_nonblocking_fd: SSL mode");
    return SSL_Socket::send_message_on_nonblocking_fd(p_client_id, message_buffer, length_of_message);
  }

  loggers::get_instance().log("sctp_layer::send_message_on_nonblocking_fd: Non secured mode");
  return Abstract_Socket::send_message_on_nonblocking_fd(p_client_id, message_buffer, length_of_message);
}

int sctp_layer::receive_message_on_fd(int p_client_id)
{
	loggers::get_instance().log(">>> sctp_layer::receive_message_on_fd: %d", p_client_id);

  if(get_user_data(p_client_id)) {
    // INFO: it is assumed that only SSL_Socket assigns user data to each peer
    loggers::get_instance().log("sctp_layer::receive_message_on_fd: SSL mode");
    return SSL_Socket::receive_message_on_fd(p_client_id);
  }

  loggers::get_instance().log("sctp_layer::receive_message_on_fd: Non secured mode");
  if (_params[std::string("sctp_fragmented")].compare("1") == 0) {
    sleep(4); // FIXME When HTTP paquet is fragmented into several SCTP packets, a timer is required. This is a Q&D solution
  }
  return Abstract_Socket::receive_message_on_fd(p_client_id);
}

void sctp_layer::peer_disconnected(int p_client_id)
{
	loggers::get_instance().log(">>> sctp_layer::peer_disconnected: %d", p_client_id);

  Abstract_Socket::peer_disconnected(p_client_id);
	_client_id = -1;
}

sctp_layer_factory sctp_layer_factory::_f;

