/*!
 * \file      params.hh
 * \brief     Header file for the parameter dictionary.
 * \author    ETSI STF525
 * \copyright ETSI Copyright Notification
 *            No part may be reproduced except as authorized by written permission.
 *            The copyright and the foregoing restriction extend to reproduction in all media.
 *            All rights reserved.
 * \version   0.1
 */
#pragma once

#include <map>
#include <string>
#include <vector>

/*!
 * \class params
 * \brief This class provides basic functionalities for an ITS dictionary
 * \implements std::map
 */
class params : public std::map<std::string, std::string> {
public: //! \publicsection
  // TODO Use static constexpr (see commsignia_layer.hh)
  static const std::string& debug;        //! Set to 1 to enable the debug mode

  static const std::string& loopback;

  static const std::string& timestamp;    //! Packet reception timestamp
  static const std::string& mac_src;      //! Source MAC address parameter name
  static const std::string& mac_dst;      //! Destination MAC address parameter name
  static const std::string& mac_bc;       //! Broadcast MAC address parameter name
  static const std::string& eth_type;     //! Ethernet type parameter name
  static const std::string& filter;       //! Additinal PCAP filter

  static const std::string& nic;          //! Network Interface Card parameter name

  static const std::string& server;       //! HTTP server address (e.g. www.etsi.org)
  static const std::string& port;         //! HTTP server port. Default: 80
  static const std::string& use_ssl;      //! Set to 1 to use SSL to communicate with the HTTP server. Default: false
  static const std::string& mutual_auth;  //! Set to 1 to use mutual authentication. Default: false
  static const std::string& trusted_ca_list; //! List of trusted CA certificates
  static const std::string& certificate;  //! Chain of certificates 
  static const std::string& privkey;      //! Certificate private key
  static const std::string& server_mode;  //! Does the test sytem acting as a server. Default: 0
  static const std::string& local_port;   //! Local listener port. Default: 80

  static const std::string& method;       //! HTTP method type. Default: POST
  static const std::string& uri;          //! HTTP URI value. Default: /
  static const std::string& host;         //! HTTP Host value. Default: 127.0.0.1
  static const std::string& content_type; //! HTTP Content-type value. Default: application/text

  static const std::string& codecs; //! List of codecs to use for HTTP application layers

  /*!
   * \brief Default constructor
   *        Create a new instance of the params class
   */
  params() : std::map<std::string, std::string>(){};
  /*!
   * \brief Copy constructor
   *        Clone an existing instance of a params object
   * \param[in] p_params An existing instance of a params object
   */
  explicit params(const params &p_params) : std::map<std::string, std::string>(p_params.begin(), p_params.end()){};

  /*!
   * \brief Default destructor
   */
  virtual ~params(){};

  /*!
   * \fn void log() const;
   * \brief Provides a dump of the content of this instance
   */
  void log() const;
  /*!
   * \fn void log() const;
   * \brief Provides a dump of the content of this instance
   */
  void log();
  /*!
   * \fn void reset();
   * \brief Reset the content of this instance
   */
  void reset();

  /*!
   * \static
   * \fn void convert(params& p_param, const std::string p_parameters);
   * \brief Create a new instance of a params object by converting a list of ITS parameters in string format (t1=v1,T2=(v0,v1v2)...)
   * \return a new instance of a params object
   */
  static void convert(params &p_param, const std::string p_parameters);
}; // End of class params
