"""
Tool entry point
"""

import sys
import re
import argparse
from bs4 import BeautifulSoup
from test_entry import TestEntry
from write_excel import ExcelWriter



class TestOutputParser:
    """
    Parser taking a file name (of the XML output from robot),
    and extracts relevant information, eventually creating a list
    of TestEntry objects
    """
    def __init__(self, input_file, output_file):
        self.test_entries = []
        self.api = ""
        self.output_file = output_file
        self.robot_file_contents = ""
        self.load_file(input_file)

    def load_file(self, fname):
        """
        Load xml file
        """
        with open(fname, "r", encoding="utf8") as robot_file:
            self.robot_file_contents = robot_file.read()

        if self.robot_file_contents == "":
            print("Empty file {}".format(fname))
            sys.exit(-1)

    def run_parser(self):
        """
        Run parser, extracting all info to create the test entries
        """

        soup = BeautifulSoup(self.robot_file_contents, "lxml")

        # Suite information
        suite = soup.find("suite")
        path = suite["source"]

        parts = path.split("\\")
        # Extract info for test entries
        self.api = parts[len(parts) - 2]
        self.robot_file_contents = parts[len(parts) - 1]

        # Tests
        tests = soup.find_all("test")
        for test in tests:
            self.test_entries.append(self.create_test_entry(test))

        # Write tests
        excel_writer = ExcelWriter(self.output_file)
        for entry in self.test_entries:
            excel_writer.write_test_entry(entry)

        # Write chart
        excel_writer.write_pie_chart_data()

        # And finally, save the work book
        excel_writer.save()

    def create_test_entry(self, xml_obj):
        """
        Takes the xml entry corresponding to the test from the output file,
        and returns a TestEntry object with the relevant information extracted.
        """
        # retrieve ID and name
        id_raw = xml_obj.find("doc", recursive=False).contents
        match_group = re.search(r"Test ID: ([0-9\.]*)$", id_raw[0].string, re.MULTILINE)
        test_id = match_group.group(1)
        name = xml_obj["name"]

        #retrieve status and error message (if FAIL)
        status_obj = xml_obj.find("status", recursive=False)
        cts = status_obj.contents
        error_msg = cts[0] if len(cts) > 0 else ""

        result = status_obj["status"]
        return TestEntry(test_id, name, (result, error_msg), (self.api, self.robot_file_contents))

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description='Robot Test Reporter written in Python.\n'
                                                 'Produces an xlsx file from a robot XML '
                                                 'output file.\n'
                                                 'The command outputs to a new xlslx file if it '
                                                 'does not exist, or appends to an existing one.')
    parser.add_argument('input_file', help='XML file generated by Robot')
    parser.add_argument('-o', dest='output_file', default='testResults.xlsx',
                        help='output file name (default: testResults.xlsx)')

    args = parser.parse_args()

    TestOutputParser(args.input_file, args.output_file).run_parser()
