from openpyxl import Workbook, load_workbook
from openpyxl.utils import get_column_letter
from openpyxl.styles import Font
from openpyxl.styles.fills import Stop, PatternFill

fname = 'testResults.xlsx'

class ExcelWriter:
    """
    Utility class which writes `TestEntry` objects to an excel file.
    Creates file if it does not exist.
    """
    PASS_COL = "00FF00"
    FAIL_COL = "FF0000"
    def __init__(self):
        self.wb = getWorkbook()
        self.ws = self.wb.active
    
    def get_entry_with_id(self, id):
        """
        When inserting an entry, it might be a test being re-run. In that case,
        the row containing that test needs to be updated, as opposed to appending
        the entry to the end of the file.
        This method finds that row, returning the row number,
        or -1 in the case where that test is not in the report yet.
        """
        for cell in self.ws["A"]:
            if cell.value is None:
                return -1
            elif cell.value == id:
                print("MATCH with id {} at row {}".format(id, cell.row))
                return cell.row
        return -1
    
    def get_last_row(self):
        for cell in self.ws["A"]:
            if cell.value is None:
                return cell.row
        return cell.row + 1
    
    def writeTestEntry(self, test_entry):
        existing_entry_row = self.get_entry_with_id(test_entry.id)
        last_row = self.get_last_row()
        # Use the above two values to pick a row
        entry_row = existing_entry_row if existing_entry_row != -1 else last_row

        # Pick a cell color based on test outcome
        cell_col = ExcelWriter.PASS_COL if test_entry.result == "PASS" else ExcelWriter.FAIL_COL

        # Test entry as a list
        entry_vals = test_entry.asList()

        for col, cell_vale in zip(self.ws.iter_cols(min_row=entry_row, max_col=len(entry_vals), max_row=entry_row), entry_vals):
            for cell in col:
                cell.value = cell_vale
                cell.fill = PatternFill("solid", fgColor=cell_col)# cellCol
        
    
    def save(self):
        self.wb.save(filename = fname)

def initWorkbook(wb):
    """
    Writes column headers to ws
    """
    headers = [("Test ID", 10), ("Test name", 80), ("Result", 6), ("Error Message", 100), ("NFV API", 25), ("Robot Test File", 25)]
    headerFont = Font(bold=True)
    ws = wb.active
    for col, header in zip(ws.iter_cols(min_row=1, max_col=len(headers), max_row=1), headers):
        for cell in col:
            headerName = header[0]
            colsize = header[1]
            ws.column_dimensions[get_column_letter(cell.column)].width = colsize
            cell.value = headerName
            cell.font = headerFont


def getWorkbook():
    try:
        wb = load_workbook(filename = fname)
        return wb
    except FileNotFoundError:
        wb = Workbook()
        initWorkbook(wb)
        return wb