from bs4 import BeautifulSoup
from sys import argv
from testEntry import TestEntry
from writeExcel import ExcelWriter
import re
import argparse

usage_str = """
=====================================
Robot Test Reporter written in Python
=====================================

Usage
    
    python parseTestResults.py output.xml 
    
            where output.xml is the xml file generated by robot


The command outputs to a new xlslx file if it does not exist, or 
appends to an existing one.
"""

class TestOutputParser:
    def __init__(self, fname):
        self.test_entries = []
        self.load_file(fname)

    def load_file(self, fname):
        self.contents = ""

        with open(fname, "r", encoding="utf8") as f:
            self.contents = f.read()

        if self.contents == "":
            print("Empty file {}".format(fname))
            exit(-1)

    def run_parser(self):
        soup = BeautifulSoup(self.contents, "lxml")

        # Suite information
        suite = soup.find("suite")
        path = suite["source"]
        # TODO This might be an issue later on. In Unix-style paths the separator is a forward slash
        parts = path.split("\\")
        # Extract info for test entries
        self.api = parts[len(parts) - 2]
        self.robot_file = parts[len(parts) - 1]

        # Tests
        tests = soup.find_all("test")
        for test in tests:
            self.test_entries.append(self.create_test_entry(test))

        # Write tests
        ew = ExcelWriter()
        for entry in self.test_entries:
            ew.write_test_entry(entry)
        ew.save()

    def create_test_entry(self, xml_obj):
        """
        Takes the xml entry corresponding to the test from the output file,
        and returns a TestEntry object with the relevant information extracted.
        """
        # retrieve ID and name
        id_raw = xml_obj.find("doc", recursive=False).contents
        mg = re.search(r"Test ID: ([0-9\.]*)$", id_raw[0].string, re.MULTILINE)
        test_id = mg.group(1)
        name = xml_obj["name"]

        #retrieve status and error message (if FAIL)
        statusObj = xml_obj.find("status", recursive=False)
        cts = statusObj.contents
        error_msg = cts[0] if len(cts) > 0 else ""

        result = statusObj["status"]
        return TestEntry(test_id, name, result, error_msg, self.api, self.robot_file)

def display_usage():
    print(usage_str)

if __name__ == "__main__":
    usage_str = """
=====================================
Robot Test Reporter written in Python
=====================================

Usage
    
    python parseTestResults.py output.xml 
    
            where output.xml is the xml file generated by robot


The command outputs to a new xlslx file if it does not exist, or 
appends to an existing one.
"""
    parser = argparse.ArgumentParser(description='Process some integers.')
    parser.add_argument('integers', metavar='N', type=int, nargs='+',
                        help='an integer for the accumulator')
    parser.add_argument('-o', dest='fname', action='store_const',
                        default='testResults.xlsx',
                        help='output file name (default: testResults.xlsx)')

    args = parser.parse_args()

    TestOutputParser(argv[1]).run_parser()
