﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using System.Net.NetworkInformation;

using MongoDB.Driver;
using MongoDB.Bson;
using MongoDB.Bson.Serialization;
using MongoDB.Bson.Serialization.Serializers;
using ETSI.ARF.OpenAPI.WorldStorage.Models;
using Microsoft.AspNetCore.Mvc;


#pragma warning disable CS1591 // Fehlendes XML-Kommentar für öffentlich sichtbaren Typ oder Element
namespace ETSI.ARF.OpenAPI.WorldStorage.Services
{
    abstract public class BaseService<T>
    {
        // The object is created by first access? so the singleton may be null if nothing here was called before
        static public BaseService<T> Singleton => _singleton;
        static protected BaseService<T> _singleton = null;

        public IDatabaseSettings Settings => _settings;
        protected IDatabaseSettings _settings;

        protected string mongoServer;
        protected MongoClient mongoClient;
        protected IMongoDatabase mongoDatabase;
        protected IMongoCollection<T> mongoCollection;

        public BaseService(IDatabaseSettings settings)
        {
            // Store the singleton
            if (_singleton != null) throw new Exception("Service can only be instantiated one time!");
            else _singleton = this;

            _settings = settings;

            // Open the mongo database
            //BsonSerializer.RegisterSerializer(new GuidSerializer(GuidRepresentation.Standard));
            mongoServer = SelectMongoServer(settings);
            mongoClient = new MongoClient(mongoServer);
            mongoDatabase = mongoClient.GetDatabase(settings.DatabaseName);
        }

        ~BaseService()
        {
            // to do?
        }

        // Check manually if a GUID document already exists!
        public bool CheckIfExist(Guid GUID)
        {
            foreach (var obj in Get())
            {
                if (((Models.IModels)obj).GUID == GUID) return true;
            }
            return false;
        }

        public long NumOfDocuments()
        {
            if (mongoCollection == null) return -1;
            else return mongoCollection.CountDocuments(new BsonDocument());
        }

        private void _print(T obj)
        {
            Console.WriteLine("[Mongo Server] User # " + ((Models.IModels)obj).GUID);
        }

        public void PrintAll()
        {
            // Execute a _print to all elements
            var a = mongoCollection.Find(_ => true);
            a.ForEachAsync(_print).Wait();
        }

        public T Create(T obj)
        {
            // Test first if GUID doesn't exist
            bool exist = CheckIfExist(((Models.IModels)obj).GUID);
            if (!exist)
            {
                mongoCollection.InsertOne(obj);
                return obj;
            }
            else return default(T);
        }

        //
        // Get the current mongo _id of the object in the database
        //
        #region Get current mongo ID or object/document
        private Guid _newObjGUID;
        private T _currentObj;
        private ObjectId _currentMongoID;
        private ObjectId getMongoIDFromName(string name)
        {
            var res = mongoCollection.Find(obj => ((Models.IModels)obj).Name == name).FirstOrDefault();
            if (res == null) return ObjectId.Empty;
            else return ((Models.IModels)res)._mongoID;
        }

        private ObjectId getMongoIDFromGUID(Guid GUID)
        {
            _newObjGUID = GUID;
            _currentMongoID = ObjectId.Empty;
            var i = mongoCollection.Find(_ => true);
            i.ForEachAsync(_compareAndRemember).Wait();
            return _currentMongoID;
        }

        private void _compareAndRemember(T obj)
        {
            if (_currentMongoID != ObjectId.Empty) return; // take the first one
            if (((Models.IModels)obj).GUID == _newObjGUID)
            {
                _currentObj = obj;
                _currentMongoID = ((Models.IModels)obj)._mongoID;
            }
        }
        #endregion

        //abstract public long Replace(T obj);
        public long Replace(T obj) //=> assetCollection.ReplaceOne(asset => asset.GUID == GUID, obj);
        {
            // Which mongo document?
            ObjectId mongoID = getMongoIDFromGUID(((Models.IModels)obj).GUID);

            // Object was found, so replace it using the _id
            if (mongoID != ObjectId.Empty)
            {
                ((Models.IModels)obj)._mongoID = mongoID;
                return mongoCollection.ReplaceOne(o => ((Models.IModels)o)._mongoID == mongoID, obj).MatchedCount;
            }
            else return 0;
        }

        public long Remove(T obj) => Remove(((Models.IModels)obj).GUID);

        public long Remove(Guid GUID)
        {
            //return mongoCollection.DeleteOne(o => ((Models.IModels)o).GUID == GUID);

            // Which mongo document?
            ObjectId mongoID = getMongoIDFromGUID(GUID);

            // Object was found, so replace it using the _id
            if (mongoID != ObjectId.Empty)
            {
                // todo: check if the user is allowed to delete the object!

                return mongoCollection.DeleteOne(o => ((Models.IModels)o)._mongoID == mongoID).DeletedCount;
            }
            else return 0;
        }

        // Return all documents
        public List<T> Get() => mongoCollection.Find(_ => true).ToList();

        // Limit the result to the first n documents
        public List<T> Get(int limit) => mongoCollection.Find(_ => true).Limit(limit).ToList();

        // Search manually the document with the id GUID
        public T Get(string name) => mongoCollection.Find(obj => ((Models.IModels)obj).Name == name).FirstOrDefault();

        // Geht nicht! public override T Get(Guid GUID) => mongoCollection.Find<T>(o => o.GUID == GUID).FirstOrDefault();
        // Work around: look in all elements if the GUIDs are equals
        public T Get(Guid GUID)
        {
            foreach (T obj in Get())
            {
                if (((Models.IModels)obj).GUID == GUID) return obj;
            }
            return default(T);
        }

        //
        // Select the server or use a local database
        //
        private string SelectMongoServer(IDatabaseSettings settings)
        {
            string server = null;
            Ping myPing = new Ping();
            try
            {
                PingReply reply = myPing.Send(settings.MongoSrv, 1000);
                if (reply != null)
                {
                    server = settings.MongoSrv;
                    Console.WriteLine("[Mongo Server] Status: " + reply.Status + " \n Time: " + reply.RoundtripTime.ToString() + " \n Address: " + reply.Address);
                }
            }
            catch (PingException e)
            {
                server = null;
                Console.WriteLine("[Mongo Server] Ping: " + e.ToString());
            }

            if (server is null)
            {
                // Use the local mongo server
                server = "localhost";
                Console.WriteLine("[Mongo Server] Take localhost");
            }
            server = "mongodb://" + server + ":" + settings.MongoPort + "/";
            mongoServer = server;
            return server;
        }
    }
}
#pragma warning restore CS1591 // Fehlendes XML-Kommentar für öffentlich sichtbaren Typ oder Element
