//
// ARF - Augmented Reality Framework (ETSI ISG ARF)
//
// Copyright 2024 ETSI
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Last change: March 2024
//

using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using UnityEngine;

using ETSI.ARF.OpenAPI;
using ETSI.ARF.OpenAPI.WorldStorage;

//#if UNITY_EDITOR
namespace ETSI.ARF.WorldStorage.REST
{
    public class WorldAnchorRequest : RequestBase<WorldAnchor>
    {
        //
        // Wrapper for the endpoints
        //
        static public WorldAnchor GetWorldAnchorSync(WorldStorageServer ws, Guid UUID)
        {
            wsServer = ws;
            var httpClient = new BasicHTTPClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Request WorldAnchor {UUID}...");
            return apiClient.GetWorldAnchorById(token, UUID);
        }

        static public ResponseObject<WorldAnchor> GetWorldAnchorAsync(WorldStorageServer ws, Guid UUID, Action<ResponseObject<WorldAnchor>> func)
        {
            wsServer = ws;
            var httpClient = new UnityWebRequestHttpClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Request WorldAnchor {UUID}...");
            ResponseObject<WorldAnchor> ro = new ResponseObject<WorldAnchor>("Request WorldAnchor " + UUID.ToString(), func);
            apiClient.GetWorldAnchorByIdAsync(token, UUID, ro.cancellationToken).ContinueWith(OnReceiveObject<WorldAnchor>, ro);
            return ro;
        }

        static public List<WorldAnchor> GetWorldAnchorsSync(WorldStorageServer ws)
        {
            wsServer = ws;
            var httpClient = new BasicHTTPClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log("[REST] Request WorldAnchors...");
            return apiClient.GetWorldAnchors(token).WorldAnchors as List<WorldAnchor>;
        }

        static public ResponseObject<WorldAnchorsResponse> GetWorldAnchorsAsync(WorldStorageServer ws, Action<ResponseObject<WorldAnchorsResponse>> func)
        {
            wsServer = ws;
            var httpClient = new UnityWebRequestHttpClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log("[REST] Request WorldAnchors...");
            ResponseObject<WorldAnchorsResponse> ro = new ResponseObject<WorldAnchorsResponse>("Request WorldAnchors", func);
            apiClient.GetWorldAnchorsAsync(token, ro.cancellationToken).ContinueWith(OnReceiveListOfObjects<WorldAnchorsResponse>, ro);
            return ro;
        }

        static public string CreateWorldAnchorSync(WorldStorageServer ws, WorldAnchor worldAnchor)
        {
            wsServer = ws;
            var httpClient = new BasicHTTPClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            // Add some management stuffs
            if (worldAnchor.UUID == Guid.Empty) worldAnchor.UUID = Guid.NewGuid();
            if (worldAnchor.CreatorUUID == Guid.Empty) worldAnchor.CreatorUUID = System.Guid.Parse("8fb169e2-8910-4cd5-a8f9-b7abff38d013");

            Debug.Log($"[REST] Create WorldAnchor {worldAnchor.UUID}...");
            return apiClient.AddWorldAnchor(token, worldAnchor).Message;
        }

        static public ResponseObject<Success> CreateWorldAnchorAsync(WorldStorageServer ws, WorldAnchor worldAnchor, Action<ResponseObject<Success>> func)
        {
            wsServer = ws;
            var httpClient = new UnityWebRequestHttpClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            // Add some management stuffs
            if (worldAnchor.UUID == Guid.Empty) worldAnchor.UUID = Guid.NewGuid();
            if (worldAnchor.CreatorUUID == Guid.Empty) worldAnchor.CreatorUUID = System.Guid.Parse("8fb169e2-8910-4cd5-a8f9-b7abff38d013");

            Debug.Log($"[REST] Create WorldAnchor {worldAnchor.UUID}...");
            ResponseObject<Success> ro = new ResponseObject<Success>("Create WorldAnchor " + worldAnchor.Name + " (no UUID)", func);
            apiClient.AddWorldAnchorAsync(token, worldAnchor, ro.cancellationToken).ContinueWith(OnReceiveObject<Success>, ro);
            return ro;
        }

        static public string UpdateWorldAnchorSync(WorldStorageServer ws, WorldAnchor worldAnchor)
        {
            wsServer = ws;
            var httpClient = new BasicHTTPClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Update WorldAnchor {worldAnchor.UUID}...");
            return apiClient.ModifyWorldAnchor(token, worldAnchor).Message;
        }

        static public ResponseObject<Success> UpdateWorldAnchorAsync(WorldStorageServer ws, WorldAnchor worldAnchor, Action<ResponseObject<Success>> func)
        {
            wsServer = ws;
            var httpClient = new UnityWebRequestHttpClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Update WorldAnchor {worldAnchor.UUID}...");
            ResponseObject<Success> ro = new ResponseObject<Success>("Update WorldAnchor " + worldAnchor.UUID.ToString(), func);
            apiClient.ModifyWorldAnchorAsync(token, worldAnchor, ro.cancellationToken).ContinueWith(OnReceiveObject<Success>, ro);
            return ro;
        }

        static public string DeleteWorldAnchorSync(WorldStorageServer ws, Guid UUID)
        {
            wsServer = ws;
            var httpClient = new BasicHTTPClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Delete WorldAnchor {UUID}...");
            return apiClient.DeleteWorldAnchor(token, UUID).Message;
        }

        static public ResponseObject<Success> DeleteWorldAnchorAsync(WorldStorageServer ws, Guid UUID, Action<ResponseObject<Success>> func)
        {
            wsServer = ws;
            var httpClient = new UnityWebRequestHttpClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Delete WorldAnchor {UUID}...");
            ResponseObject<Success> ro = new ResponseObject<Success>("Delete WorldAnchor " + UUID.ToString(), func);
            apiClient.DeleteWorldAnchorAsync(token, UUID, ro.cancellationToken).ContinueWith(OnReceiveObject<Success>, ro);
            return ro;
        }
    }
}
//#endif