//
// ARF - Augmented Reality Framework (ETSI ISG ARF)
//
// Copyright 2024 ETSI
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Last change: June 2024
//

#define isDEBUG

using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;

using UnityEngine;
using UnityEditor;
using TMPro;

using ETSI.ARF.WorldStorage.REST;
using ETSI.ARF.OpenAPI.WorldStorage;

namespace ETSI.ARF.WorldStorage.UI
{
    public class TrackableWindow : BaseWindow<Trackable>
    {
        static public TrackableWindow winSingleton;

        // Trackable params
        string UUID = System.Guid.Empty.ToString();
        string customName = "NotDefined";
        string creatorUUID = System.Guid.Empty.ToString();
        double confidence = 0f;
        TrackableType type = TrackableType.OTHER;
        EncodingInformationStructureDataFormat format = EncodingInformationStructureDataFormat.ARCORE;
        float versionEncoding = 1.01f; 
        UnitSystem unit = UnitSystem.CM;
        Vector3 trackableSize;
        Vector3 localCRS_pos;
        Vector3 localCRS_rot;
        byte[] trackablePayload = new byte[1] { 0 };

        //graph params to generate the node
        public bool useCoord;
        public float nodePosX = 0;
        public float nodePosY = 0;

        public TrackableWindow()
        {
            // init somne stuffs
        }

        public static void ShowWindow(WorldStorageServer ws, WorldStorageUser user, string UUID = "")
        {
            winSingleton = EditorWindow.GetWindow(typeof(TrackableWindow), false, "ETSI ARF - Trackable") as TrackableWindow;
            winSingleton.worldStorageServer = ws;
            winSingleton.worldStorageUser = user;
            if (!string.IsNullOrEmpty(UUID))
            {
                winSingleton.saveText = "Update";
                winSingleton.UUID = UUID;
                winSingleton.GetParams();
            }
            else
            {
                // Create new one
                winSingleton.saveText = "Create";
                winSingleton.AddObject();
            }
        }

        public static GameObject GenerateAndUpdateVisual(string UUID, string name, Vector3 pos, Vector3 rot)
        {
            ETSI.ARF.WorldStorage.UI.Prefabs.WorldStoragePrefabs prefabs;
            prefabs = (Prefabs.WorldStoragePrefabs)Resources.Load("ARFPrefabs");
            GameObject arf = GameObject.Find("ARF Visuals");
            GameObject visual = GameObject.Find(UUID);

            if (arf == null) arf = new GameObject("ARF Visuals");
            if (visual == null)
            {
                visual = SceneAsset.Instantiate<GameObject>(prefabs.trackablePrefab, pos, Quaternion.Euler(rot), arf.transform); // TODO rot
                visual.name = UUID;
            }
            else
            {
                visual.transform.SetPositionAndRotation(pos, Quaternion.Euler(rot));
            }
            visual.transform.Find("Canvas/Text").GetComponent<TextMeshProUGUI>().text = $"Name: { name }\nUUID: { UUID }";
            return visual;
        }

        void OnGUI()
        {
            ori = GUI.backgroundColor; // remember ori color

            gsTest = new GUIStyle("window");
            //gsTest.normal.textColor = WorldStorageWindow.arfColors[0];
            gsTest.fontStyle = FontStyle.Bold;
            gsTest.alignment = TextAnchor.UpperLeft;
            gsTest.fontSize = 16;

            scrollPos = EditorGUILayout.BeginScrollView(scrollPos, GUILayout.ExpandWidth(true));
            WorldStorageWindow.DrawCopyright();

            DrawUIStuffs();

            EditorGUILayout.EndScrollView();

            if (GUILayout.Button("Close Window"))
            {
                Close();
            }
        }

        public override void DrawUIStuffs()// Trackable trackable)
        {
            GUILayout.BeginVertical(); // "Trackable Editor", gsTest);
            EditorGUILayout.Space();

            GUILayout.BeginHorizontal();
            GUI.backgroundColor = WorldStorageWindow.arfColors[7];
            Texture trackableImage = (Texture)AssetDatabase.LoadAssetAtPath("Packages/etsi.isg.arf.worldstorage/Editor/Images/trackable.png", typeof(Texture));
            GUILayout.Box(trackableImage, GUILayout.Width(24), GUILayout.Height(24));
            GUI.backgroundColor = ori;
            GUILayout.Label("Trackable Parameters:", EditorStyles.whiteBoldLabel);
            GUILayout.EndHorizontal();

            Rect rect = EditorGUILayout.GetControlRect(false, WorldStorageWindow.lineH);
            EditorGUI.DrawRect(rect, WorldStorageWindow.arfColors[7]);

            //
            GUILayout.Label("Server: " + worldStorageServer.serverName, EditorStyles.whiteLargeLabel);
            GUILayout.Label("User: " + worldStorageUser.userName, EditorStyles.whiteLargeLabel);
            EditorGUILayout.Space();

#if isDEBUG
            GUILayout.Label("UUID: " + UUID, EditorStyles.miniLabel); // readonly
            GUILayout.Label("Creator UID: " + creatorUUID, EditorStyles.miniLabel); // readonly
            EditorGUILayout.Space();
#endif
            customName = EditorGUILayout.TextField("Name of Trackable:", customName);
            EditorGUILayout.Space();

            // ---------------------
            // Toolbar
            // ---------------------
            EditorGUILayout.BeginHorizontal();
            GUI.backgroundColor = WorldStorageWindow.arfColors[2];
            if (GUILayout.Button(saveText))
            {
                Debug.Log("PUT Trackable");

                if (!string.IsNullOrEmpty(UUID) && UUID != "0" && UUID != System.Guid.Empty.ToString())
                {
                    Trackable obj = GenerateObject();
                    TrackableRequest.UpdateTrackableAsync(worldStorageServer, obj, (response) =>
                    {
                        UUID = response.result.Message;
                        UUID = UUID.Trim('"'); //Bugfix: remove " from server return value

                        if (WorldStorageWindow.WorldStorageWindowSingleton != null)
                        {
                            WorldStorageWindow.WorldStorageWindowSingleton.GetTrackables();
                        }
                        Close();
                    });
                }
            }

            GUI.backgroundColor = WorldStorageWindow.arfColors[3];
            if (GUILayout.Button("Delete"))
            {
                if (EditorUtility.DisplayDialog("Delete", "Are you sure you want to delete this Trackable?", "Delete", "Cancel"))
                {
                    Debug.Log("Delete Trackable");
                    TrackableRequest.DeleteTrackableAsync(worldStorageServer, Guid.Parse(UUID), (response) =>
                    {
                        UUID = System.Guid.Empty.ToString();
                        customName = "Warning: Object deleted !";
                        creatorUUID = System.Guid.Empty.ToString();
                        confidence = 0f;
                        type = TrackableType.OTHER;
                        unit = UnitSystem.CM;
                        if (WorldStorageWindow.WorldStorageWindowSingleton != null)
                        {
                            WorldStorageWindow.WorldStorageWindowSingleton.GetTrackables();
                        }
                        Close();
                    });
                }
            }
            GUI.backgroundColor = ori;

            GUI.backgroundColor = WorldStorageWindow.arfColors[5];
            if (GUILayout.Button("Generate/Update GameObject"))
            {
                GenerateAndUpdateVisual(UUID, customName, localCRS_pos, localCRS_rot);
            }
            GUI.backgroundColor = ori;
            EditorGUILayout.EndHorizontal();
            EditorGUILayout.Space();

            // ---------------------
            // Params
            // ---------------------
            type = (TrackableType)EditorGUILayout.EnumPopup("Trackable Type:", type);
            unit = (UnitSystem)EditorGUILayout.EnumPopup("Unit System:", unit);
            confidence = EditorGUILayout.DoubleField("Confidence:", confidence);

            EditorGUILayout.Space();
            GUILayout.Label("Transform (local CRS):");
            localCRS_pos = EditorGUILayout.Vector3Field("   Position:", localCRS_pos);
            localCRS_rot = EditorGUILayout.Vector3Field("   Rotation:", localCRS_rot);

            EditorGUILayout.Space();
            trackableSize = EditorGUILayout.Vector3Field("Trackable Size:", trackableSize);

            EditorGUILayout.Space();
            if (GUILayout.Button("Generate Dummy Payload"))
            {
                // dummy
                trackablePayload = new byte[100];
                for (int i = 0; i < trackablePayload.Length; i++)
                {
                    trackablePayload[i] = (byte)i;
                }
            }

            EditorGUILayout.BeginHorizontal();
            EditorGUILayout.LabelField("Trackable Information ", EditorStyles.boldLabel);
            EditorGUILayout.EndHorizontal();
            EditorGUILayout.BeginHorizontal();
            EditorGUILayout.LabelField("Format ", GUILayout.Width(50));
            format = (EncodingInformationStructureDataFormat)EditorGUILayout.EnumPopup(format);
            EditorGUILayout.LabelField("Version ", GUILayout.Width(50));
            versionEncoding = EditorGUILayout.FloatField(versionEncoding);
            EditorGUILayout.EndHorizontal();


            // ---------------------
            // Keyvalues
            // ---------------------
            EditorGUILayout.Space();
            groupEnabled = EditorGUILayout.BeginToggleGroup("Optional Parameters:", groupEnabled);
            if (keyValuesFixed.Count > 0)
            {
                OutputKeyValue(0);
                OutputKeyValue(1);
                OutputKeyValue(2);
                OutputKeyValue(3);
            }
            EditorGUILayout.EndToggleGroup();
            //
            GUILayout.EndVertical();
        }

        public override void GetParams()
        {
            //customName = "Requesting information...";
            TrackableRequest.GetTrackableAsync(worldStorageServer, Guid.Parse(UUID), (response) =>
            {
                Trackable obj = response.result;
                customName = obj.Name;
                Debug.Log("name=" + customName);
                creatorUUID = obj.CreatorUUID.ToString();
                type = obj.TrackableType;
                unit = obj.Unit;
                confidence = obj.Confidence;
                if (obj.TrackableSize.Count == 3)
                {
                    trackableSize = new Vector3((float)obj.TrackableSize[0], (float)obj.TrackableSize[1], (float)obj.TrackableSize[2]);
                }
                else trackableSize = Vector3.zero;

                if (obj.LocalCRS.Count == 16)
                {
                    Matrix4x4 localCRS = WorldStorageUnityHelper.ConvertETSIARFTransform3DToUnity(obj.LocalCRS);
                    localCRS_pos = localCRS.GetPosition();
                    localCRS_rot = localCRS.rotation.eulerAngles;
                }
                else
                {
                    localCRS_pos = Vector3.zero;
                    localCRS_rot = Vector3.zero;
                }

                EncodingInformationStructure encoding = obj.TrackableEncodingInformation;
                format = encoding.DataFormat;
                float.TryParse(encoding.Version, System.Globalization.NumberStyles.Float, System.Globalization.CultureInfo.InvariantCulture, out versionEncoding);

                // ---------------------
                // Keyvalues
                // ---------------------
                //var first = WorldStorageWindow.GetFirstKeyValueTags(obj.KeyvalueTags);
                //keyValuesFixed.Clear(); // no
                for (int i = 0; i < keyValuesFixed.Count; i++) keyValuesFixed[i] = ("", "");

                if (obj.KeyvalueTags.Count > 0)
                {
                    int cnt = 0;
                    foreach (var item in obj.KeyvalueTags)
                    {
                        if (item.Key == "unityAuthoringPosX" )
                        {
                            nodePosX = float.Parse(item.Value[0], System.Globalization.CultureInfo.InvariantCulture);
                            continue; // ignore internal params
                        }
                        else if (item.Key == "unityAuthoringPosY")
                        {
                            nodePosY = float.Parse(item.Value[0], System.Globalization.CultureInfo.InvariantCulture);
                            continue;
                        }
                        if (cnt < keyValuesFixed.Count) keyValuesFixed[cnt] = (item.Key, item.Value[0]);
                        cnt++;
                    }
                }
                repaint = true;
            });
        }

        public override void AddObject()
        {
            Debug.Log("POST Trackable");
            UUID = System.Guid.Empty.ToString();
            customName = "Default Trackable";

            Trackable obj = GenerateObject();
            TrackableRequest.CreateTrackableAsync(worldStorageServer, obj, (response) =>
            {
                UUID = response.result.Message;
                UUID = UUID.Trim('"'); //Bugfix: remove " from server return value
                WorldStorageWindow.WorldStorageWindowSingleton.GetTrackables();
            });
        }

        public override Trackable GenerateObject()
        {

            Debug.Log(" Coucou " +versionEncoding);
            EncodingInformationStructure trackableEncodingInformation = new EncodingInformationStructure()
            {
                DataFormat = format ,
                Version = versionEncoding.ToString(System.Globalization.CultureInfo.InvariantCulture)
            };
            Debug.Log("Created encoding information");

            Size _trackableSize = new Size();

            _trackableSize.Add(trackableSize.x);
            _trackableSize.Add(trackableSize.y);
            _trackableSize.Add(trackableSize.z);
            Debug.Log("Created dimension");

            Matrix4x4 localCRS = new Matrix4x4();
            localCRS = Matrix4x4.TRS(localCRS_pos, Quaternion.Euler(localCRS_rot), Vector3.one);
            Transform3D _localCRS = WorldStorageUnityHelper.ConvertUnityToETSIARFTransform3D(localCRS);

            // Remember the position of the Unity graph node
            var posX = new Collection<String>();
            posX.Add(nodePosX.ToString());
            var posY = new Collection<String>();
            posY.Add(nodePosY.ToString());

            // ---------------------
            // Keyvalues
            // ---------------------
            keyValueTags.Clear();
            keyValueTags.Add("unityAuthoringPosX", posX);
            keyValueTags.Add("unityAuthoringPosY", posY);
            if (keyValuesFixed.Count > 0)
                foreach (var item in keyValuesFixed)
                {
                    if (!string.IsNullOrEmpty(item.Item1)) keyValueTags.Add(item.Item1, new Collection<string> { item.Item2 });
                }

            System.Guid _uuid = System.Guid.Parse(UUID);
            System.Guid _creator = System.Guid.Parse(worldStorageUser.UUID);
            Trackable t = new Trackable(customName)
            {
                UUID = _uuid,
                CreatorUUID = _creator,
                TrackableType = type,
                TrackableEncodingInformation = trackableEncodingInformation,
                TrackablePayload = trackablePayload,
                LocalCRS = _localCRS,
                Unit = unit,
                Confidence = confidence,
                TrackableSize = _trackableSize,
                KeyvalueTags = keyValueTags
            };
            return t;
        }
    }
}