//
// ARF - Augmented Reality Framework (ETSI ISG ARF)
//
// Copyright 2024 ETSI
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Last change: June 2024
//

using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using UnityEngine;

using ETSI.ARF.OpenAPI;
using ETSI.ARF.OpenAPI.WorldStorage;

namespace ETSI.ARF.WorldStorage.REST
{
    public class WorldLinkRequest : RequestBase<WorldLink>
    {
        //
        // Wrapper for the endpoints
        //
        static public WorldLink GetWorldLinkSync(WorldStorageServer ws, Guid UUID)
        {
            wsServer = ws;
            var httpClient = new BasicHTTPClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Request WorldLink {UUID}...");
            return apiClient.GetWorldLinkById(token, UUID);
        }

        static public ResponseObject<WorldLink> GetWorldLinkAsync(WorldStorageServer ws, Guid UUID, Action<ResponseObject<WorldLink>> func)
        {
            wsServer = ws;
            var httpClient = new UnityWebRequestHttpClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Request WorldLink {UUID}...");
            ResponseObject<WorldLink> ro = new ResponseObject<WorldLink>("Request WorldLink " + UUID.ToString(), func);
            apiClient.GetWorldLinkByIdAsync(token, UUID, ro.cancellationToken).ContinueWith(OnReceiveObject<WorldLink>, ro);
            return ro;
        }

        static public List<WorldLink> GetWorldLinksSync(WorldStorageServer ws)
        {
            wsServer = ws;
            var httpClient = new BasicHTTPClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log("[REST] Request WorldLinks...");
            return apiClient.GetWorldLinks(token).WorldLinks as List<WorldLink>;
        }

        static public ResponseObject<WorldLinksResponse> GetWorldLinksAsync(WorldStorageServer ws, Action<ResponseObject<WorldLinksResponse>> func)
        {
            wsServer = ws;
            var httpClient = new UnityWebRequestHttpClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log("[REST] Request WorldLinks...");
            ResponseObject<WorldLinksResponse> ro = new ResponseObject<WorldLinksResponse>("Request WorldLinks", func);
            apiClient.GetWorldLinksAsync(token, ro.cancellationToken).ContinueWith(OnReceiveListOfObjects<WorldLinksResponse>, ro);
            return ro;
        }

        static public string CreateWorldLinkSync(WorldStorageServer ws, WorldLink worldLink)
        {
            wsServer = ws;
            var httpClient = new BasicHTTPClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            // Add some management stuffs
            if (worldLink.UUID == Guid.Empty) worldLink.UUID = Guid.NewGuid();
            if (worldLink.CreatorUUID == Guid.Empty) worldLink.CreatorUUID = System.Guid.Parse("8fb169e2-8910-4cd5-a8f9-b7abff38d013");

            Debug.Log($"[REST] Create WorldLink {worldLink.UUID}...");
            return apiClient.AddWorldLink(token, worldLink).Message;
        }

        static public ResponseObject<Success> CreateWorldLinkAsync(WorldStorageServer ws, WorldLink worldLink, Action<ResponseObject<Success>> func)
        {
            wsServer = ws;
            var httpClient = new UnityWebRequestHttpClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            // Add some management stuffs
            if (worldLink.UUID == Guid.Empty) worldLink.UUID = Guid.NewGuid();
            if (worldLink.CreatorUUID == Guid.Empty) worldLink.CreatorUUID = System.Guid.Parse("8fb169e2-8910-4cd5-a8f9-b7abff38d013");

            Debug.Log($"[REST] Create WorldLink {worldLink.UUID}...");
            ResponseObject<Success> ro = new ResponseObject<Success>("Create WorldLink (no UUID)", func);
            apiClient.AddWorldLinkAsync(token, worldLink, ro.cancellationToken).ContinueWith(OnReceiveObject<Success>, ro);
            return ro;
        }

        static public string UpdateWorldLinkSync(WorldStorageServer ws, WorldLink worldLink)
        {
            wsServer = ws;
            var httpClient = new BasicHTTPClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Update WorldLink {worldLink.UUID}...");
            return apiClient.ModifyWorldLink(token, worldLink).Message;
        }

        static public ResponseObject<Success> UpdateWorldLinkAsync(WorldStorageServer ws, WorldLink worldLink, Action<ResponseObject<Success>> func)
        {
            wsServer = ws;
            var httpClient = new UnityWebRequestHttpClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Update WorldLink {worldLink.UUID}...");
            ResponseObject<Success> ro = new ResponseObject<Success>("Update WorldLink " + worldLink.UUID.ToString(), func);
            apiClient.ModifyWorldLinkAsync(token, worldLink, ro.cancellationToken).ContinueWith(OnReceiveObject<Success>, ro);
            return ro;
        }

        static public string DeleteWorldLinkSync(WorldStorageServer ws, Guid UUID)
        {
            wsServer = ws;
            var httpClient = new BasicHTTPClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Delete WorldLink {UUID}...");
            return apiClient.DeleteWorldLink(token, UUID).Message;
        }

        static public ResponseObject<Success> DeleteWorldLinkAsync(WorldStorageServer ws, Guid UUID, Action<ResponseObject<Success>> func)
        {
            wsServer = ws;
            var httpClient = new UnityWebRequestHttpClient(ws.URI);
            apiClient = new WorldStorageClient(httpClient);

            Debug.Log($"[REST] Delete WorldLink {UUID}...");
            ResponseObject<Success> ro = new ResponseObject<Success>("Delete WorldLink " + UUID.ToString(), func);
            apiClient.DeleteWorldLinkAsync(token, UUID, ro.cancellationToken).ContinueWith(OnReceiveObject<Success>, ro);
            return ro;
        }
    }
}
//#endif