// The Fraunhofer HHI Unity Framework
//  ___________                          .__            _____                ___ ___    ___ ___ .___ 
//  \_   _____/___________   __ __  ____ |  |__   _____/ ____\___________   /   |   \  /   |   \|   |
//   |    __) \_  __ \__  \ |  |  \/    \|  |  \ /  _ \   __\/ __ \_  __ \ /    ~    \/    ~    \   |
//   |     \   |  | \// __ \|  |  /   |  \   Y  (  <_> )  | \  ___/|  | \/ \    Y    /\    Y    /   |
//   \___  /   |__|  (____  /____/|___|  /___|  /\____/|__|  \___  >__|     \___|_  /  \___|_  /|___|
//       \/               \/           \/     \/                 \/               \/         \/      
// (C) Fraunhofer HHI, 2024

using System;
using System.IO;
using System.Net;
using System.Collections.Generic;
using System.Threading;
using System.Drawing;
using System.Threading.Tasks;
using UnityEngine;

using ETSI.ARF.OpenAPI;
using ETSI.ARF.OpenAPI.WorldStorage;
using ETSI.ARF.WorldStorage.REST;

namespace ETSI.ARF.WorldStorage
{
    /// <summary>
    /// Manager class for OpenAPI REST calls.
    /// This class insert create a response object with cancellation token for all call.
    /// Also usefull for measuring the request-response time.
    /// </summary>
    public class WorldStorageRequest : WorldStorageCollections
    {
        private WorldStorageServer wsServer;
        private WorldStorageClient apiClient;
        private string serverUri;

        public string URI => serverUri;
        public WorldStorageClient Client => apiClient;

        public WorldStorageRequest(WorldStorageServer server) : base()
        {
            wsServer = server;
            serverUri = wsServer.URI;
            var httpClient = new UnityWebRequestHttpClient(serverUri);
            apiClient = new WorldStorageClient(httpClient);
        }
        
        public WorldStorageRequest(string uri) : base()
        {
            wsServer = new WorldStorageServer();
            wsServer.basePath = uri;
            wsServer.port = 0;
            serverUri = uri;
            var httpClient = new UnityWebRequestHttpClient(serverUri);
            apiClient = new WorldStorageClient(httpClient);
        }

        //
        // Callback method
        // Call the callback method defined by the calling class when the result is there.
        //
        #region Callbacks
        private void ReceiveTextCallback(Task<string> t, object id)
        {
            if (t.IsCompleted)
            {
                string res = t.Result;

                ResponseObject<string> o = (ResponseObject<string>)id;
                o.responseTime = DateTime.Now;
                o.result = res;
                Debug.Log($"[REST] Server Response = {o.result} (#{o.transactionId}, {o.message})");

                o.callback?.Invoke(o);
            }
            else Debug.Log("[REST] OpenAPI Timeout!");
        }

        private void ReceiveTrackableCallback(Task<Trackable> t, object id)
        {
            if (t.IsCompleted)
            {
                ResponseObject<Trackable> o = (ResponseObject<Trackable>)id;
                o.responseTime = DateTime.Now;
                o.result = t.Result;
                Debug.Log($"[REST] Server Response = Got asset (#{o.transactionId}, {o.message})");

                o.callback?.Invoke(o);
            }
            else Debug.Log("[REST] OpenAPI Timeout!");
        }
        #endregion

        //
        // Admin requests
        //
        #region Handle admin requests
        public string Ping() => apiClient.GetPing();

        public ResponseObject<string> PingAsync(Action<ResponseObject<string>> func)
        {
            return AdminRequest.PingAsync(wsServer, func);
        }

        public string Status() => apiClient.GetAdmin();

        public ResponseObject<string> StatusAsync(Action<ResponseObject<string>> func)
        {
            return AdminRequest.AdminAsync(wsServer, func);
        }

        public string Version() => apiClient.GetVersion();
        public Task<string> GetVersion() => apiClient.GetVersionAsync();

        public ResponseObject<string> VersionAsync(Action<ResponseObject<string>> func)
        {
            return AdminRequest.VersionAsync(wsServer, func);
        }
        #endregion

        //
        // TRACKABLES
        //
        #region Trackables

        //public Task<int> GetNumOfAssets() => apiClient.NumOfAssetsAsync();
        //public bool IsAssetExists(Guid id) => apiClient.IsAssetId(id);
        public ResponseObject<Success> AddTrackableAsync(Action<ResponseObject<Success>> func, Trackable trackable)
        {
            return TrackableRequest.CreateTrackableAsync(wsServer, trackable, func);
        }

        public ResponseObject<Success> DeleteTrackableAsync(Action<ResponseObject<Success>> func, Guid id)
        {
            return TrackableRequest.DeleteTrackableAsync(wsServer, id, func);
        }

        public ResponseObject<Success> ModifyTrackableAsync(Action<ResponseObject<Success>> func, Trackable trackable)
        {
            return TrackableRequest.UpdateTrackableAsync(wsServer, trackable, func);
        }

        public ResponseObject<Trackable> GetTrackableByIdAsync(Action<ResponseObject<Trackable>> func, Guid id)
        {
            return TrackableRequest.GetTrackableAsync(wsServer, id, func);
        }

        public ResponseObject<TrackablesResponse> GetTrackableListAsync(Action<ResponseObject<TrackablesResponse>> func, int? max = 1)
        {
            return TrackableRequest.GetTrackablesAsync(wsServer, func);
        }
        #endregion


        //
        // Tools
        //
        #region Tools 
        public static float FloatParse(string stValue)
        {
            float v = 0;
            float.TryParse(stValue,  System.Globalization.NumberStyles.AllowDecimalPoint | System.Globalization.NumberStyles.AllowLeadingSign, System.Globalization.CultureInfo.InvariantCulture, out v);
            return v; 
        }

        #endregion
    }
}
