//
// ARF - Augmented Reality Framework (ETSI ISG ARF)
//
// Copyright 2024 ETSI
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// Last change: June 2024
//

#define isDEBUG

using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;

using UnityEngine;
using UnityEditor;
using TMPro;

using ETSI.ARF.WorldStorage.REST;
using ETSI.ARF.OpenAPI.WorldStorage;

namespace ETSI.ARF.WorldStorage.UI
{
    public class WorldLinkWindow : BaseWindow<WorldLink>
    {
        static public WorldLinkWindow winSingleton;

        public class Element
        {
            public string UUID = System.Guid.Empty.ToString();
            public string name = "(none)";
            public TypeWorldStorage type = TypeWorldStorage.UNKNOWN;
            public Vector3 pos = Vector3.zero;
        }
        
        [SerializeField] public List<string> anchors = new List<string>();

        private static GUILayoutOption miniButtonWidth = GUILayout.Width(50);
        private static GUILayoutOption middleButtonWidth = GUILayout.Width(70);

        // World Anchors params
        string UUID = System.Guid.Empty.ToString();
        string customName = "(no name for World Links)";
        string creatorUUID = System.Guid.Empty.ToString();

        // From & To elements:
        private bool showListFrom = true;
        private bool showListTo = true;
        private Element FROM = new Element();
        private Element TO = new Element();

        UnitSystem unit = UnitSystem.CM;
        Vector3 transf_pos;
        Vector3 transf_rot;

        public WorldLinkWindow()
        {
            // init somne stuffs
        }

        public static void ShowWindow(WorldStorageServer ws, WorldStorageUser user, string UUID = "")
        {
            winSingleton = EditorWindow.GetWindow(typeof(WorldLinkWindow), false, "ETSI ARF - World Link") as WorldLinkWindow;
            winSingleton.worldStorageServer = ws;
            winSingleton.worldStorageUser = user;
            if (!string.IsNullOrEmpty(UUID))
            {
                winSingleton.saveText = "Update";
                winSingleton.UUID = UUID;
                winSingleton.GetParams();
            }
            else
            {
                // Create new one
                winSingleton.saveText = "Create";
                winSingleton.AddObject();
            }
        }

        public static GameObject GenerateAndUpdateVisual(string UUID, Element from, Element to)
        {
            ETSI.ARF.WorldStorage.UI.Prefabs.WorldStoragePrefabs prefabs;
            prefabs = (Prefabs.WorldStoragePrefabs)Resources.Load("ARFPrefabs");
            GameObject arf = GameObject.Find("ARF Visuals");
            GameObject visual = GameObject.Find(UUID);

            //Value between from and to
            Vector3 centerPos = (from.pos + to.pos) * 0.5f;
            Vector3 rot = Vector3.zero;  // Direction

            if (arf == null) arf = new GameObject("ARF Visuals");
            if (visual == null)
            {
                visual = SceneAsset.Instantiate<GameObject>(prefabs.worldLinkPrefab, centerPos, Quaternion.Euler(rot), arf.transform); // TODO rot
                visual.name = UUID;
            }
            else
            {
                visual.transform.SetPositionAndRotation(centerPos, Quaternion.Euler(rot));
            }

            // Update the gizno, if GaneObject are founds!!!
            GameObject go1 = GameObject.Find(from.UUID);
            GameObject go2 = GameObject.Find(to.UUID);
            if (go1 && go2)
            {
                LinkVisual gizmo = visual.GetComponent<LinkVisual>();
                if (gizmo)
                {
                    gizmo.fromElement = go1;
                    gizmo.toElement = go2;
                }
            }

            // Update the annotation
            visual.transform.Find("Canvas/Text").GetComponent<TextMeshProUGUI>().text = $"UUID: { UUID }\nFrom: { from.name }\nTo: { to.name }";
            return visual;
        }

      
        void OnGUI()
        {
            ori = GUI.backgroundColor; // remember ori color

            gsTest = new GUIStyle("window");
            //gsTest.normal.textColor = WorldStorageWindow.arfColors[0];
            gsTest.fontStyle = FontStyle.Bold;
            gsTest.alignment = TextAnchor.UpperLeft;
            gsTest.fontSize = 16;
            gsTest.fixedHeight = 100;

            scrollPos = EditorGUILayout.BeginScrollView(scrollPos, GUILayout.ExpandWidth(true));
            WorldStorageWindow.DrawCopyright();

            DrawUIStuffs();

            EditorGUILayout.EndScrollView();

            if (GUILayout.Button("Close Window"))
            {
                Close();
            }
        }

        private void GetElementFROM()
        {
            // Trackable?
            TrackableRequest.GetTrackableAsync(worldStorageServer, Guid.Parse(FROM.UUID), (response) =>
            {
                try
                {
                    Trackable result = response.result;
                    FROM.name = result.Name;
                    FROM.type = TypeWorldStorage.TRACKABLE;

                    Matrix4x4 localCRS = WorldStorageUnityHelper.ConvertETSIARFTransform3DToUnity(result.LocalCRS);
                    FROM.pos = localCRS.GetPosition();
                }
                catch
                {
                    // Anchor?
                    WorldAnchorRequest.GetWorldAnchorAsync(worldStorageServer, Guid.Parse(FROM.UUID), (response) =>
                    {
                        try
                        {
                            WorldAnchor result = response.result;
                            FROM.name = result.Name;
                            FROM.type = TypeWorldStorage.ANCHOR;

                            Matrix4x4 localCRS = WorldStorageUnityHelper.ConvertETSIARFTransform3DToUnity(result.LocalCRS);
                            FROM.pos = localCRS.GetPosition();
                        }
                        catch
                        {
                            // Nothing!
                            FROM.name = "";
                            FROM.type = TypeWorldStorage.UNKNOWN;
                        }
                    });
                }
            });
        }

        private void GetElementTO()
        {
            // Trackable?
            TrackableRequest.GetTrackableAsync(worldStorageServer, Guid.Parse(TO.UUID), (response) =>
            {
                try
                {
                    Trackable result = response.result;
                    TO.name = result.Name;
                    TO.type = TypeWorldStorage.TRACKABLE;

                    Matrix4x4 localCRS = WorldStorageUnityHelper.ConvertETSIARFTransform3DToUnity(result.LocalCRS);
                    TO.pos = localCRS.GetPosition();
                }
                catch
                {
                    // Anchor?
                    WorldAnchorRequest.GetWorldAnchorAsync(worldStorageServer, Guid.Parse(TO.UUID), (response) =>
                    {
                        try
                        {
                            WorldAnchor result = response.result;
                            TO.name = result.Name;
                            TO.type = TypeWorldStorage.ANCHOR;

                            Matrix4x4 localCRS = WorldStorageUnityHelper.ConvertETSIARFTransform3DToUnity(result.LocalCRS);
                            TO.pos = localCRS.GetPosition();
                        }
                        catch
                        {
                            // Nothing!
                            TO.UUID = System.Guid.Empty.ToString();
                            TO.name = "";
                            TO.type = TypeWorldStorage.UNKNOWN;
                        }
                    });
                }
            });
        }

        public override void DrawUIStuffs()
        {
            GUILayout.BeginVertical(); // "World Link Editor", gsTest);
            EditorGUILayout.Space();

            GUILayout.BeginHorizontal();
            GUI.backgroundColor = WorldStorageWindow.arfColors[9];
            Texture linkImage = (Texture)AssetDatabase.LoadAssetAtPath("Packages/etsi.isg.arf.worldstorage/Editor/Images/link.png", typeof(Texture));
            GUILayout.Box(linkImage, GUILayout.Width(24), GUILayout.Height(24));
            GUI.backgroundColor = ori;
            GUILayout.Label("World Link Parameters:", EditorStyles.whiteBoldLabel);
            GUILayout.EndHorizontal();

            Rect rect = EditorGUILayout.GetControlRect(false, WorldStorageWindow.lineH);
            EditorGUI.DrawRect(rect, WorldStorageWindow.arfColors[9]);

            //
            GUILayout.Label("Server: " + worldStorageServer.serverName, EditorStyles.whiteLargeLabel);
            GUILayout.Label("User: " + worldStorageUser.userName, EditorStyles.whiteLargeLabel);
            EditorGUILayout.Space();

#if isDEBUG
            GUILayout.Label("UUID: " + UUID, EditorStyles.miniLabel); // readonly
            GUILayout.Label("Creator UID: " + creatorUUID, EditorStyles.miniLabel); // readonly
#endif

            EditorGUILayout.Space();

            // ---------------------
            // Toolbar
            // ---------------------
            EditorGUILayout.BeginHorizontal();
            GUI.backgroundColor = WorldStorageWindow.arfColors[2];
            if (GUILayout.Button(saveText))
            {
                Debug.Log("PUT World Link");

                if (!string.IsNullOrEmpty(UUID) && UUID != "0" && UUID != System.Guid.Empty.ToString())
                {
                    WorldLink obj = GenerateObject();
                    WorldLinkRequest.UpdateWorldLinkAsync(worldStorageServer, obj, (response) =>
                    {
                        UUID = response.result.Message;
                        UUID = UUID.Trim('"'); //Bugfix: remove " from server return value

                        if (WorldStorageWindow.WorldStorageWindowSingleton != null)
                        {
                            WorldStorageWindow.WorldStorageWindowSingleton.GetWorldLinks();
                        }
                        Close();
                    });
                }
            }

            GUI.backgroundColor = WorldStorageWindow.arfColors[3];
            if (GUILayout.Button("Delete"))
            {
                if (EditorUtility.DisplayDialog("Delete", "Are you sure you want to delete this World Link?", "Delete", "Cancel"))
                {
                    Debug.Log("Delete World Link");
                    WorldLinkRequest.DeleteWorldLinkAsync(worldStorageServer, Guid.Parse(UUID), (response) =>
                    {
                        UUID = System.Guid.Empty.ToString();
                        customName = "Warning: Object deleted !";
                        creatorUUID = System.Guid.Empty.ToString();
                        unit = UnitSystem.CM;
                        WorldStorageWindow.WorldStorageWindowSingleton.GetWorldLinks();
                    });
                }
            }
            GUI.backgroundColor = ori;

            GUI.backgroundColor = WorldStorageWindow.arfColors[5];
            if (GUILayout.Button("Generate/Update GameObject"))
            {
                GenerateAndUpdateVisual(UUID, FROM, TO);
            }
            GUI.backgroundColor = ori;
            EditorGUILayout.EndHorizontal();
            EditorGUILayout.Space();

            // ---------------------
            // Params
            // ---------------------
            string lastFromUUID = FROM.UUID;
            string lastToUUID = TO.UUID;

            if (GUILayout.Button("Use 'From-To' Objects from Scene Selection"))
            {
                GameObject from, to;
                GameObject[] SelectedObjects = Selection.gameObjects;

                if (SelectedObjects.Length == 2)
                {
                    Debug.Log("Creation du lien (Many thanks Eric ;-)");
                    from = SelectedObjects[0];
                    to = SelectedObjects[1];
                    FROM.UUID = from.name;
                    TO.UUID = to.name;
                }
                else
                {
                    EditorUtility.DisplayDialog("Selection", "Please select exactly 2 GameObject from typ Trackable(s) and/or WorldAnchor(s) in the scene!", "OK");
                }
            }

            showListFrom = EditorGUILayout.Foldout(showListFrom, "Parent Object (From)");
            if (showListFrom)
            {
                EditorGUILayout.BeginHorizontal();
                FROM.UUID = EditorGUILayout.TextField("UUID:", FROM.UUID);
                if (FROM.UUID.Contains("["))
                {
                    // extract the UUID
                    FROM.UUID = FROM.UUID.Split('[', ']')[1];
                }

                GUI.backgroundColor = WorldStorageWindow.arfColors[0];
                if (GUILayout.Button(" Request ", EditorStyles.miniButtonLeft, middleButtonWidth) || lastFromUUID != FROM.UUID)
                {
                    GetElementFROM();
                }
                EditorGUILayout.EndHorizontal();
                GUI.backgroundColor = ori;
                EditorGUILayout.LabelField("Name:", FROM.name);
                EditorGUILayout.LabelField("Type:", FROM.type.ToString());
            }

            EditorGUILayout.Space();
            showListTo = EditorGUILayout.Foldout(showListTo, "Child Object (To)");
            if (showListTo)
            {
                EditorGUILayout.BeginHorizontal();
                TO.UUID = EditorGUILayout.TextField("UUID:", TO.UUID);
                if (TO.UUID.Contains("["))
                {
                    // extract the UUID
                    TO.UUID = TO.UUID.Split('[', ']')[1];
                }
                GUI.backgroundColor = WorldStorageWindow.arfColors[0];
                if (GUILayout.Button("Request ", EditorStyles.miniButtonLeft, middleButtonWidth) || lastToUUID != TO.UUID)
                {
                    GetElementTO();
                }
                EditorGUILayout.EndHorizontal();
                GUI.backgroundColor = ori;
                EditorGUILayout.LabelField("Name:", TO.name);
                EditorGUILayout.LabelField("Type:", TO.type.ToString());
            }

            EditorGUILayout.Space();
            unit = (UnitSystem)EditorGUILayout.EnumPopup("Unit System:", unit);

            EditorGUILayout.Space();
            //TODO Is this required???
            GUILayout.Label("Transform:");
            transf_pos = EditorGUILayout.Vector3Field("Position:", transf_pos);
            transf_rot = EditorGUILayout.Vector3Field("Rotation:", transf_rot);

            // ---------------------
            // Keyvalues
            // ---------------------
            EditorGUILayout.Space();
            groupEnabled = EditorGUILayout.BeginToggleGroup("Optional Parameters:", groupEnabled);
            if (keyValuesFixed.Count > 0)
            {
                OutputKeyValue(0);
                OutputKeyValue(1);
                OutputKeyValue(2);
            }
            EditorGUILayout.EndToggleGroup();
            //
            GUILayout.EndVertical();
        }

        public override void GetParams()
        {
            //customName = "Requesting information...";
            WorldLinkRequest.GetWorldLinkAsync(worldStorageServer, Guid.Parse(UUID), (response) =>
            {
                WorldLink obj = response.result;
                //customName = obj.Name;
                creatorUUID = obj.CreatorUUID.ToString();

                FROM.UUID = obj.UUIDFrom.ToString();
                FROM.type = obj.TypeFrom;

                TO.UUID = obj.UUIDTo.ToString();
                TO.type = obj.TypeTo;

                unit = obj.Unit;
                if (obj.Transform.Count == 16)
                {
                    Matrix4x4 transf = WorldStorageUnityHelper.ConvertETSIARFTransform3DToUnity(obj.Transform);
                    transf_pos = transf.GetPosition();
                    transf_rot = transf.rotation.eulerAngles;
                }
                else
                {
                    transf_pos = Vector3.zero;
                    transf_rot = Vector3.zero;
                }

                // Get here the params of the from/to elements (GET)
                GetElementFROM();
                GetElementTO();

                // ---------------------
                // Keyvalues
                // ---------------------
                //var first = WorldStorageWindow.GetFirstKeyValueTags(obj.KeyvalueTags);
                //keyValuesFixed.Clear(); // no
                for (int i = 0; i < keyValuesFixed.Count; i++) keyValuesFixed[i] = ("", "");

                if (obj.KeyvalueTags.Count > 0)
                {
                    int cnt = 0;
                    foreach (var item in obj.KeyvalueTags)
                    {
                        if (item.Key == "unityAuthoringPosX" || item.Key == "unityAuthoringPosY") continue; // ignore internal params
                        if (cnt < keyValuesFixed.Count) keyValuesFixed[cnt] = (item.Key, item.Value[0]);
                        cnt++;
                    }
                }
                repaint = true;
            });
        }

        public override void AddObject()
        {
            Debug.Log("POST World Link");
            UUID = System.Guid.Empty.ToString();
            customName = "Default Link";

            WorldLink obj = GenerateObject();
            WorldLinkRequest.CreateWorldLinkAsync(worldStorageServer, obj, (response) =>
            {
                UUID = response.result.Message;
                UUID = UUID.Trim('"'); //Bugfix: remove " from server return value
                WorldStorageWindow.WorldStorageWindowSingleton.GetWorldLinks();
            });
        }

        public override WorldLink GenerateObject()
        {
            Matrix4x4 localCRS = new Matrix4x4();
            localCRS = Matrix4x4.TRS(transf_pos, Quaternion.Euler(transf_rot), Vector3.one);
            Transform3D transform3d = WorldStorageUnityHelper.ConvertUnityToETSIARFTransform3D(localCRS);

            // ---------------------
            // Keyvalues
            // ---------------------
            keyValueTags.Clear();
            //keyValueTags.Add("unityAuthoringPosX", posX);
            //keyValueTags.Add("unityAuthoringPosY", posY);
            if (keyValuesFixed.Count > 0)
                foreach (var item in keyValuesFixed)
                {
                    if (!string.IsNullOrEmpty(item.Item1)) keyValueTags.Add(item.Item1, new Collection<string> { item.Item2 });
                }

            System.Guid _uuid = System.Guid.Parse(UUID);
            System.Guid _creator = System.Guid.Parse(worldStorageUser.UUID);
            System.Guid _from = System.Guid.Parse(FROM.UUID);
            System.Guid _to = System.Guid.Parse(TO.UUID);
            WorldLink t = new WorldLink()
            {
                UUID = _uuid,
                CreatorUUID = _creator,
                UUIDFrom = _from,
                UUIDTo = _to,
                TypeFrom = FROM.type,
                TypeTo = TO.type,
                Transform = transform3d,
                Unit = unit,
                KeyvalueTags = keyValueTags
            };
            return t;
        }
    }
}